/*
 * Copyright 2016 Intel Corporation
 * Copyright 2016 Broadcom
 * Copyright 2020 Collabora, Ltd.
 * Copyright 2024 Alyssa Rosenzweig
 * SPDX-License-Identifier: MIT
 */

#pragma once

#include "compiler/libcl/libcl.h"
#include "util/bitpack_helpers.h"

#ifndef __OPENCL_VERSION__
#include <inttypes.h>
#include <stdio.h>
#include "util/half_float.h"
#endif

#define __gen_unpack_float(x, y, z) uif(__gen_unpack_uint(x, y, z))
#define __gen_unpack_half(x, y, z)                                             \
   _mesa_half_to_float(__gen_unpack_uint(x, y, z))

static inline uint64_t
__gen_unpack_uint(CONST uint32_t *restrict cl, uint32_t start, uint32_t end)
{
   uint64_t val = 0;
   const int width = end - start + 1;
   const uint64_t mask =
      (width == 64) ? ~((uint64_t)0) : ((uint64_t)1 << width) - 1;

   for (unsigned word = start / 32; word < (end / 32) + 1; word++) {
      val |= ((uint64_t)cl[word]) << ((word - start / 32) * 32);
   }

   return (val >> (start % 32)) & mask;
}

/*
 * LODs are 4:6 fixed point. We must clamp before converting to integers to
 * avoid undefined behaviour for out-of-bounds inputs like +/- infinity.
 */
static inline uint32_t
__gen_pack_lod(float f, uint32_t start, uint32_t end)
{
   uint32_t fixed = CLAMP(f * (1 << 6), 0 /* 0.0 */, 0x380 /* 14.0 */);
   return util_bitpack_uint(fixed, start, end);
}

static inline float
__gen_unpack_lod(CONST uint32_t *restrict cl, uint32_t start, uint32_t end)
{
   return ((float)__gen_unpack_uint(cl, start, end)) / (1 << 6);
}

static inline uint64_t
__gen_unpack_sint(CONST uint32_t *restrict cl, uint32_t start, uint32_t end)
{
   int size = end - start + 1;
   int64_t val = __gen_unpack_uint(cl, start, end);

   return util_sign_extend(val, size);
}

static inline uint64_t
__gen_to_groups(uint32_t value, uint32_t group_size, uint32_t length)
{
   /* Zero is not representable, clamp to minimum */
   if (value == 0)
      return 1;

   /* Round up to the nearest number of groups */
   uint32_t groups = DIV_ROUND_UP(value, group_size);

   /* The 0 encoding means "all" */
   if (groups == ((uint64_t)1) << length)
      return 0;

   /* Otherwise it's encoded as the identity */
   assert(groups < (1u << length) && "out of bounds");
   assert(groups >= 1 && "exhaustive");
   return groups;
}

static inline uint64_t
__gen_from_groups(uint32_t value, uint32_t group_size, uint32_t length)
{
   return group_size * (value ? value : (1 << length));
}

#define agx_pack(dst, T, name)                                                 \
   for (struct AGX_##T name = {AGX_##T##_header},                              \
                       *_loop_count = (GLOBAL void *)((uintptr_t)0);           \
        (uintptr_t)_loop_count < 1; (                                          \
           {                                                                   \
              AGX_##T##_pack((GLOBAL uint32_t *)(dst), &name);                 \
              _loop_count = (GLOBAL void *)(((uintptr_t)_loop_count) + 1);     \
           }))

#define agx_unpack(fp, src, T, name)                                           \
   struct AGX_##T name;                                                        \
   AGX_##T##_unpack(fp, (CONST uint8_t *)(src), &name)

#define agx_print(fp, T, var, indent) AGX_##T##_print(fp, &(var), indent)

static inline void
agx_merge_helper(uint32_t *dst, const uint32_t *src, size_t bytes)
{
   assert((bytes & 3) == 0);

   for (unsigned i = 0; i < (bytes / 4); ++i)
      dst[i] |= src[i];
}

#define agx_merge(packed1, packed2, type)                                      \
   agx_merge_helper((packed1).opaque, (packed2).opaque, AGX_##type##_LENGTH)

#if defined(NDEBUG) || defined(__OPENCL_VERSION__)
#define agx_genxml_validate_bounds(a, b, c)
#define agx_genxml_validate_mask(a, b, c, d, e) true
#define agx_genxml_validate_exact(a, b, c, d)   true
#else
static inline void
agx_genxml_validate_bounds(const char *name, uint64_t value, uint64_t bound)
{
   if (unlikely(value >= bound)) {
      fprintf(stderr, "%s out-of-bounds, got 0x%" PRIx64 ", max %" PRIx64 "\n",
              name, value, bound);

      unreachable("Out-of-bounds pack");
   }
}

static inline bool
agx_genxml_validate_mask(FILE *fp, const char *name, const void *cl_,
                         uint32_t index, uint32_t bad_mask)
{
   const uint32_t *cl = (const uint32_t *)cl_;
   uint32_t bad = cl[index] & bad_mask;

   if (bad && fp != NULL) {
      fprintf(
         fp,
         "XXX: Unknown field of %s unpacked at word %u got %X, bad mask %X\n",
         name, index, cl[index], bad);
   }

   return bad == 0;
}

static bool
agx_genxml_validate_exact(FILE *fp, const char *name, uint64_t value,
                          uint64_t exact)
{
   if (value != exact && fp != NULL) {
      fprintf(fp, "XXX: Expected %s to equal %" PRIx64 " but got %" PRIx64 "\n",
              name, value, exact);
   }

   return value == exact;
}

#endif

/* Everything after this is autogenerated from XML. Do not hand edit. */

enum agx_channel {
   AGX_CHANNEL_R = 0,
   AGX_CHANNEL_G = 1,
   AGX_CHANNEL_B = 2,
   AGX_CHANNEL_A = 3,
   AGX_CHANNEL_1 = 4,
   AGX_CHANNEL_0 = 5,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_channel_as_str(enum agx_channel imm)
{
    switch (imm) {
    case AGX_CHANNEL_R: return "R";
    case AGX_CHANNEL_G: return "G";
    case AGX_CHANNEL_B: return "B";
    case AGX_CHANNEL_A: return "A";
    case AGX_CHANNEL_1: return "1";
    case AGX_CHANNEL_0: return "0";
    default: return NULL;
    }
}
#endif

enum agx_zs_func {
   AGX_ZS_FUNC_NEVER = 0,
   AGX_ZS_FUNC_LESS = 1,
   AGX_ZS_FUNC_EQUAL = 2,
   AGX_ZS_FUNC_LEQUAL = 3,
   AGX_ZS_FUNC_GREATER = 4,
   AGX_ZS_FUNC_NOT_EQUAL = 5,
   AGX_ZS_FUNC_GEQUAL = 6,
   AGX_ZS_FUNC_ALWAYS = 7,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_zs_func_as_str(enum agx_zs_func imm)
{
    switch (imm) {
    case AGX_ZS_FUNC_NEVER: return "Never";
    case AGX_ZS_FUNC_LESS: return "Less";
    case AGX_ZS_FUNC_EQUAL: return "Equal";
    case AGX_ZS_FUNC_LEQUAL: return "Lequal";
    case AGX_ZS_FUNC_GREATER: return "Greater";
    case AGX_ZS_FUNC_NOT_EQUAL: return "Not Equal";
    case AGX_ZS_FUNC_GEQUAL: return "Gequal";
    case AGX_ZS_FUNC_ALWAYS: return "Always";
    default: return NULL;
    }
}
#endif

enum agx_compare_func {
   AGX_COMPARE_FUNC_LEQUAL = 0,
   AGX_COMPARE_FUNC_GEQUAL = 1,
   AGX_COMPARE_FUNC_LESS = 2,
   AGX_COMPARE_FUNC_GREATER = 3,
   AGX_COMPARE_FUNC_EQUAL = 4,
   AGX_COMPARE_FUNC_NOT_EQUAL = 5,
   AGX_COMPARE_FUNC_ALWAYS = 6,
   AGX_COMPARE_FUNC_NEVER = 7,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_compare_func_as_str(enum agx_compare_func imm)
{
    switch (imm) {
    case AGX_COMPARE_FUNC_LEQUAL: return "Lequal";
    case AGX_COMPARE_FUNC_GEQUAL: return "Gequal";
    case AGX_COMPARE_FUNC_LESS: return "Less";
    case AGX_COMPARE_FUNC_GREATER: return "Greater";
    case AGX_COMPARE_FUNC_EQUAL: return "Equal";
    case AGX_COMPARE_FUNC_NOT_EQUAL: return "Not Equal";
    case AGX_COMPARE_FUNC_ALWAYS: return "Always";
    case AGX_COMPARE_FUNC_NEVER: return "Never";
    default: return NULL;
    }
}
#endif

enum agx_stencil_op {
   AGX_STENCIL_OP_KEEP = 0,
   AGX_STENCIL_OP_ZERO = 1,
   AGX_STENCIL_OP_REPLACE = 2,
   AGX_STENCIL_OP_INCR_SAT = 3,
   AGX_STENCIL_OP_DECR_SAT = 4,
   AGX_STENCIL_OP_INVERT = 5,
   AGX_STENCIL_OP_INCR_WRAP = 6,
   AGX_STENCIL_OP_DECR_WRAP = 7,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_stencil_op_as_str(enum agx_stencil_op imm)
{
    switch (imm) {
    case AGX_STENCIL_OP_KEEP: return "Keep";
    case AGX_STENCIL_OP_ZERO: return "Zero";
    case AGX_STENCIL_OP_REPLACE: return "Replace";
    case AGX_STENCIL_OP_INCR_SAT: return "Incr Sat";
    case AGX_STENCIL_OP_DECR_SAT: return "Decr Sat";
    case AGX_STENCIL_OP_INVERT: return "Invert";
    case AGX_STENCIL_OP_INCR_WRAP: return "Incr Wrap";
    case AGX_STENCIL_OP_DECR_WRAP: return "Decr Wrap";
    default: return NULL;
    }
}
#endif

enum agx_visibility_mode {
   AGX_VISIBILITY_MODE_NONE = 0,
   AGX_VISIBILITY_MODE_COUNTING = 2,
   AGX_VISIBILITY_MODE_BOOLEAN = 3,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_visibility_mode_as_str(enum agx_visibility_mode imm)
{
    switch (imm) {
    case AGX_VISIBILITY_MODE_NONE: return "None";
    case AGX_VISIBILITY_MODE_COUNTING: return "Counting";
    case AGX_VISIBILITY_MODE_BOOLEAN: return "Boolean";
    default: return NULL;
    }
}
#endif

enum agx_polygon_mode {
   AGX_POLYGON_MODE_FILL = 0,
   AGX_POLYGON_MODE_LINE = 1,
   AGX_POLYGON_MODE_POINT = 2,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_polygon_mode_as_str(enum agx_polygon_mode imm)
{
    switch (imm) {
    case AGX_POLYGON_MODE_FILL: return "Fill";
    case AGX_POLYGON_MODE_LINE: return "Line";
    case AGX_POLYGON_MODE_POINT: return "Point";
    default: return NULL;
    }
}
#endif

enum agx_primitive {
   AGX_PRIMITIVE_POINTS = 0,
   AGX_PRIMITIVE_LINES = 1,
   AGX_PRIMITIVE_LINE_STRIP = 3,
   AGX_PRIMITIVE_LINE_LOOP = 5,
   AGX_PRIMITIVE_TRIANGLES = 6,
   AGX_PRIMITIVE_TRIANGLE_STRIP = 9,
   AGX_PRIMITIVE_TRIANGLE_FAN = 11,
   AGX_PRIMITIVE_QUADS = 14,
   AGX_PRIMITIVE_QUAD_STRIP = 15,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_primitive_as_str(enum agx_primitive imm)
{
    switch (imm) {
    case AGX_PRIMITIVE_POINTS: return "Points";
    case AGX_PRIMITIVE_LINES: return "Lines";
    case AGX_PRIMITIVE_LINE_STRIP: return "Line strip";
    case AGX_PRIMITIVE_LINE_LOOP: return "Line loop";
    case AGX_PRIMITIVE_TRIANGLES: return "Triangles";
    case AGX_PRIMITIVE_TRIANGLE_STRIP: return "Triangle strip";
    case AGX_PRIMITIVE_TRIANGLE_FAN: return "Triangle fan";
    case AGX_PRIMITIVE_QUADS: return "Quads";
    case AGX_PRIMITIVE_QUAD_STRIP: return "Quad strip";
    default: return NULL;
    }
}
#endif

enum agx_object_type {
   AGX_OBJECT_TYPE_TRIANGLE = 0,
   AGX_OBJECT_TYPE_LINE = 1,
   AGX_OBJECT_TYPE_POINT_SPRITE_UV10 = 2,
   AGX_OBJECT_TYPE_POINT_SPRITE_UV01 = 4,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_object_type_as_str(enum agx_object_type imm)
{
    switch (imm) {
    case AGX_OBJECT_TYPE_TRIANGLE: return "Triangle";
    case AGX_OBJECT_TYPE_LINE: return "Line";
    case AGX_OBJECT_TYPE_POINT_SPRITE_UV10: return "Point sprite UV=10";
    case AGX_OBJECT_TYPE_POINT_SPRITE_UV01: return "Point sprite UV=01";
    default: return NULL;
    }
}
#endif

enum agx_layout {
   AGX_LAYOUT_LINEAR = 0,
   AGX_LAYOUT_TWIDDLED = 1,
   AGX_LAYOUT_GPU = 2,
   AGX_LAYOUT_INTERCHANGE = 3,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_layout_as_str(enum agx_layout imm)
{
    switch (imm) {
    case AGX_LAYOUT_LINEAR: return "Linear";
    case AGX_LAYOUT_TWIDDLED: return "Twiddled";
    case AGX_LAYOUT_GPU: return "GPU";
    case AGX_LAYOUT_INTERCHANGE: return "Interchange";
    default: return NULL;
    }
}
#endif

enum agx_channels {
   AGX_CHANNELS_R8 = 0,
   AGX_CHANNELS_R16 = 9,
   AGX_CHANNELS_R8G8 = 10,
   AGX_CHANNELS_R5G6B5 = 11,
   AGX_CHANNELS_R4G4B4A4 = 12,
   AGX_CHANNELS_A1R5G5B5 = 13,
   AGX_CHANNELS_R5G5B5A1 = 14,
   AGX_CHANNELS_R32 = 33,
   AGX_CHANNELS_R16G16 = 35,
   AGX_CHANNELS_R11G11B10 = 37,
   AGX_CHANNELS_R10G10B10A2 = 38,
   AGX_CHANNELS_R9G9B9E5 = 39,
   AGX_CHANNELS_R8G8B8A8 = 40,
   AGX_CHANNELS_R32G32 = 49,
   AGX_CHANNELS_R16G16B16A16 = 50,
   AGX_CHANNELS_R32G32B32_EMULATED = 54,
   AGX_CHANNELS_R32G32B32A32 = 56,
   AGX_CHANNELS_GBGR_422 = 64,
   AGX_CHANNELS_BGRG_422 = 65,
   AGX_CHANNELS_PVRTC_2BPP = 80,
   AGX_CHANNELS_PVRTC_4BPP = 81,
   AGX_CHANNELS_ETC2_RGB8 = 88,
   AGX_CHANNELS_ETC2_RGBA8 = 89,
   AGX_CHANNELS_ETC2_RGB8A1 = 90,
   AGX_CHANNELS_EAC_R11 = 91,
   AGX_CHANNELS_EAC_RG11 = 92,
   AGX_CHANNELS_ASTC_4X4 = 96,
   AGX_CHANNELS_ASTC_5X4 = 97,
   AGX_CHANNELS_ASTC_5X5 = 98,
   AGX_CHANNELS_ASTC_6X5 = 99,
   AGX_CHANNELS_ASTC_6X6 = 100,
   AGX_CHANNELS_ASTC_8X5 = 101,
   AGX_CHANNELS_ASTC_8X6 = 102,
   AGX_CHANNELS_ASTC_8X8 = 103,
   AGX_CHANNELS_ASTC_10X5 = 104,
   AGX_CHANNELS_ASTC_10X6 = 105,
   AGX_CHANNELS_ASTC_10X8 = 106,
   AGX_CHANNELS_ASTC_10X10 = 107,
   AGX_CHANNELS_ASTC_12X10 = 108,
   AGX_CHANNELS_ASTC_12X12 = 109,
   AGX_CHANNELS_BC1 = 116,
   AGX_CHANNELS_BC2 = 117,
   AGX_CHANNELS_BC3 = 118,
   AGX_CHANNELS_BC4 = 119,
   AGX_CHANNELS_BC5 = 120,
   AGX_CHANNELS_BC6H = 121,
   AGX_CHANNELS_BC6H_UFLOAT = 122,
   AGX_CHANNELS_BC7 = 123,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_channels_as_str(enum agx_channels imm)
{
    switch (imm) {
    case AGX_CHANNELS_R8: return "R8";
    case AGX_CHANNELS_R16: return "R16";
    case AGX_CHANNELS_R8G8: return "R8G8";
    case AGX_CHANNELS_R5G6B5: return "R5G6B5";
    case AGX_CHANNELS_R4G4B4A4: return "R4G4B4A4";
    case AGX_CHANNELS_A1R5G5B5: return "A1R5G5B5";
    case AGX_CHANNELS_R5G5B5A1: return "R5G5B5A1";
    case AGX_CHANNELS_R32: return "R32";
    case AGX_CHANNELS_R16G16: return "R16G16";
    case AGX_CHANNELS_R11G11B10: return "R11G11B10";
    case AGX_CHANNELS_R10G10B10A2: return "R10G10B10A2";
    case AGX_CHANNELS_R9G9B9E5: return "R9G9B9E5";
    case AGX_CHANNELS_R8G8B8A8: return "R8G8B8A8";
    case AGX_CHANNELS_R32G32: return "R32G32";
    case AGX_CHANNELS_R16G16B16A16: return "R16G16B16A16";
    case AGX_CHANNELS_R32G32B32_EMULATED: return "R32G32B32 (Emulated)";
    case AGX_CHANNELS_R32G32B32A32: return "R32G32B32A32";
    case AGX_CHANNELS_GBGR_422: return "GBGR 422";
    case AGX_CHANNELS_BGRG_422: return "BGRG 422";
    case AGX_CHANNELS_PVRTC_2BPP: return "PVRTC 2bpp";
    case AGX_CHANNELS_PVRTC_4BPP: return "PVRTC 4bpp";
    case AGX_CHANNELS_ETC2_RGB8: return "ETC2 RGB8";
    case AGX_CHANNELS_ETC2_RGBA8: return "ETC2 RGBA8";
    case AGX_CHANNELS_ETC2_RGB8A1: return "ETC2 RGB8A1";
    case AGX_CHANNELS_EAC_R11: return "EAC R11";
    case AGX_CHANNELS_EAC_RG11: return "EAC RG11";
    case AGX_CHANNELS_ASTC_4X4: return "ASTC 4x4";
    case AGX_CHANNELS_ASTC_5X4: return "ASTC 5x4";
    case AGX_CHANNELS_ASTC_5X5: return "ASTC 5x5";
    case AGX_CHANNELS_ASTC_6X5: return "ASTC 6x5";
    case AGX_CHANNELS_ASTC_6X6: return "ASTC 6x6";
    case AGX_CHANNELS_ASTC_8X5: return "ASTC 8x5";
    case AGX_CHANNELS_ASTC_8X6: return "ASTC 8x6";
    case AGX_CHANNELS_ASTC_8X8: return "ASTC 8x8";
    case AGX_CHANNELS_ASTC_10X5: return "ASTC 10x5";
    case AGX_CHANNELS_ASTC_10X6: return "ASTC 10x6";
    case AGX_CHANNELS_ASTC_10X8: return "ASTC 10x8";
    case AGX_CHANNELS_ASTC_10X10: return "ASTC 10x10";
    case AGX_CHANNELS_ASTC_12X10: return "ASTC 12x10";
    case AGX_CHANNELS_ASTC_12X12: return "ASTC 12x12";
    case AGX_CHANNELS_BC1: return "BC1";
    case AGX_CHANNELS_BC2: return "BC2";
    case AGX_CHANNELS_BC3: return "BC3";
    case AGX_CHANNELS_BC4: return "BC4";
    case AGX_CHANNELS_BC5: return "BC5";
    case AGX_CHANNELS_BC6H: return "BC6H";
    case AGX_CHANNELS_BC6H_UFLOAT: return "BC6H Ufloat";
    case AGX_CHANNELS_BC7: return "BC7";
    default: return NULL;
    }
}
#endif

enum agx_texture_type {
   AGX_TEXTURE_TYPE_UNORM = 0,
   AGX_TEXTURE_TYPE_SNORM = 1,
   AGX_TEXTURE_TYPE_UINT = 2,
   AGX_TEXTURE_TYPE_SINT = 3,
   AGX_TEXTURE_TYPE_FLOAT = 4,
   AGX_TEXTURE_TYPE_XR = 5,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_texture_type_as_str(enum agx_texture_type imm)
{
    switch (imm) {
    case AGX_TEXTURE_TYPE_UNORM: return "Unorm";
    case AGX_TEXTURE_TYPE_SNORM: return "Snorm";
    case AGX_TEXTURE_TYPE_UINT: return "Uint";
    case AGX_TEXTURE_TYPE_SINT: return "Sint";
    case AGX_TEXTURE_TYPE_FLOAT: return "Float";
    case AGX_TEXTURE_TYPE_XR: return "XR";
    default: return NULL;
    }
}
#endif

struct AGX_CF_BINDING_HEADER {
   uint32_t                             number_of_32_bit_slots;
   uint32_t                             number_of_coefficient_registers;
};

#define AGX_CF_BINDING_HEADER_header 0

static inline void
AGX_CF_BINDING_HEADER_pack(GLOBAL uint32_t * restrict cl,
                           const struct AGX_CF_BINDING_HEADER * restrict values)
{
   agx_genxml_validate_bounds("CF binding header::number_of_32_bit_slots", values->number_of_32_bit_slots, 0x100ull);
   agx_genxml_validate_bounds("CF binding header::number_of_coefficient_registers", values->number_of_coefficient_registers, 0x100ull);
   cl[ 0] = util_bitpack_uint(values->number_of_32_bit_slots, 0, 7) |
            util_bitpack_uint(values->number_of_coefficient_registers, 8, 15);
}

#define AGX_CF_BINDING_HEADER_LENGTH 4
struct agx_cf_binding_header_packed { uint32_t opaque[1];};
static inline bool
AGX_CF_BINDING_HEADER_unpack(FILE *fp, CONST uint8_t * restrict cl,
                             struct AGX_CF_BINDING_HEADER * restrict values)
{
   values->number_of_32_bit_slots = __gen_unpack_uint((CONST uint32_t *) cl, 0, 7);
   values->number_of_coefficient_registers = __gen_unpack_uint((CONST uint32_t *) cl, 8, 15);
   return agx_genxml_validate_mask(fp, "CF binding header", cl, 0, 0xffff0000);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_CF_BINDING_HEADER_print(FILE *fp, const struct AGX_CF_BINDING_HEADER * values, unsigned indent)
{
   fprintf(fp, "%*sNumber of 32-bit slots: %u\n", indent, "", values->number_of_32_bit_slots);
   fprintf(fp, "%*sNumber of coefficient registers: %u\n", indent, "", values->number_of_coefficient_registers);
}
#endif

enum agx_shade_model {
   AGX_SHADE_MODEL_FLAT_VERTEX_0 = 0,
   AGX_SHADE_MODEL_FLAT_VERTEX_2 = 2,
   AGX_SHADE_MODEL_LINEAR = 3,
   AGX_SHADE_MODEL_FLAT_VERTEX_1 = 6,
   AGX_SHADE_MODEL_PERSPECTIVE = 7,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_shade_model_as_str(enum agx_shade_model imm)
{
    switch (imm) {
    case AGX_SHADE_MODEL_FLAT_VERTEX_0: return "Flat vertex 0";
    case AGX_SHADE_MODEL_FLAT_VERTEX_2: return "Flat vertex 2";
    case AGX_SHADE_MODEL_LINEAR: return "Linear";
    case AGX_SHADE_MODEL_FLAT_VERTEX_1: return "Flat vertex 1";
    case AGX_SHADE_MODEL_PERSPECTIVE: return "Perspective";
    default: return NULL;
    }
}
#endif

enum agx_coefficient_source {
   AGX_COEFFICIENT_SOURCE_VARYING = 0,
   AGX_COEFFICIENT_SOURCE_FRAGCOORD_Z = 1,
   AGX_COEFFICIENT_SOURCE_POINT_COORD = 2,
   AGX_COEFFICIENT_SOURCE_PRIMITIVE_ID = 3,
   AGX_COEFFICIENT_SOURCE_BARYCENTRIC_COORD = 5,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_coefficient_source_as_str(enum agx_coefficient_source imm)
{
    switch (imm) {
    case AGX_COEFFICIENT_SOURCE_VARYING: return "Varying";
    case AGX_COEFFICIENT_SOURCE_FRAGCOORD_Z: return "Fragcoord Z";
    case AGX_COEFFICIENT_SOURCE_POINT_COORD: return "Point coord";
    case AGX_COEFFICIENT_SOURCE_PRIMITIVE_ID: return "Primitive ID";
    case AGX_COEFFICIENT_SOURCE_BARYCENTRIC_COORD: return "Barycentric coord";
    default: return NULL;
    }
}
#endif

struct AGX_CF_BINDING {
   uint32_t                             components;
   enum agx_shade_model                 shade_model;
   enum agx_coefficient_source          source;
   uint32_t                             base_slot;
   uint32_t                             base_coefficient_register;
};

#define AGX_CF_BINDING_header                   \
   .components = 1

static inline void
AGX_CF_BINDING_pack(GLOBAL uint32_t * restrict cl,
                    const struct AGX_CF_BINDING * restrict values)
{
   assert(values->components >= 1);
   agx_genxml_validate_bounds("CF binding::components", values->components - 1, 0x4ull);
   agx_genxml_validate_bounds("CF binding::shade_model", values->shade_model, 0x8ull);
   agx_genxml_validate_bounds("CF binding::source", values->source, 0x8ull);
   agx_genxml_validate_bounds("CF binding::base_slot", values->base_slot, 0x100ull);
   agx_genxml_validate_bounds("CF binding::base_coefficient_register", values->base_coefficient_register, 0x100ull);
   cl[ 0] = util_bitpack_uint(values->components - 1, 0, 1) |
            util_bitpack_uint(values->shade_model, 2, 4) |
            util_bitpack_uint(values->source, 5, 7) |
            util_bitpack_uint(values->base_slot, 8, 15) |
            util_bitpack_uint(values->base_coefficient_register, 16, 23);
}

#define AGX_CF_BINDING_LENGTH 4
struct agx_cf_binding_packed { uint32_t opaque[1];};
static inline bool
AGX_CF_BINDING_unpack(FILE *fp, CONST uint8_t * restrict cl,
                      struct AGX_CF_BINDING * restrict values)
{
   values->components = __gen_unpack_uint((CONST uint32_t *) cl, 0, 1) + 1;
   values->shade_model = (enum agx_shade_model) __gen_unpack_uint((CONST uint32_t *) cl, 2, 4);
   values->source = (enum agx_coefficient_source) __gen_unpack_uint((CONST uint32_t *) cl, 5, 7);
   values->base_slot = __gen_unpack_uint((CONST uint32_t *) cl, 8, 15);
   values->base_coefficient_register = __gen_unpack_uint((CONST uint32_t *) cl, 16, 23);
   return agx_genxml_validate_mask(fp, "CF binding", cl, 0, 0xff000000);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_CF_BINDING_print(FILE *fp, const struct AGX_CF_BINDING * values, unsigned indent)
{
   fprintf(fp, "%*sComponents: %u\n", indent, "", values->components);
   if (agx_shade_model_as_str(values->shade_model))
     fprintf(fp, "%*sShade model: %s\n", indent, "", agx_shade_model_as_str(values->shade_model));
   else
     fprintf(fp, "%*sShade model: unknown %X (XXX)\n", indent, "", values->shade_model);
   if (agx_coefficient_source_as_str(values->source))
     fprintf(fp, "%*sSource: %s\n", indent, "", agx_coefficient_source_as_str(values->source));
   else
     fprintf(fp, "%*sSource: unknown %X (XXX)\n", indent, "", values->source);
   fprintf(fp, "%*sBase slot: %u\n", indent, "", values->base_slot);
   fprintf(fp, "%*sBase coefficient register: %u\n", indent, "", values->base_coefficient_register);
}
#endif

enum agx_texture_dimension {
   AGX_TEXTURE_DIMENSION_1D = 0,
   AGX_TEXTURE_DIMENSION_1D_ARRAY = 1,
   AGX_TEXTURE_DIMENSION_2D = 2,
   AGX_TEXTURE_DIMENSION_2D_ARRAY = 3,
   AGX_TEXTURE_DIMENSION_2D_MULTISAMPLED = 4,
   AGX_TEXTURE_DIMENSION_3D = 5,
   AGX_TEXTURE_DIMENSION_CUBE = 6,
   AGX_TEXTURE_DIMENSION_CUBE_ARRAY = 7,
   AGX_TEXTURE_DIMENSION_2D_ARRAY_MULTISAMPLED = 8,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_texture_dimension_as_str(enum agx_texture_dimension imm)
{
    switch (imm) {
    case AGX_TEXTURE_DIMENSION_1D: return "1D";
    case AGX_TEXTURE_DIMENSION_1D_ARRAY: return "1D Array";
    case AGX_TEXTURE_DIMENSION_2D: return "2D";
    case AGX_TEXTURE_DIMENSION_2D_ARRAY: return "2D Array";
    case AGX_TEXTURE_DIMENSION_2D_MULTISAMPLED: return "2D Multisampled";
    case AGX_TEXTURE_DIMENSION_3D: return "3D";
    case AGX_TEXTURE_DIMENSION_CUBE: return "Cube";
    case AGX_TEXTURE_DIMENSION_CUBE_ARRAY: return "Cube Array";
    case AGX_TEXTURE_DIMENSION_2D_ARRAY_MULTISAMPLED: return "2D Array Multisampled";
    default: return NULL;
    }
}
#endif

enum agx_sample_count {
   AGX_SAMPLE_COUNT_2 = 0,
   AGX_SAMPLE_COUNT_4 = 1,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_sample_count_as_str(enum agx_sample_count imm)
{
    switch (imm) {
    case AGX_SAMPLE_COUNT_2: return "2";
    case AGX_SAMPLE_COUNT_4: return "4";
    default: return NULL;
    }
}
#endif

enum agx_image_mode {
   AGX_IMAGE_MODE_NORMAL = 0,
   AGX_IMAGE_MODE_NOT_SPARSE_BUT_PROBABLY_WITH_COUNTERS = 1,
   AGX_IMAGE_MODE_SPARSE_BUT_PROBABLY_NO_COUNTERS = 2,
   AGX_IMAGE_MODE_SPARSE = 3,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_image_mode_as_str(enum agx_image_mode imm)
{
    switch (imm) {
    case AGX_IMAGE_MODE_NORMAL: return "Normal";
    case AGX_IMAGE_MODE_NOT_SPARSE_BUT_PROBABLY_WITH_COUNTERS: return "Not sparse but probably with counters";
    case AGX_IMAGE_MODE_SPARSE_BUT_PROBABLY_NO_COUNTERS: return "Sparse but probably no counters";
    case AGX_IMAGE_MODE_SPARSE: return "Sparse";
    default: return NULL;
    }
}
#endif

struct AGX_SPARSE_BLOCK {
   uint64_t                             address;
   bool                                 unknown;
   bool                                 enabled;
};

#define AGX_SPARSE_BLOCK_header 0

static inline void
AGX_SPARSE_BLOCK_pack(GLOBAL uint32_t * restrict cl,
                      const struct AGX_SPARSE_BLOCK * restrict values)
{
   assert((values->address & 0x3fff) == 0);
   agx_genxml_validate_bounds("Sparse block::address", values->address >> 14, 0x4000000ull);
   agx_genxml_validate_bounds("Sparse block::unknown", values->unknown, 0x2ull);
   agx_genxml_validate_bounds("Sparse block::enabled", values->enabled, 0x2ull);
   cl[ 0] = util_bitpack_uint(values->address >> 14, 0, 25) |
            util_bitpack_uint(values->unknown, 30, 30) |
            util_bitpack_uint(values->enabled, 31, 31);
}

#define AGX_SPARSE_BLOCK_LENGTH 4
struct agx_sparse_block_packed { uint32_t opaque[1];};
static inline bool
AGX_SPARSE_BLOCK_unpack(FILE *fp, CONST uint8_t * restrict cl,
                        struct AGX_SPARSE_BLOCK * restrict values)
{
   values->address = __gen_unpack_uint((CONST uint32_t *) cl, 0, 25) << 14;
   values->unknown = __gen_unpack_uint((CONST uint32_t *) cl, 30, 30);
   values->enabled = __gen_unpack_uint((CONST uint32_t *) cl, 31, 31);
   return agx_genxml_validate_mask(fp, "Sparse block", cl, 0, 0x3c000000);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_SPARSE_BLOCK_print(FILE *fp, const struct AGX_SPARSE_BLOCK * values, unsigned indent)
{
   fprintf(fp, "%*sAddress: 0x%" PRIx64 "\n", indent, "", values->address);
   fprintf(fp, "%*sUnknown: %s\n", indent, "", values->unknown ? "true" : "false");
   fprintf(fp, "%*sEnabled: %s\n", indent, "", values->enabled ? "true" : "false");
}
#endif

struct AGX_PBE {
   enum agx_texture_dimension           dimension;
   enum agx_layout                      layout;
   enum agx_channels                    channels;
   enum agx_texture_type                type;
   enum agx_channel                     swizzle_r;
   enum agx_channel                     swizzle_g;
   enum agx_channel                     swizzle_b;
   enum agx_channel                     swizzle_a;
   uint32_t                             width;
   uint32_t                             height;
   bool                                 unk_52;
   bool                                 rotate_90;
   bool                                 flip_vertical;
   enum agx_sample_count                samples;
   bool                                 unk_mipmapped;
   bool                                 compressed_1;
   enum agx_image_mode                  mode;
   uint64_t                             buffer;
   uint32_t                             level;
   uint32_t                             levels;
   uint32_t                             layers;
   bool                                 page_aligned_layers;
   bool                                 srgb;
   bool                                 extended;
   uint32_t                             stride;
   uint64_t                             acceleration_buffer;
   uint32_t                             depth_linear;
   uint32_t                             layer_stride_linear;
   uint32_t                             level_offset_sw;
   uint32_t                             aligned_width_msaa_sw;
   uint32_t                             sample_count_log2_sw;
   uint32_t                             tile_width_sw;
   uint32_t                             tile_height_sw;
   uint32_t                             layer_stride_sw;
   uint32_t                             buffer_offset_sw;
};

#define AGX_PBE_header                          \
   .width = 1,  \
   .height = 1,  \
   .levels = 1,  \
   .layers = 1,  \
   .depth_linear = 1,  \
   .tile_width_sw = 1,  \
   .tile_height_sw = 1

static inline void
AGX_PBE_pack(GLOBAL uint32_t * restrict cl,
             const struct AGX_PBE * restrict values)
{
   assert(values->width >= 1);
   assert(values->height >= 1);
   assert((values->buffer & 0xf) == 0);
   assert(values->levels >= 1);
   assert(values->layers >= 1);
   assert((values->acceleration_buffer & 0xf) == 0);
   assert(values->depth_linear >= 1);
   assert((values->layer_stride_linear & 0x7f) == 0);
   assert((values->level_offset_sw & 0x7f) == 0);
   assert(IS_POT_NONZERO(values->tile_width_sw));
   assert(IS_POT_NONZERO(values->tile_height_sw));
   assert((values->layer_stride_sw & 0x7f) == 0);
   agx_genxml_validate_bounds("PBE::dimension", values->dimension, 0x10ull);
   agx_genxml_validate_bounds("PBE::layout", values->layout, 0x4ull);
   agx_genxml_validate_bounds("PBE::channels", values->channels, 0x80ull);
   agx_genxml_validate_bounds("PBE::type", values->type, 0x8ull);
   agx_genxml_validate_bounds("PBE::swizzle_r", values->swizzle_r, 0x4ull);
   agx_genxml_validate_bounds("PBE::swizzle_g", values->swizzle_g, 0x4ull);
   agx_genxml_validate_bounds("PBE::swizzle_b", values->swizzle_b, 0x4ull);
   agx_genxml_validate_bounds("PBE::swizzle_a", values->swizzle_a, 0x4ull);
   agx_genxml_validate_bounds("PBE::width", values->width - 1, 0x4000ull);
   cl[ 0] = util_bitpack_uint(values->dimension, 0, 3) |
            util_bitpack_uint(values->layout, 4, 5) |
            util_bitpack_uint(values->channels, 6, 12) |
            util_bitpack_uint(values->type, 13, 15) |
            util_bitpack_uint(values->swizzle_r, 16, 17) |
            util_bitpack_uint(values->swizzle_g, 18, 19) |
            util_bitpack_uint(values->swizzle_b, 20, 21) |
            util_bitpack_uint(values->swizzle_a, 22, 23) |
            util_bitpack_uint(values->width - 1, 24, 37);
   agx_genxml_validate_bounds("PBE::width", values->width - 1, 0x4000ull);
   agx_genxml_validate_bounds("PBE::height", values->height - 1, 0x4000ull);
   agx_genxml_validate_bounds("PBE::unk_52", values->unk_52, 0x2ull);
   agx_genxml_validate_bounds("PBE::rotate_90", values->rotate_90, 0x2ull);
   agx_genxml_validate_bounds("PBE::flip_vertical", values->flip_vertical, 0x2ull);
   agx_genxml_validate_bounds("PBE::samples", values->samples, 0x2ull);
   agx_genxml_validate_bounds("PBE::unk_mipmapped", values->unk_mipmapped, 0x2ull);
   agx_genxml_validate_bounds("PBE::compressed_1", values->compressed_1, 0x2ull);
   agx_genxml_validate_bounds("PBE::mode", values->mode, 0x4ull);
   cl[ 1] = util_bitpack_uint(values->width - 1, 24, 37) >> 32 |
            util_bitpack_uint(values->height - 1, 6, 19) |
            util_bitpack_uint(values->unk_52, 20, 20) |
            util_bitpack_uint(values->rotate_90, 21, 21) |
            util_bitpack_uint(values->flip_vertical, 22, 22) |
            util_bitpack_uint(values->samples, 24, 24) |
            util_bitpack_uint(values->unk_mipmapped, 26, 26) |
            util_bitpack_uint(values->compressed_1, 27, 27) |
            util_bitpack_uint(values->mode, 28, 29);
   agx_genxml_validate_bounds("PBE::buffer", values->buffer >> 4, 0x1000000000ull);
   cl[ 2] = util_bitpack_uint(values->buffer >> 4, 0, 35);
   agx_genxml_validate_bounds("PBE::buffer", values->buffer >> 4, 0x1000000000ull);
   agx_genxml_validate_bounds("PBE::level", values->level, 0x10ull);
   agx_genxml_validate_bounds("PBE::levels", values->levels - 1, 0x10ull);
   agx_genxml_validate_bounds("PBE::layers", values->layers - 1, 0x4000ull);
   agx_genxml_validate_bounds("PBE::page_aligned_layers", values->page_aligned_layers, 0x2ull);
   agx_genxml_validate_bounds("PBE::srgb", values->srgb, 0x2ull);
   agx_genxml_validate_bounds("PBE::extended", values->extended, 0x2ull);
   agx_genxml_validate_bounds("PBE::stride", values->stride, 0x200000ull);
   cl[ 3] = util_bitpack_uint(values->buffer >> 4, 0, 35) >> 32 |
            util_bitpack_uint(values->level, 4, 7) |
            util_bitpack_uint(values->levels - 1, 8, 11) |
            util_bitpack_uint(values->layers - 1, 12, 25) |
            util_bitpack_uint(values->page_aligned_layers, 28, 28) |
            util_bitpack_uint(values->srgb, 29, 29) |
            util_bitpack_uint(values->extended, 31, 31) |
            util_bitpack_uint(values->stride, 8, 28);
   agx_genxml_validate_bounds("PBE::depth_linear", values->depth_linear - 1, 0x800ull);
   agx_genxml_validate_bounds("PBE::layer_stride_linear", values->layer_stride_linear, 0x400000000ull);
   agx_genxml_validate_bounds("PBE::level_offset_sw", values->level_offset_sw >> 7, 0x8000000ull);
   agx_genxml_validate_bounds("PBE::aligned_width_msaa_sw", values->aligned_width_msaa_sw, 0x8000ull);
   agx_genxml_validate_bounds("PBE::sample_count_log2_sw", values->sample_count_log2_sw, 0x4ull);
   agx_genxml_validate_bounds("PBE::tile_width_sw", util_logbase2(values->tile_width_sw), 0x10ull);
   agx_genxml_validate_bounds("PBE::tile_height_sw", util_logbase2(values->tile_height_sw), 0x10ull);
   agx_genxml_validate_bounds("PBE::buffer_offset_sw", values->buffer_offset_sw, 0x100000000ull);
   cl[ 4] = util_bitpack_uint(values->acceleration_buffer >> 4, 0, 63) |
            util_bitpack_uint(values->depth_linear - 1, 0, 10) |
            util_bitpack_uint(values->layer_stride_linear, 4, 37) |
            util_bitpack_uint(values->level_offset_sw >> 7, 0, 26) |
            util_bitpack_uint(values->aligned_width_msaa_sw, 0, 14) |
            util_bitpack_uint(values->sample_count_log2_sw, 15, 16) |
            util_bitpack_uint(util_logbase2(values->tile_width_sw), 27, 30) |
            util_bitpack_uint(util_logbase2(values->tile_height_sw), 31, 34) |
            util_bitpack_uint(values->buffer_offset_sw, 0, 31);
   agx_genxml_validate_bounds("PBE::layer_stride_linear", values->layer_stride_linear, 0x400000000ull);
   agx_genxml_validate_bounds("PBE::tile_height_sw", util_logbase2(values->tile_height_sw), 0x10ull);
   agx_genxml_validate_bounds("PBE::layer_stride_sw", values->layer_stride_sw >> 7, 0x8000000ull);
   cl[ 5] = util_bitpack_uint(values->acceleration_buffer >> 4, 0, 63) >> 32 |
            util_bitpack_uint(values->layer_stride_linear, 4, 37) >> 32 |
            util_bitpack_uint(util_logbase2(values->tile_height_sw), 31, 34) >> 32 |
            util_bitpack_uint(values->layer_stride_sw >> 7, 3, 29);
}

#define AGX_PBE_LENGTH 24
struct agx_pbe_packed { uint32_t opaque[6];};
static inline bool
AGX_PBE_unpack(FILE *fp, CONST uint8_t * restrict cl,
               struct AGX_PBE * restrict values)
{
   values->dimension = (enum agx_texture_dimension) __gen_unpack_uint((CONST uint32_t *) cl, 0, 3);
   values->layout = (enum agx_layout) __gen_unpack_uint((CONST uint32_t *) cl, 4, 5);
   values->channels = (enum agx_channels) __gen_unpack_uint((CONST uint32_t *) cl, 6, 12);
   values->type = (enum agx_texture_type) __gen_unpack_uint((CONST uint32_t *) cl, 13, 15);
   values->swizzle_r = (enum agx_channel) __gen_unpack_uint((CONST uint32_t *) cl, 16, 17);
   values->swizzle_g = (enum agx_channel) __gen_unpack_uint((CONST uint32_t *) cl, 18, 19);
   values->swizzle_b = (enum agx_channel) __gen_unpack_uint((CONST uint32_t *) cl, 20, 21);
   values->swizzle_a = (enum agx_channel) __gen_unpack_uint((CONST uint32_t *) cl, 22, 23);
   values->width = __gen_unpack_uint((CONST uint32_t *) cl, 24, 37) + 1;
   values->height = __gen_unpack_uint((CONST uint32_t *) cl, 38, 51) + 1;
   values->unk_52 = __gen_unpack_uint((CONST uint32_t *) cl, 52, 52);
   values->rotate_90 = __gen_unpack_uint((CONST uint32_t *) cl, 53, 53);
   values->flip_vertical = __gen_unpack_uint((CONST uint32_t *) cl, 54, 54);
   values->samples = (enum agx_sample_count) __gen_unpack_uint((CONST uint32_t *) cl, 56, 56);
   values->unk_mipmapped = __gen_unpack_uint((CONST uint32_t *) cl, 58, 58);
   values->compressed_1 = __gen_unpack_uint((CONST uint32_t *) cl, 59, 59);
   values->mode = (enum agx_image_mode) __gen_unpack_uint((CONST uint32_t *) cl, 60, 61);
   values->buffer = __gen_unpack_uint((CONST uint32_t *) cl, 64, 99) << 4;
   values->level = __gen_unpack_uint((CONST uint32_t *) cl, 100, 103);
   values->levels = __gen_unpack_uint((CONST uint32_t *) cl, 104, 107) + 1;
   values->layers = __gen_unpack_uint((CONST uint32_t *) cl, 108, 121) + 1;
   values->page_aligned_layers = __gen_unpack_uint((CONST uint32_t *) cl, 124, 124);
   values->srgb = __gen_unpack_uint((CONST uint32_t *) cl, 125, 125);
   values->extended = __gen_unpack_uint((CONST uint32_t *) cl, 127, 127);
   values->stride = __gen_unpack_uint((CONST uint32_t *) cl, 104, 124);
   values->acceleration_buffer = __gen_unpack_uint((CONST uint32_t *) cl, 128, 191) << 4;
   values->depth_linear = __gen_unpack_uint((CONST uint32_t *) cl, 128, 138) + 1;
   values->layer_stride_linear = __gen_unpack_uint((CONST uint32_t *) cl, 139, 165) << 7;
   values->level_offset_sw = __gen_unpack_uint((CONST uint32_t *) cl, 128, 154) << 7;
   values->aligned_width_msaa_sw = __gen_unpack_uint((CONST uint32_t *) cl, 128, 142);
   values->sample_count_log2_sw = __gen_unpack_uint((CONST uint32_t *) cl, 143, 144);
   values->tile_width_sw = 1 << __gen_unpack_uint((CONST uint32_t *) cl, 155, 158);
   values->tile_height_sw = 1 << __gen_unpack_uint((CONST uint32_t *) cl, 159, 162);
   values->layer_stride_sw = __gen_unpack_uint((CONST uint32_t *) cl, 163, 189) << 7;
   values->buffer_offset_sw = __gen_unpack_uint((CONST uint32_t *) cl, 128, 159);
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "PBE", cl, 1, 0xc2800000);
   valid &= agx_genxml_validate_mask(fp, "PBE", cl, 3, 0x40000000);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_PBE_print(FILE *fp, const struct AGX_PBE * values, unsigned indent)
{
   if (agx_texture_dimension_as_str(values->dimension))
     fprintf(fp, "%*sDimension: %s\n", indent, "", agx_texture_dimension_as_str(values->dimension));
   else
     fprintf(fp, "%*sDimension: unknown %X (XXX)\n", indent, "", values->dimension);
   if (agx_layout_as_str(values->layout))
     fprintf(fp, "%*sLayout: %s\n", indent, "", agx_layout_as_str(values->layout));
   else
     fprintf(fp, "%*sLayout: unknown %X (XXX)\n", indent, "", values->layout);
   if (agx_channels_as_str(values->channels))
     fprintf(fp, "%*sChannels: %s\n", indent, "", agx_channels_as_str(values->channels));
   else
     fprintf(fp, "%*sChannels: unknown %X (XXX)\n", indent, "", values->channels);
   if (agx_texture_type_as_str(values->type))
     fprintf(fp, "%*sType: %s\n", indent, "", agx_texture_type_as_str(values->type));
   else
     fprintf(fp, "%*sType: unknown %X (XXX)\n", indent, "", values->type);
   if (agx_channel_as_str(values->swizzle_r))
     fprintf(fp, "%*sSwizzle R: %s\n", indent, "", agx_channel_as_str(values->swizzle_r));
   else
     fprintf(fp, "%*sSwizzle R: unknown %X (XXX)\n", indent, "", values->swizzle_r);
   if (agx_channel_as_str(values->swizzle_g))
     fprintf(fp, "%*sSwizzle G: %s\n", indent, "", agx_channel_as_str(values->swizzle_g));
   else
     fprintf(fp, "%*sSwizzle G: unknown %X (XXX)\n", indent, "", values->swizzle_g);
   if (agx_channel_as_str(values->swizzle_b))
     fprintf(fp, "%*sSwizzle B: %s\n", indent, "", agx_channel_as_str(values->swizzle_b));
   else
     fprintf(fp, "%*sSwizzle B: unknown %X (XXX)\n", indent, "", values->swizzle_b);
   if (agx_channel_as_str(values->swizzle_a))
     fprintf(fp, "%*sSwizzle A: %s\n", indent, "", agx_channel_as_str(values->swizzle_a));
   else
     fprintf(fp, "%*sSwizzle A: unknown %X (XXX)\n", indent, "", values->swizzle_a);
   fprintf(fp, "%*sWidth: %u\n", indent, "", values->width);
   fprintf(fp, "%*sHeight: %u\n", indent, "", values->height);
   fprintf(fp, "%*sUnk 52: %s\n", indent, "", values->unk_52 ? "true" : "false");
   fprintf(fp, "%*sRotate 90: %s\n", indent, "", values->rotate_90 ? "true" : "false");
   fprintf(fp, "%*sFlip vertical: %s\n", indent, "", values->flip_vertical ? "true" : "false");
   if (agx_sample_count_as_str(values->samples))
     fprintf(fp, "%*sSamples: %s\n", indent, "", agx_sample_count_as_str(values->samples));
   else
     fprintf(fp, "%*sSamples: unknown %X (XXX)\n", indent, "", values->samples);
   fprintf(fp, "%*sUnk mipmapped: %s\n", indent, "", values->unk_mipmapped ? "true" : "false");
   fprintf(fp, "%*sCompressed 1: %s\n", indent, "", values->compressed_1 ? "true" : "false");
   if (agx_image_mode_as_str(values->mode))
     fprintf(fp, "%*sMode: %s\n", indent, "", agx_image_mode_as_str(values->mode));
   else
     fprintf(fp, "%*sMode: unknown %X (XXX)\n", indent, "", values->mode);
   fprintf(fp, "%*sBuffer: 0x%" PRIx64 "\n", indent, "", values->buffer);
   fprintf(fp, "%*sLevel: %u\n", indent, "", values->level);
   fprintf(fp, "%*sLevels: %u\n", indent, "", values->levels);
   fprintf(fp, "%*sLayers: %u\n", indent, "", values->layers);
   fprintf(fp, "%*sPage-aligned layers: %s\n", indent, "", values->page_aligned_layers ? "true" : "false");
   fprintf(fp, "%*ssRGB: %s\n", indent, "", values->srgb ? "true" : "false");
   fprintf(fp, "%*sExtended: %s\n", indent, "", values->extended ? "true" : "false");
   fprintf(fp, "%*sStride: 0x%" PRIx32 "\n", indent, "", values->stride);
   fprintf(fp, "%*sAcceleration buffer: 0x%" PRIx64 "\n", indent, "", values->acceleration_buffer);
   fprintf(fp, "%*sDepth (linear): %u\n", indent, "", values->depth_linear);
   fprintf(fp, "%*sLayer stride (linear): %u\n", indent, "", values->layer_stride_linear);
   fprintf(fp, "%*sLevel offset (sw): %u\n", indent, "", values->level_offset_sw);
   fprintf(fp, "%*sAligned width MSAA (sw): %u\n", indent, "", values->aligned_width_msaa_sw);
   fprintf(fp, "%*sSample count log2 (sw): %u\n", indent, "", values->sample_count_log2_sw);
   fprintf(fp, "%*sTile width (sw): %u\n", indent, "", values->tile_width_sw);
   fprintf(fp, "%*sTile height (sw): %u\n", indent, "", values->tile_height_sw);
   fprintf(fp, "%*sLayer stride (sw): %u\n", indent, "", values->layer_stride_sw);
   fprintf(fp, "%*sBuffer offset (sw): %u\n", indent, "", values->buffer_offset_sw);
}
#endif

struct AGX_TEXTURE {
   enum agx_texture_dimension           dimension;
   enum agx_layout                      layout;
   enum agx_channels                    channels;
   enum agx_texture_type                type;
   enum agx_channel                     swizzle_r;
   enum agx_channel                     swizzle_g;
   enum agx_channel                     swizzle_b;
   enum agx_channel                     swizzle_a;
   uint32_t                             width;
   uint32_t                             height;
   uint32_t                             first_level;
   uint32_t                             last_level;
   enum agx_sample_count                samples;
   uint64_t                             address;
   bool                                 unk_mipmapped;
   bool                                 compressed_1;
   enum agx_image_mode                  mode;
   uint32_t                             compression;
   bool                                 srgb;
   bool                                 srgb_2_channel;
   uint32_t                             stride;
   uint32_t                             depth;
   bool                                 page_aligned_layers;
   bool                                 extended;
   uint64_t                             acceleration_buffer;
   uint32_t                             depth_linear;
   uint32_t                             layer_stride_linear;
   uint32_t                             buffer_size_sw;
   uint32_t                             buffer_offset_sw;
};

#define AGX_TEXTURE_header                      \
   .width = 1,  \
   .height = 1,  \
   .depth = 1,  \
   .depth_linear = 1

static inline void
AGX_TEXTURE_pack(GLOBAL uint32_t * restrict cl,
                 const struct AGX_TEXTURE * restrict values)
{
   assert(values->width >= 1);
   assert(values->height >= 1);
   assert((values->address & 0xf) == 0);
   assert((values->stride & 0xf) == 0);
   assert(values->depth >= 1);
   assert((values->acceleration_buffer & 0xf) == 0);
   assert(values->depth_linear >= 1);
   assert((values->layer_stride_linear & 0x7f) == 0);
   agx_genxml_validate_bounds("Texture::dimension", values->dimension, 0x10ull);
   agx_genxml_validate_bounds("Texture::layout", values->layout, 0x4ull);
   agx_genxml_validate_bounds("Texture::channels", values->channels, 0x80ull);
   agx_genxml_validate_bounds("Texture::type", values->type, 0x8ull);
   agx_genxml_validate_bounds("Texture::swizzle_r", values->swizzle_r, 0x8ull);
   agx_genxml_validate_bounds("Texture::swizzle_g", values->swizzle_g, 0x8ull);
   agx_genxml_validate_bounds("Texture::swizzle_b", values->swizzle_b, 0x8ull);
   agx_genxml_validate_bounds("Texture::swizzle_a", values->swizzle_a, 0x8ull);
   agx_genxml_validate_bounds("Texture::width", values->width - 1, 0x4000ull);
   cl[ 0] = util_bitpack_uint(values->dimension, 0, 3) |
            util_bitpack_uint(values->layout, 4, 5) |
            util_bitpack_uint(values->channels, 6, 12) |
            util_bitpack_uint(values->type, 13, 15) |
            util_bitpack_uint(values->swizzle_r, 16, 18) |
            util_bitpack_uint(values->swizzle_g, 19, 21) |
            util_bitpack_uint(values->swizzle_b, 22, 24) |
            util_bitpack_uint(values->swizzle_a, 25, 27) |
            util_bitpack_uint(values->width - 1, 28, 41);
   agx_genxml_validate_bounds("Texture::width", values->width - 1, 0x4000ull);
   agx_genxml_validate_bounds("Texture::height", values->height - 1, 0x4000ull);
   agx_genxml_validate_bounds("Texture::first_level", values->first_level, 0x10ull);
   agx_genxml_validate_bounds("Texture::last_level", values->last_level, 0x10ull);
   cl[ 1] = util_bitpack_uint(values->width - 1, 28, 41) >> 32 |
            util_bitpack_uint(values->height - 1, 10, 23) |
            util_bitpack_uint(values->first_level, 24, 27) |
            util_bitpack_uint(values->last_level, 28, 31);
   agx_genxml_validate_bounds("Texture::samples", values->samples, 0x2ull);
   agx_genxml_validate_bounds("Texture::address", values->address >> 4, 0x1000000000ull);
   cl[ 2] = util_bitpack_uint(values->samples, 0, 0) |
            util_bitpack_uint(values->address >> 4, 2, 37);
   agx_genxml_validate_bounds("Texture::address", values->address >> 4, 0x1000000000ull);
   agx_genxml_validate_bounds("Texture::unk_mipmapped", values->unk_mipmapped, 0x2ull);
   agx_genxml_validate_bounds("Texture::compressed_1", values->compressed_1, 0x2ull);
   agx_genxml_validate_bounds("Texture::mode", values->mode, 0x4ull);
   agx_genxml_validate_bounds("Texture::compression", values->compression, 0x4ull);
   agx_genxml_validate_bounds("Texture::srgb", values->srgb, 0x2ull);
   agx_genxml_validate_bounds("Texture::srgb_2_channel", values->srgb_2_channel, 0x2ull);
   agx_genxml_validate_bounds("Texture::stride", values->stride, 0x400000ull);
   agx_genxml_validate_bounds("Texture::depth", values->depth - 1, 0x4000ull);
   agx_genxml_validate_bounds("Texture::page_aligned_layers", values->page_aligned_layers, 0x2ull);
   agx_genxml_validate_bounds("Texture::extended", values->extended, 0x2ull);
   cl[ 3] = util_bitpack_uint(values->address >> 4, 2, 37) >> 32 |
            util_bitpack_uint(values->unk_mipmapped, 6, 6) |
            util_bitpack_uint(values->compressed_1, 7, 7) |
            util_bitpack_uint(values->mode, 8, 9) |
            util_bitpack_uint(values->compression, 10, 11) |
            util_bitpack_uint(values->srgb, 12, 12) |
            util_bitpack_uint(values->srgb_2_channel, 13, 13) |
            util_bitpack_uint(values->stride, 10, 31) |
            util_bitpack_uint(values->depth - 1, 14, 27) |
            util_bitpack_uint(values->page_aligned_layers, 30, 30) |
            util_bitpack_uint(values->extended, 31, 31);
   agx_genxml_validate_bounds("Texture::depth_linear", values->depth_linear - 1, 0x800ull);
   agx_genxml_validate_bounds("Texture::layer_stride_linear", values->layer_stride_linear, 0x400000000ull);
   agx_genxml_validate_bounds("Texture::buffer_size_sw", values->buffer_size_sw, 0x100000000ull);
   cl[ 4] = util_bitpack_uint(values->acceleration_buffer >> 4, 0, 63) |
            util_bitpack_uint(values->depth_linear - 1, 0, 10) |
            util_bitpack_uint(values->layer_stride_linear, 4, 37) |
            util_bitpack_uint(values->buffer_size_sw, 0, 31);
   agx_genxml_validate_bounds("Texture::layer_stride_linear", values->layer_stride_linear, 0x400000000ull);
   agx_genxml_validate_bounds("Texture::buffer_offset_sw", values->buffer_offset_sw, 0x100000000ull);
   cl[ 5] = util_bitpack_uint(values->acceleration_buffer >> 4, 0, 63) >> 32 |
            util_bitpack_uint(values->layer_stride_linear, 4, 37) >> 32 |
            util_bitpack_uint(values->buffer_offset_sw, 0, 31);
}

#define AGX_TEXTURE_LENGTH 24
struct agx_texture_packed { uint32_t opaque[6];};
static inline bool
AGX_TEXTURE_unpack(FILE *fp, CONST uint8_t * restrict cl,
                   struct AGX_TEXTURE * restrict values)
{
   values->dimension = (enum agx_texture_dimension) __gen_unpack_uint((CONST uint32_t *) cl, 0, 3);
   values->layout = (enum agx_layout) __gen_unpack_uint((CONST uint32_t *) cl, 4, 5);
   values->channels = (enum agx_channels) __gen_unpack_uint((CONST uint32_t *) cl, 6, 12);
   values->type = (enum agx_texture_type) __gen_unpack_uint((CONST uint32_t *) cl, 13, 15);
   values->swizzle_r = (enum agx_channel) __gen_unpack_uint((CONST uint32_t *) cl, 16, 18);
   values->swizzle_g = (enum agx_channel) __gen_unpack_uint((CONST uint32_t *) cl, 19, 21);
   values->swizzle_b = (enum agx_channel) __gen_unpack_uint((CONST uint32_t *) cl, 22, 24);
   values->swizzle_a = (enum agx_channel) __gen_unpack_uint((CONST uint32_t *) cl, 25, 27);
   values->width = __gen_unpack_uint((CONST uint32_t *) cl, 28, 41) + 1;
   values->height = __gen_unpack_uint((CONST uint32_t *) cl, 42, 55) + 1;
   values->first_level = __gen_unpack_uint((CONST uint32_t *) cl, 56, 59);
   values->last_level = __gen_unpack_uint((CONST uint32_t *) cl, 60, 63);
   values->samples = (enum agx_sample_count) __gen_unpack_uint((CONST uint32_t *) cl, 64, 64);
   values->address = __gen_unpack_uint((CONST uint32_t *) cl, 66, 101) << 4;
   values->unk_mipmapped = __gen_unpack_uint((CONST uint32_t *) cl, 102, 102);
   values->compressed_1 = __gen_unpack_uint((CONST uint32_t *) cl, 103, 103);
   values->mode = (enum agx_image_mode) __gen_unpack_uint((CONST uint32_t *) cl, 104, 105);
   values->compression = __gen_unpack_uint((CONST uint32_t *) cl, 106, 107);
   values->srgb = __gen_unpack_uint((CONST uint32_t *) cl, 108, 108);
   values->srgb_2_channel = __gen_unpack_uint((CONST uint32_t *) cl, 109, 109);
   values->stride = __gen_unpack_uint((CONST uint32_t *) cl, 110, 127) << 4;
   values->depth = __gen_unpack_uint((CONST uint32_t *) cl, 110, 123) + 1;
   values->page_aligned_layers = __gen_unpack_uint((CONST uint32_t *) cl, 126, 126);
   values->extended = __gen_unpack_uint((CONST uint32_t *) cl, 127, 127);
   values->acceleration_buffer = __gen_unpack_uint((CONST uint32_t *) cl, 128, 191) << 4;
   values->depth_linear = __gen_unpack_uint((CONST uint32_t *) cl, 128, 138) + 1;
   values->layer_stride_linear = __gen_unpack_uint((CONST uint32_t *) cl, 139, 165) << 7;
   values->buffer_size_sw = __gen_unpack_uint((CONST uint32_t *) cl, 128, 159);
   values->buffer_offset_sw = __gen_unpack_uint((CONST uint32_t *) cl, 160, 191);
   return agx_genxml_validate_mask(fp, "Texture", cl, 2, 0x2);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_TEXTURE_print(FILE *fp, const struct AGX_TEXTURE * values, unsigned indent)
{
   if (agx_texture_dimension_as_str(values->dimension))
     fprintf(fp, "%*sDimension: %s\n", indent, "", agx_texture_dimension_as_str(values->dimension));
   else
     fprintf(fp, "%*sDimension: unknown %X (XXX)\n", indent, "", values->dimension);
   if (agx_layout_as_str(values->layout))
     fprintf(fp, "%*sLayout: %s\n", indent, "", agx_layout_as_str(values->layout));
   else
     fprintf(fp, "%*sLayout: unknown %X (XXX)\n", indent, "", values->layout);
   if (agx_channels_as_str(values->channels))
     fprintf(fp, "%*sChannels: %s\n", indent, "", agx_channels_as_str(values->channels));
   else
     fprintf(fp, "%*sChannels: unknown %X (XXX)\n", indent, "", values->channels);
   if (agx_texture_type_as_str(values->type))
     fprintf(fp, "%*sType: %s\n", indent, "", agx_texture_type_as_str(values->type));
   else
     fprintf(fp, "%*sType: unknown %X (XXX)\n", indent, "", values->type);
   if (agx_channel_as_str(values->swizzle_r))
     fprintf(fp, "%*sSwizzle R: %s\n", indent, "", agx_channel_as_str(values->swizzle_r));
   else
     fprintf(fp, "%*sSwizzle R: unknown %X (XXX)\n", indent, "", values->swizzle_r);
   if (agx_channel_as_str(values->swizzle_g))
     fprintf(fp, "%*sSwizzle G: %s\n", indent, "", agx_channel_as_str(values->swizzle_g));
   else
     fprintf(fp, "%*sSwizzle G: unknown %X (XXX)\n", indent, "", values->swizzle_g);
   if (agx_channel_as_str(values->swizzle_b))
     fprintf(fp, "%*sSwizzle B: %s\n", indent, "", agx_channel_as_str(values->swizzle_b));
   else
     fprintf(fp, "%*sSwizzle B: unknown %X (XXX)\n", indent, "", values->swizzle_b);
   if (agx_channel_as_str(values->swizzle_a))
     fprintf(fp, "%*sSwizzle A: %s\n", indent, "", agx_channel_as_str(values->swizzle_a));
   else
     fprintf(fp, "%*sSwizzle A: unknown %X (XXX)\n", indent, "", values->swizzle_a);
   fprintf(fp, "%*sWidth: %u\n", indent, "", values->width);
   fprintf(fp, "%*sHeight: %u\n", indent, "", values->height);
   fprintf(fp, "%*sFirst level: %u\n", indent, "", values->first_level);
   fprintf(fp, "%*sLast level: %u\n", indent, "", values->last_level);
   if (agx_sample_count_as_str(values->samples))
     fprintf(fp, "%*sSamples: %s\n", indent, "", agx_sample_count_as_str(values->samples));
   else
     fprintf(fp, "%*sSamples: unknown %X (XXX)\n", indent, "", values->samples);
   fprintf(fp, "%*sAddress: 0x%" PRIx64 "\n", indent, "", values->address);
   fprintf(fp, "%*sUnk mipmapped: %s\n", indent, "", values->unk_mipmapped ? "true" : "false");
   fprintf(fp, "%*sCompressed 1: %s\n", indent, "", values->compressed_1 ? "true" : "false");
   if (agx_image_mode_as_str(values->mode))
     fprintf(fp, "%*sMode: %s\n", indent, "", agx_image_mode_as_str(values->mode));
   else
     fprintf(fp, "%*sMode: unknown %X (XXX)\n", indent, "", values->mode);
   fprintf(fp, "%*sCompression: 0x%" PRIx32 "\n", indent, "", values->compression);
   fprintf(fp, "%*ssRGB: %s\n", indent, "", values->srgb ? "true" : "false");
   fprintf(fp, "%*ssRGB 2-channel: %s\n", indent, "", values->srgb_2_channel ? "true" : "false");
   fprintf(fp, "%*sStride: 0x%" PRIx32 "\n", indent, "", values->stride);
   fprintf(fp, "%*sDepth: %u\n", indent, "", values->depth);
   fprintf(fp, "%*sPage-aligned layers: %s\n", indent, "", values->page_aligned_layers ? "true" : "false");
   fprintf(fp, "%*sExtended: %s\n", indent, "", values->extended ? "true" : "false");
   fprintf(fp, "%*sAcceleration buffer: 0x%" PRIx64 "\n", indent, "", values->acceleration_buffer);
   fprintf(fp, "%*sDepth (linear): %u\n", indent, "", values->depth_linear);
   fprintf(fp, "%*sLayer stride (linear): %u\n", indent, "", values->layer_stride_linear);
   fprintf(fp, "%*sBuffer size (sw): %u\n", indent, "", values->buffer_size_sw);
   fprintf(fp, "%*sBuffer offset (sw): %u\n", indent, "", values->buffer_offset_sw);
}
#endif

enum agx_wrap {
   AGX_WRAP_CLAMP_TO_EDGE = 0,
   AGX_WRAP_REPEAT = 1,
   AGX_WRAP_MIRRORED_REPEAT = 2,
   AGX_WRAP_CLAMP_TO_BORDER = 3,
   AGX_WRAP_CLAMP_GL = 4,
   AGX_WRAP_MIRRORED_CLAMP_TO_EDGE = 5,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_wrap_as_str(enum agx_wrap imm)
{
    switch (imm) {
    case AGX_WRAP_CLAMP_TO_EDGE: return "Clamp to edge";
    case AGX_WRAP_REPEAT: return "Repeat";
    case AGX_WRAP_MIRRORED_REPEAT: return "Mirrored repeat";
    case AGX_WRAP_CLAMP_TO_BORDER: return "Clamp to border";
    case AGX_WRAP_CLAMP_GL: return "Clamp (GL)";
    case AGX_WRAP_MIRRORED_CLAMP_TO_EDGE: return "Mirrored clamp to edge";
    default: return NULL;
    }
}
#endif

enum agx_mip_filter {
   AGX_MIP_FILTER_NONE = 0,
   AGX_MIP_FILTER_NEAREST = 1,
   AGX_MIP_FILTER_LINEAR = 2,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_mip_filter_as_str(enum agx_mip_filter imm)
{
    switch (imm) {
    case AGX_MIP_FILTER_NONE: return "None";
    case AGX_MIP_FILTER_NEAREST: return "Nearest";
    case AGX_MIP_FILTER_LINEAR: return "Linear";
    default: return NULL;
    }
}
#endif

enum agx_border_colour {
   AGX_BORDER_COLOUR_TRANSPARENT_BLACK = 0,
   AGX_BORDER_COLOUR_OPAQUE_BLACK = 1,
   AGX_BORDER_COLOUR_OPAQUE_WHITE = 2,
   AGX_BORDER_COLOUR_CUSTOM = 3,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_border_colour_as_str(enum agx_border_colour imm)
{
    switch (imm) {
    case AGX_BORDER_COLOUR_TRANSPARENT_BLACK: return "Transparent black";
    case AGX_BORDER_COLOUR_OPAQUE_BLACK: return "Opaque black";
    case AGX_BORDER_COLOUR_OPAQUE_WHITE: return "Opaque white";
    case AGX_BORDER_COLOUR_CUSTOM: return "Custom";
    default: return NULL;
    }
}
#endif

enum agx_filter {
   AGX_FILTER_NEAREST = 0,
   AGX_FILTER_LINEAR = 1,
   AGX_FILTER_BICUBIC = 2,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_filter_as_str(enum agx_filter imm)
{
    switch (imm) {
    case AGX_FILTER_NEAREST: return "Nearest";
    case AGX_FILTER_LINEAR: return "Linear";
    case AGX_FILTER_BICUBIC: return "Bicubic";
    default: return NULL;
    }
}
#endif

struct AGX_SAMPLER {
   float                                minimum_lod;
   float                                maximum_lod;
   uint32_t                             maximum_anisotropy;
   enum agx_filter                      magnify;
   enum agx_filter                      minify;
   enum agx_mip_filter                  mip_filter;
   enum agx_wrap                        wrap_s;
   enum agx_wrap                        wrap_t;
   enum agx_wrap                        wrap_r;
   bool                                 pixel_coordinates;
   enum agx_compare_func                compare_func;
   bool                                 compare_enable;
   enum agx_border_colour               border_colour;
   bool                                 seamful_cube_maps;
};

#define AGX_SAMPLER_header                      \
   .maximum_anisotropy = 1

static inline void
AGX_SAMPLER_pack(GLOBAL uint32_t * restrict cl,
                 const struct AGX_SAMPLER * restrict values)
{
   assert(IS_POT_NONZERO(values->maximum_anisotropy));
   agx_genxml_validate_bounds("Sampler::maximum_anisotropy", util_logbase2(values->maximum_anisotropy), 0x8ull);
   agx_genxml_validate_bounds("Sampler::magnify", values->magnify, 0x4ull);
   agx_genxml_validate_bounds("Sampler::minify", values->minify, 0x4ull);
   agx_genxml_validate_bounds("Sampler::mip_filter", values->mip_filter, 0x4ull);
   agx_genxml_validate_bounds("Sampler::wrap_s", values->wrap_s, 0x8ull);
   cl[ 0] = __gen_pack_lod(values->minimum_lod, 0, 9) |
            __gen_pack_lod(values->maximum_lod, 10, 19) |
            util_bitpack_uint(util_logbase2(values->maximum_anisotropy), 20, 22) |
            util_bitpack_uint(values->magnify, 23, 24) |
            util_bitpack_uint(values->minify, 25, 26) |
            util_bitpack_uint(values->mip_filter, 27, 28) |
            util_bitpack_uint(values->wrap_s, 29, 31);
   agx_genxml_validate_bounds("Sampler::wrap_t", values->wrap_t, 0x8ull);
   agx_genxml_validate_bounds("Sampler::wrap_r", values->wrap_r, 0x8ull);
   agx_genxml_validate_bounds("Sampler::pixel_coordinates", values->pixel_coordinates, 0x2ull);
   agx_genxml_validate_bounds("Sampler::compare_func", values->compare_func, 0x8ull);
   agx_genxml_validate_bounds("Sampler::compare_enable", values->compare_enable, 0x2ull);
   agx_genxml_validate_bounds("Sampler::border_colour", values->border_colour, 0x4ull);
   agx_genxml_validate_bounds("Sampler::seamful_cube_maps", values->seamful_cube_maps, 0x2ull);
   cl[ 1] = util_bitpack_uint(values->wrap_t, 0, 2) |
            util_bitpack_uint(values->wrap_r, 3, 5) |
            util_bitpack_uint(values->pixel_coordinates, 6, 6) |
            util_bitpack_uint(values->compare_func, 7, 9) |
            util_bitpack_uint(values->compare_enable, 10, 10) |
            util_bitpack_uint(values->border_colour, 23, 24) |
            util_bitpack_uint(values->seamful_cube_maps, 25, 25);
}

#define AGX_SAMPLER_LENGTH 8
struct agx_sampler_packed { uint32_t opaque[2];};
static inline bool
AGX_SAMPLER_unpack(FILE *fp, CONST uint8_t * restrict cl,
                   struct AGX_SAMPLER * restrict values)
{
   values->minimum_lod = __gen_unpack_lod((CONST uint32_t *) cl, 0, 9);
   values->maximum_lod = __gen_unpack_lod((CONST uint32_t *) cl, 10, 19);
   values->maximum_anisotropy = 1 << __gen_unpack_uint((CONST uint32_t *) cl, 20, 22);
   values->magnify = (enum agx_filter) __gen_unpack_uint((CONST uint32_t *) cl, 23, 24);
   values->minify = (enum agx_filter) __gen_unpack_uint((CONST uint32_t *) cl, 25, 26);
   values->mip_filter = (enum agx_mip_filter) __gen_unpack_uint((CONST uint32_t *) cl, 27, 28);
   values->wrap_s = (enum agx_wrap) __gen_unpack_uint((CONST uint32_t *) cl, 29, 31);
   values->wrap_t = (enum agx_wrap) __gen_unpack_uint((CONST uint32_t *) cl, 32, 34);
   values->wrap_r = (enum agx_wrap) __gen_unpack_uint((CONST uint32_t *) cl, 35, 37);
   values->pixel_coordinates = __gen_unpack_uint((CONST uint32_t *) cl, 38, 38);
   values->compare_func = (enum agx_compare_func) __gen_unpack_uint((CONST uint32_t *) cl, 39, 41);
   values->compare_enable = __gen_unpack_uint((CONST uint32_t *) cl, 42, 42);
   values->border_colour = (enum agx_border_colour) __gen_unpack_uint((CONST uint32_t *) cl, 55, 56);
   values->seamful_cube_maps = __gen_unpack_uint((CONST uint32_t *) cl, 57, 57);
   return agx_genxml_validate_mask(fp, "Sampler", cl, 1, 0xfc7ff800);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_SAMPLER_print(FILE *fp, const struct AGX_SAMPLER * values, unsigned indent)
{
   fprintf(fp, "%*sMinimum LOD: %f\n", indent, "", values->minimum_lod);
   fprintf(fp, "%*sMaximum LOD: %f\n", indent, "", values->maximum_lod);
   fprintf(fp, "%*sMaximum anisotropy: %u\n", indent, "", values->maximum_anisotropy);
   if (agx_filter_as_str(values->magnify))
     fprintf(fp, "%*sMagnify: %s\n", indent, "", agx_filter_as_str(values->magnify));
   else
     fprintf(fp, "%*sMagnify: unknown %X (XXX)\n", indent, "", values->magnify);
   if (agx_filter_as_str(values->minify))
     fprintf(fp, "%*sMinify: %s\n", indent, "", agx_filter_as_str(values->minify));
   else
     fprintf(fp, "%*sMinify: unknown %X (XXX)\n", indent, "", values->minify);
   if (agx_mip_filter_as_str(values->mip_filter))
     fprintf(fp, "%*sMip filter: %s\n", indent, "", agx_mip_filter_as_str(values->mip_filter));
   else
     fprintf(fp, "%*sMip filter: unknown %X (XXX)\n", indent, "", values->mip_filter);
   if (agx_wrap_as_str(values->wrap_s))
     fprintf(fp, "%*sWrap S: %s\n", indent, "", agx_wrap_as_str(values->wrap_s));
   else
     fprintf(fp, "%*sWrap S: unknown %X (XXX)\n", indent, "", values->wrap_s);
   if (agx_wrap_as_str(values->wrap_t))
     fprintf(fp, "%*sWrap T: %s\n", indent, "", agx_wrap_as_str(values->wrap_t));
   else
     fprintf(fp, "%*sWrap T: unknown %X (XXX)\n", indent, "", values->wrap_t);
   if (agx_wrap_as_str(values->wrap_r))
     fprintf(fp, "%*sWrap R: %s\n", indent, "", agx_wrap_as_str(values->wrap_r));
   else
     fprintf(fp, "%*sWrap R: unknown %X (XXX)\n", indent, "", values->wrap_r);
   fprintf(fp, "%*sPixel coordinates: %s\n", indent, "", values->pixel_coordinates ? "true" : "false");
   if (agx_compare_func_as_str(values->compare_func))
     fprintf(fp, "%*sCompare func: %s\n", indent, "", agx_compare_func_as_str(values->compare_func));
   else
     fprintf(fp, "%*sCompare func: unknown %X (XXX)\n", indent, "", values->compare_func);
   fprintf(fp, "%*sCompare enable: %s\n", indent, "", values->compare_enable ? "true" : "false");
   if (agx_border_colour_as_str(values->border_colour))
     fprintf(fp, "%*sBorder colour: %s\n", indent, "", agx_border_colour_as_str(values->border_colour));
   else
     fprintf(fp, "%*sBorder colour: unknown %X (XXX)\n", indent, "", values->border_colour);
   fprintf(fp, "%*sSeamful cube maps: %s\n", indent, "", values->seamful_cube_maps ? "true" : "false");
}
#endif

struct AGX_BORDER {
   uint32_t                             channel_0;
   uint32_t                             channel_1;
   uint32_t                             channel_2;
   uint32_t                             channel_3;
};

#define AGX_BORDER_header 0

static inline void
AGX_BORDER_pack(GLOBAL uint32_t * restrict cl,
                const struct AGX_BORDER * restrict values)
{
   agx_genxml_validate_bounds("Border::channel_0", values->channel_0, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->channel_0, 0, 31);
   agx_genxml_validate_bounds("Border::channel_1", values->channel_1, 0x100000000ull);
   cl[ 1] = util_bitpack_uint(values->channel_1, 0, 31);
   agx_genxml_validate_bounds("Border::channel_2", values->channel_2, 0x100000000ull);
   cl[ 2] = util_bitpack_uint(values->channel_2, 0, 31);
   agx_genxml_validate_bounds("Border::channel_3", values->channel_3, 0x100000000ull);
   cl[ 3] = util_bitpack_uint(values->channel_3, 0, 31);
}

#define AGX_BORDER_LENGTH 16
struct agx_border_packed { uint32_t opaque[4];};
static inline bool
AGX_BORDER_unpack(FILE *fp, CONST uint8_t * restrict cl,
                  struct AGX_BORDER * restrict values)
{
   values->channel_0 = __gen_unpack_uint((CONST uint32_t *) cl, 0, 31);
   values->channel_1 = __gen_unpack_uint((CONST uint32_t *) cl, 32, 63);
   values->channel_2 = __gen_unpack_uint((CONST uint32_t *) cl, 64, 95);
   values->channel_3 = __gen_unpack_uint((CONST uint32_t *) cl, 96, 127);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_BORDER_print(FILE *fp, const struct AGX_BORDER * values, unsigned indent)
{
   fprintf(fp, "%*sChannel 0: 0x%" PRIx32 "\n", indent, "", values->channel_0);
   fprintf(fp, "%*sChannel 1: 0x%" PRIx32 "\n", indent, "", values->channel_1);
   fprintf(fp, "%*sChannel 2: 0x%" PRIx32 "\n", indent, "", values->channel_2);
   fprintf(fp, "%*sChannel 3: 0x%" PRIx32 "\n", indent, "", values->channel_3);
}
#endif

struct AGX_SCISSOR {
   uint32_t                             max_x;
   uint32_t                             min_x;
   uint32_t                             max_y;
   uint32_t                             min_y;
   float                                min_z;
   float                                max_z;
};

#define AGX_SCISSOR_header 0

static inline void
AGX_SCISSOR_pack(GLOBAL uint32_t * restrict cl,
                 const struct AGX_SCISSOR * restrict values)
{
   agx_genxml_validate_bounds("Scissor::max_x", values->max_x, 0x10000ull);
   agx_genxml_validate_bounds("Scissor::min_x", values->min_x, 0x10000ull);
   cl[ 0] = util_bitpack_uint(values->max_x, 0, 15) |
            util_bitpack_uint(values->min_x, 16, 31);
   agx_genxml_validate_bounds("Scissor::max_y", values->max_y, 0x10000ull);
   agx_genxml_validate_bounds("Scissor::min_y", values->min_y, 0x10000ull);
   cl[ 1] = util_bitpack_uint(values->max_y, 0, 15) |
            util_bitpack_uint(values->min_y, 16, 31);
   cl[ 2] = util_bitpack_float(values->min_z);
   cl[ 3] = util_bitpack_float(values->max_z);
}

#define AGX_SCISSOR_LENGTH 16
struct agx_scissor_packed { uint32_t opaque[4];};
static inline bool
AGX_SCISSOR_unpack(FILE *fp, CONST uint8_t * restrict cl,
                   struct AGX_SCISSOR * restrict values)
{
   values->max_x = __gen_unpack_uint((CONST uint32_t *) cl, 0, 15);
   values->min_x = __gen_unpack_uint((CONST uint32_t *) cl, 16, 31);
   values->max_y = __gen_unpack_uint((CONST uint32_t *) cl, 32, 47);
   values->min_y = __gen_unpack_uint((CONST uint32_t *) cl, 48, 63);
   values->min_z = __gen_unpack_float((CONST uint32_t *) cl, 64, 95);
   values->max_z = __gen_unpack_float((CONST uint32_t *) cl, 96, 127);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_SCISSOR_print(FILE *fp, const struct AGX_SCISSOR * values, unsigned indent)
{
   fprintf(fp, "%*sMax X: %u\n", indent, "", values->max_x);
   fprintf(fp, "%*sMin X: %u\n", indent, "", values->min_x);
   fprintf(fp, "%*sMax Y: %u\n", indent, "", values->max_y);
   fprintf(fp, "%*sMin Y: %u\n", indent, "", values->min_y);
   fprintf(fp, "%*sMin Z: %f\n", indent, "", values->min_z);
   fprintf(fp, "%*sMax Z: %f\n", indent, "", values->max_z);
}
#endif

struct AGX_DEPTH_BIAS {
   float                                depth_bias;
   float                                slope_scale;
   float                                clamp;
};

#define AGX_DEPTH_BIAS_header 0

static inline void
AGX_DEPTH_BIAS_pack(GLOBAL uint32_t * restrict cl,
                    const struct AGX_DEPTH_BIAS * restrict values)
{
   cl[ 0] = util_bitpack_float(values->depth_bias);
   cl[ 1] = util_bitpack_float(values->slope_scale);
   cl[ 2] = util_bitpack_float(values->clamp);
}

#define AGX_DEPTH_BIAS_LENGTH 12
struct agx_depth_bias_packed { uint32_t opaque[3];};
static inline bool
AGX_DEPTH_BIAS_unpack(FILE *fp, CONST uint8_t * restrict cl,
                      struct AGX_DEPTH_BIAS * restrict values)
{
   values->depth_bias = __gen_unpack_float((CONST uint32_t *) cl, 0, 31);
   values->slope_scale = __gen_unpack_float((CONST uint32_t *) cl, 32, 63);
   values->clamp = __gen_unpack_float((CONST uint32_t *) cl, 64, 95);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_DEPTH_BIAS_print(FILE *fp, const struct AGX_DEPTH_BIAS * values, unsigned indent)
{
   fprintf(fp, "%*sDepth bias: %f\n", indent, "", values->depth_bias);
   fprintf(fp, "%*sSlope scale: %f\n", indent, "", values->slope_scale);
   fprintf(fp, "%*sClamp: %f\n", indent, "", values->clamp);
}
#endif

enum agx_sampler_states {
   AGX_SAMPLER_STATES_0 = 0,
   AGX_SAMPLER_STATES_4_COMPACT = 1,
   AGX_SAMPLER_STATES_8_COMPACT = 2,
   AGX_SAMPLER_STATES_12_COMPACT = 3,
   AGX_SAMPLER_STATES_16_COMPACT = 4,
   AGX_SAMPLER_STATES_8_EXTENDED = 6,
   AGX_SAMPLER_STATES_16_EXTENDED = 7,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_sampler_states_as_str(enum agx_sampler_states imm)
{
    switch (imm) {
    case AGX_SAMPLER_STATES_0: return "0";
    case AGX_SAMPLER_STATES_4_COMPACT: return "4 compact";
    case AGX_SAMPLER_STATES_8_COMPACT: return "8 compact";
    case AGX_SAMPLER_STATES_12_COMPACT: return "12 compact";
    case AGX_SAMPLER_STATES_16_COMPACT: return "16 compact";
    case AGX_SAMPLER_STATES_8_EXTENDED: return "8 extended";
    case AGX_SAMPLER_STATES_16_EXTENDED: return "16 extended";
    default: return NULL;
    }
}
#endif

struct AGX_COUNTS {
   uint32_t                             unknown_0;
   uint32_t                             uniform_register_count;
   uint32_t                             texture_state_register_count;
   enum agx_sampler_states              sampler_state_register_count;
   uint32_t                             preshader_register_count;
   uint32_t                             cf_binding_count;
   uint32_t                             unknown;
};

#define AGX_COUNTS_header                       \
   .uniform_register_count = 512,  \
   .texture_state_register_count = 256,  \
   .preshader_register_count = 256

static inline void
AGX_COUNTS_pack(GLOBAL uint32_t * restrict cl,
                const struct AGX_COUNTS * restrict values)
{
   agx_genxml_validate_bounds("Counts::unknown_0", values->unknown_0, 0x2ull);
   agx_genxml_validate_bounds("Counts::uniform_register_count", __gen_to_groups(values->uniform_register_count, 64, 3), 0x8ull);
   agx_genxml_validate_bounds("Counts::texture_state_register_count", __gen_to_groups(values->texture_state_register_count, 8, 5), 0x20ull);
   agx_genxml_validate_bounds("Counts::sampler_state_register_count", values->sampler_state_register_count, 0x8ull);
   agx_genxml_validate_bounds("Counts::preshader_register_count", __gen_to_groups(values->preshader_register_count, 16, 4), 0x10ull);
   agx_genxml_validate_bounds("Counts::cf_binding_count", values->cf_binding_count, 0x80ull);
   agx_genxml_validate_bounds("Counts::unknown", values->unknown, 0x10000ull);
   cl[ 0] = util_bitpack_uint(values->unknown_0, 0, 0) |
            util_bitpack_uint(__gen_to_groups(values->uniform_register_count, 64, 3), 1, 3) |
            util_bitpack_uint(__gen_to_groups(values->texture_state_register_count, 8, 5), 4, 8) |
            util_bitpack_uint(values->sampler_state_register_count, 9, 11) |
            util_bitpack_uint(__gen_to_groups(values->preshader_register_count, 16, 4), 12, 15) |
            util_bitpack_uint(values->cf_binding_count, 16, 22) |
            util_bitpack_uint(values->unknown, 16, 31);
}

#define AGX_COUNTS_LENGTH 4
struct agx_counts_packed { uint32_t opaque[1];};
static inline bool
AGX_COUNTS_unpack(FILE *fp, CONST uint8_t * restrict cl,
                  struct AGX_COUNTS * restrict values)
{
   values->unknown_0 = __gen_unpack_uint((CONST uint32_t *) cl, 0, 0);
   values->uniform_register_count = __gen_from_groups(__gen_unpack_uint((CONST uint32_t *) cl, 1, 3), 64, 3);
   values->texture_state_register_count = __gen_from_groups(__gen_unpack_uint((CONST uint32_t *) cl, 4, 8), 8, 5);
   values->sampler_state_register_count = (enum agx_sampler_states) __gen_unpack_uint((CONST uint32_t *) cl, 9, 11);
   values->preshader_register_count = __gen_from_groups(__gen_unpack_uint((CONST uint32_t *) cl, 12, 15), 16, 4);
   values->cf_binding_count = __gen_unpack_uint((CONST uint32_t *) cl, 16, 22);
   values->unknown = __gen_unpack_uint((CONST uint32_t *) cl, 16, 31);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_COUNTS_print(FILE *fp, const struct AGX_COUNTS * values, unsigned indent)
{
   fprintf(fp, "%*sUnknown 0: 0x%" PRIx32 "\n", indent, "", values->unknown_0);
   fprintf(fp, "%*sUniform register count: %u\n", indent, "", values->uniform_register_count);
   fprintf(fp, "%*sTexture state register count: %u\n", indent, "", values->texture_state_register_count);
   if (agx_sampler_states_as_str(values->sampler_state_register_count))
     fprintf(fp, "%*sSampler state register count: %s\n", indent, "", agx_sampler_states_as_str(values->sampler_state_register_count));
   else
     fprintf(fp, "%*sSampler state register count: unknown %X (XXX)\n", indent, "", values->sampler_state_register_count);
   fprintf(fp, "%*sPreshader register count: %u\n", indent, "", values->preshader_register_count);
   fprintf(fp, "%*sCF binding count: %u\n", indent, "", values->cf_binding_count);
   fprintf(fp, "%*sUnknown: 0x%" PRIx32 "\n", indent, "", values->unknown);
}
#endif

struct AGX_PPP_HEADER {
   bool                                 fragment_control;
   bool                                 fragment_control_2;
   bool                                 fragment_front_face;
   bool                                 fragment_front_face_2;
   bool                                 fragment_front_stencil;
   bool                                 fragment_back_face;
   bool                                 fragment_back_face_2;
   bool                                 fragment_back_stencil;
   bool                                 depth_bias_scissor;
   bool                                 region_clip;
   bool                                 viewport;
   uint32_t                             viewport_count;
   bool                                 w_clamp;
   bool                                 output_select;
   bool                                 varying_counts_32;
   bool                                 varying_counts_16;
   bool                                 cull;
   bool                                 cull_2;
   bool                                 fragment_shader;
   bool                                 occlusion_query;
   bool                                 occlusion_query_2;
   bool                                 output_unknown;
   bool                                 output_size;
   bool                                 varying_word_2;
};

#define AGX_PPP_HEADER_header                   \
   .viewport_count = 1

static inline void
AGX_PPP_HEADER_pack(GLOBAL uint32_t * restrict cl,
                    const struct AGX_PPP_HEADER * restrict values)
{
   assert(values->viewport_count >= 1);
   agx_genxml_validate_bounds("PPP Header::fragment_control", values->fragment_control, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::fragment_control_2", values->fragment_control_2, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::fragment_front_face", values->fragment_front_face, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::fragment_front_face_2", values->fragment_front_face_2, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::fragment_front_stencil", values->fragment_front_stencil, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::fragment_back_face", values->fragment_back_face, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::fragment_back_face_2", values->fragment_back_face_2, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::fragment_back_stencil", values->fragment_back_stencil, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::depth_bias_scissor", values->depth_bias_scissor, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::region_clip", values->region_clip, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::viewport", values->viewport, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::viewport_count", values->viewport_count - 1, 0x10ull);
   agx_genxml_validate_bounds("PPP Header::w_clamp", values->w_clamp, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::output_select", values->output_select, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::varying_counts_32", values->varying_counts_32, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::varying_counts_16", values->varying_counts_16, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::cull", values->cull, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::cull_2", values->cull_2, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::fragment_shader", values->fragment_shader, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::occlusion_query", values->occlusion_query, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::occlusion_query_2", values->occlusion_query_2, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::output_unknown", values->output_unknown, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::output_size", values->output_size, 0x2ull);
   agx_genxml_validate_bounds("PPP Header::varying_word_2", values->varying_word_2, 0x2ull);
   cl[ 0] = util_bitpack_uint(values->fragment_control, 0, 0) |
            util_bitpack_uint(values->fragment_control_2, 1, 1) |
            util_bitpack_uint(values->fragment_front_face, 2, 2) |
            util_bitpack_uint(values->fragment_front_face_2, 3, 3) |
            util_bitpack_uint(values->fragment_front_stencil, 4, 4) |
            util_bitpack_uint(values->fragment_back_face, 5, 5) |
            util_bitpack_uint(values->fragment_back_face_2, 6, 6) |
            util_bitpack_uint(values->fragment_back_stencil, 7, 7) |
            util_bitpack_uint(values->depth_bias_scissor, 8, 8) |
            util_bitpack_uint(values->region_clip, 10, 10) |
            util_bitpack_uint(values->viewport, 11, 11) |
            util_bitpack_uint(values->viewport_count - 1, 12, 15) |
            util_bitpack_uint(values->w_clamp, 16, 16) |
            util_bitpack_uint(values->output_select, 17, 17) |
            util_bitpack_uint(values->varying_counts_32, 18, 18) |
            util_bitpack_uint(values->varying_counts_16, 19, 19) |
            util_bitpack_uint(values->cull, 21, 21) |
            util_bitpack_uint(values->cull_2, 22, 22) |
            util_bitpack_uint(values->fragment_shader, 23, 23) |
            util_bitpack_uint(values->occlusion_query, 24, 24) |
            util_bitpack_uint(values->occlusion_query_2, 25, 25) |
            util_bitpack_uint(values->output_unknown, 26, 26) |
            util_bitpack_uint(values->output_size, 27, 27) |
            util_bitpack_uint(values->varying_word_2, 28, 28);
}

#define AGX_PPP_HEADER_LENGTH 4
struct agx_ppp_header_packed { uint32_t opaque[1];};
static inline bool
AGX_PPP_HEADER_unpack(FILE *fp, CONST uint8_t * restrict cl,
                      struct AGX_PPP_HEADER * restrict values)
{
   values->fragment_control = __gen_unpack_uint((CONST uint32_t *) cl, 0, 0);
   values->fragment_control_2 = __gen_unpack_uint((CONST uint32_t *) cl, 1, 1);
   values->fragment_front_face = __gen_unpack_uint((CONST uint32_t *) cl, 2, 2);
   values->fragment_front_face_2 = __gen_unpack_uint((CONST uint32_t *) cl, 3, 3);
   values->fragment_front_stencil = __gen_unpack_uint((CONST uint32_t *) cl, 4, 4);
   values->fragment_back_face = __gen_unpack_uint((CONST uint32_t *) cl, 5, 5);
   values->fragment_back_face_2 = __gen_unpack_uint((CONST uint32_t *) cl, 6, 6);
   values->fragment_back_stencil = __gen_unpack_uint((CONST uint32_t *) cl, 7, 7);
   values->depth_bias_scissor = __gen_unpack_uint((CONST uint32_t *) cl, 8, 8);
   values->region_clip = __gen_unpack_uint((CONST uint32_t *) cl, 10, 10);
   values->viewport = __gen_unpack_uint((CONST uint32_t *) cl, 11, 11);
   values->viewport_count = __gen_unpack_uint((CONST uint32_t *) cl, 12, 15) + 1;
   values->w_clamp = __gen_unpack_uint((CONST uint32_t *) cl, 16, 16);
   values->output_select = __gen_unpack_uint((CONST uint32_t *) cl, 17, 17);
   values->varying_counts_32 = __gen_unpack_uint((CONST uint32_t *) cl, 18, 18);
   values->varying_counts_16 = __gen_unpack_uint((CONST uint32_t *) cl, 19, 19);
   values->cull = __gen_unpack_uint((CONST uint32_t *) cl, 21, 21);
   values->cull_2 = __gen_unpack_uint((CONST uint32_t *) cl, 22, 22);
   values->fragment_shader = __gen_unpack_uint((CONST uint32_t *) cl, 23, 23);
   values->occlusion_query = __gen_unpack_uint((CONST uint32_t *) cl, 24, 24);
   values->occlusion_query_2 = __gen_unpack_uint((CONST uint32_t *) cl, 25, 25);
   values->output_unknown = __gen_unpack_uint((CONST uint32_t *) cl, 26, 26);
   values->output_size = __gen_unpack_uint((CONST uint32_t *) cl, 27, 27);
   values->varying_word_2 = __gen_unpack_uint((CONST uint32_t *) cl, 28, 28);
   return agx_genxml_validate_mask(fp, "PPP Header", cl, 0, 0xe0100200);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_PPP_HEADER_print(FILE *fp, const struct AGX_PPP_HEADER * values, unsigned indent)
{
   fprintf(fp, "%*sFragment control: %s\n", indent, "", values->fragment_control ? "true" : "false");
   fprintf(fp, "%*sFragment control 2: %s\n", indent, "", values->fragment_control_2 ? "true" : "false");
   fprintf(fp, "%*sFragment front face: %s\n", indent, "", values->fragment_front_face ? "true" : "false");
   fprintf(fp, "%*sFragment front face 2: %s\n", indent, "", values->fragment_front_face_2 ? "true" : "false");
   fprintf(fp, "%*sFragment front stencil: %s\n", indent, "", values->fragment_front_stencil ? "true" : "false");
   fprintf(fp, "%*sFragment back face: %s\n", indent, "", values->fragment_back_face ? "true" : "false");
   fprintf(fp, "%*sFragment back face 2: %s\n", indent, "", values->fragment_back_face_2 ? "true" : "false");
   fprintf(fp, "%*sFragment back stencil: %s\n", indent, "", values->fragment_back_stencil ? "true" : "false");
   fprintf(fp, "%*sDepth bias/scissor: %s\n", indent, "", values->depth_bias_scissor ? "true" : "false");
   fprintf(fp, "%*sRegion clip: %s\n", indent, "", values->region_clip ? "true" : "false");
   fprintf(fp, "%*sViewport: %s\n", indent, "", values->viewport ? "true" : "false");
   fprintf(fp, "%*sViewport count: %u\n", indent, "", values->viewport_count);
   fprintf(fp, "%*sW clamp: %s\n", indent, "", values->w_clamp ? "true" : "false");
   fprintf(fp, "%*sOutput select: %s\n", indent, "", values->output_select ? "true" : "false");
   fprintf(fp, "%*sVarying counts 32: %s\n", indent, "", values->varying_counts_32 ? "true" : "false");
   fprintf(fp, "%*sVarying counts 16: %s\n", indent, "", values->varying_counts_16 ? "true" : "false");
   fprintf(fp, "%*sCull: %s\n", indent, "", values->cull ? "true" : "false");
   fprintf(fp, "%*sCull 2: %s\n", indent, "", values->cull_2 ? "true" : "false");
   fprintf(fp, "%*sFragment shader: %s\n", indent, "", values->fragment_shader ? "true" : "false");
   fprintf(fp, "%*sOcclusion query: %s\n", indent, "", values->occlusion_query ? "true" : "false");
   fprintf(fp, "%*sOcclusion query 2: %s\n", indent, "", values->occlusion_query_2 ? "true" : "false");
   fprintf(fp, "%*sOutput unknown: %s\n", indent, "", values->output_unknown ? "true" : "false");
   fprintf(fp, "%*sOutput size: %s\n", indent, "", values->output_size ? "true" : "false");
   fprintf(fp, "%*sVarying word 2: %s\n", indent, "", values->varying_word_2 ? "true" : "false");
}
#endif

struct AGX_REGION_CLIP {
   uint32_t                             max_x;
   uint32_t                             min_x;
   bool                                 enable;
   uint32_t                             max_y;
   uint32_t                             min_y;
};

#define AGX_REGION_CLIP_header                  \
   .max_x = 1,  \
   .max_y = 1

static inline void
AGX_REGION_CLIP_pack(GLOBAL uint32_t * restrict cl,
                     const struct AGX_REGION_CLIP * restrict values)
{
   assert(values->max_x >= 1);
   assert(values->max_y >= 1);
   agx_genxml_validate_bounds("Region clip::max_x", values->max_x - 1, 0x200ull);
   agx_genxml_validate_bounds("Region clip::min_x", values->min_x, 0x200ull);
   agx_genxml_validate_bounds("Region clip::enable", values->enable, 0x2ull);
   cl[ 0] = util_bitpack_uint(values->max_x - 1, 0, 8) |
            util_bitpack_uint(values->min_x, 16, 24) |
            util_bitpack_uint(values->enable, 31, 31);
   agx_genxml_validate_bounds("Region clip::max_y", values->max_y - 1, 0x200ull);
   agx_genxml_validate_bounds("Region clip::min_y", values->min_y, 0x200ull);
   cl[ 1] = util_bitpack_uint(values->max_y - 1, 0, 8) |
            util_bitpack_uint(values->min_y, 16, 24);
}

#define AGX_REGION_CLIP_LENGTH 8
struct agx_region_clip_packed { uint32_t opaque[2];};
static inline bool
AGX_REGION_CLIP_unpack(FILE *fp, CONST uint8_t * restrict cl,
                       struct AGX_REGION_CLIP * restrict values)
{
   values->max_x = __gen_unpack_uint((CONST uint32_t *) cl, 0, 8) + 1;
   values->min_x = __gen_unpack_uint((CONST uint32_t *) cl, 16, 24);
   values->enable = __gen_unpack_uint((CONST uint32_t *) cl, 31, 31);
   values->max_y = __gen_unpack_uint((CONST uint32_t *) cl, 32, 40) + 1;
   values->min_y = __gen_unpack_uint((CONST uint32_t *) cl, 48, 56);
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "Region clip", cl, 0, 0x7e00fe00);
   valid &= agx_genxml_validate_mask(fp, "Region clip", cl, 1, 0xfe00fe00);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_REGION_CLIP_print(FILE *fp, const struct AGX_REGION_CLIP * values, unsigned indent)
{
   fprintf(fp, "%*sMax X: %u\n", indent, "", values->max_x);
   fprintf(fp, "%*sMin X: %u\n", indent, "", values->min_x);
   fprintf(fp, "%*sEnable: %s\n", indent, "", values->enable ? "true" : "false");
   fprintf(fp, "%*sMax Y: %u\n", indent, "", values->max_y);
   fprintf(fp, "%*sMin Y: %u\n", indent, "", values->min_y);
}
#endif

struct AGX_VIEWPORT_CONTROL {
   int dummy;
};

#define AGX_VIEWPORT_CONTROL_header 0

static inline void
AGX_VIEWPORT_CONTROL_pack(GLOBAL uint32_t * restrict cl,
                          const struct AGX_VIEWPORT_CONTROL * restrict values)
{
   cl[ 0] = 0;
}

#define AGX_VIEWPORT_CONTROL_LENGTH 4
struct agx_viewport_control_packed { uint32_t opaque[1];};
static inline bool
AGX_VIEWPORT_CONTROL_unpack(FILE *fp, CONST uint8_t * restrict cl,
                            struct AGX_VIEWPORT_CONTROL * restrict values)
{
   return agx_genxml_validate_mask(fp, "Viewport control", cl, 0, 0xffffffff);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VIEWPORT_CONTROL_print(FILE *fp, const struct AGX_VIEWPORT_CONTROL * values, unsigned indent)
{
}
#endif

struct AGX_VIEWPORT {
   float                                translate_x;
   float                                scale_x;
   float                                translate_y;
   float                                scale_y;
   float                                translate_z;
   float                                scale_z;
};

#define AGX_VIEWPORT_header 0

static inline void
AGX_VIEWPORT_pack(GLOBAL uint32_t * restrict cl,
                  const struct AGX_VIEWPORT * restrict values)
{
   cl[ 0] = util_bitpack_float(values->translate_x);
   cl[ 1] = util_bitpack_float(values->scale_x);
   cl[ 2] = util_bitpack_float(values->translate_y);
   cl[ 3] = util_bitpack_float(values->scale_y);
   cl[ 4] = util_bitpack_float(values->translate_z);
   cl[ 5] = util_bitpack_float(values->scale_z);
}

#define AGX_VIEWPORT_LENGTH 24
struct agx_viewport_packed { uint32_t opaque[6];};
static inline bool
AGX_VIEWPORT_unpack(FILE *fp, CONST uint8_t * restrict cl,
                    struct AGX_VIEWPORT * restrict values)
{
   values->translate_x = __gen_unpack_float((CONST uint32_t *) cl, 0, 31);
   values->scale_x = __gen_unpack_float((CONST uint32_t *) cl, 32, 63);
   values->translate_y = __gen_unpack_float((CONST uint32_t *) cl, 64, 95);
   values->scale_y = __gen_unpack_float((CONST uint32_t *) cl, 96, 127);
   values->translate_z = __gen_unpack_float((CONST uint32_t *) cl, 128, 159);
   values->scale_z = __gen_unpack_float((CONST uint32_t *) cl, 160, 191);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VIEWPORT_print(FILE *fp, const struct AGX_VIEWPORT * values, unsigned indent)
{
   fprintf(fp, "%*sTranslate X: %f\n", indent, "", values->translate_x);
   fprintf(fp, "%*sScale X: %f\n", indent, "", values->scale_x);
   fprintf(fp, "%*sTranslate Y: %f\n", indent, "", values->translate_y);
   fprintf(fp, "%*sScale Y: %f\n", indent, "", values->scale_y);
   fprintf(fp, "%*sTranslate Z: %f\n", indent, "", values->translate_z);
   fprintf(fp, "%*sScale Z: %f\n", indent, "", values->scale_z);
}
#endif

struct AGX_FRAGMENT_FACE {
   uint32_t                             stencil_reference;
   uint32_t                             line_width;
   enum agx_polygon_mode                polygon_mode;
   bool                                 disable_depth_write;
   uint32_t                             unk___visibility_test_internal;
   enum agx_zs_func                     depth_function;
};

#define AGX_FRAGMENT_FACE_header 0

static inline void
AGX_FRAGMENT_FACE_pack(GLOBAL uint32_t * restrict cl,
                       const struct AGX_FRAGMENT_FACE * restrict values)
{
   agx_genxml_validate_bounds("Fragment face::stencil_reference", values->stencil_reference, 0x100ull);
   agx_genxml_validate_bounds("Fragment face::line_width", values->line_width, 0x100ull);
   agx_genxml_validate_bounds("Fragment face::polygon_mode", values->polygon_mode, 0x4ull);
   agx_genxml_validate_bounds("Fragment face::disable_depth_write", values->disable_depth_write, 0x2ull);
   agx_genxml_validate_bounds("Fragment face::unk___visibility_test_internal", values->unk___visibility_test_internal, 0x4ull);
   agx_genxml_validate_bounds("Fragment face::depth_function", values->depth_function, 0x8ull);
   cl[ 0] = util_bitpack_uint(values->stencil_reference, 0, 7) |
            util_bitpack_uint(values->line_width, 8, 15) |
            util_bitpack_uint(values->polygon_mode, 18, 19) |
            util_bitpack_uint(values->disable_depth_write, 21, 21) |
            util_bitpack_uint(values->unk___visibility_test_internal, 22, 23) |
            util_bitpack_uint(values->depth_function, 24, 26);
}

#define AGX_FRAGMENT_FACE_LENGTH 4
struct agx_fragment_face_packed { uint32_t opaque[1];};
static inline bool
AGX_FRAGMENT_FACE_unpack(FILE *fp, CONST uint8_t * restrict cl,
                         struct AGX_FRAGMENT_FACE * restrict values)
{
   values->stencil_reference = __gen_unpack_uint((CONST uint32_t *) cl, 0, 7);
   values->line_width = __gen_unpack_uint((CONST uint32_t *) cl, 8, 15);
   values->polygon_mode = (enum agx_polygon_mode) __gen_unpack_uint((CONST uint32_t *) cl, 18, 19);
   values->disable_depth_write = __gen_unpack_uint((CONST uint32_t *) cl, 21, 21);
   values->unk___visibility_test_internal = __gen_unpack_uint((CONST uint32_t *) cl, 22, 23);
   values->depth_function = (enum agx_zs_func) __gen_unpack_uint((CONST uint32_t *) cl, 24, 26);
   return agx_genxml_validate_mask(fp, "Fragment face", cl, 0, 0xf8130000);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_FRAGMENT_FACE_print(FILE *fp, const struct AGX_FRAGMENT_FACE * values, unsigned indent)
{
   fprintf(fp, "%*sStencil reference: 0x%" PRIx32 "\n", indent, "", values->stencil_reference);
   fprintf(fp, "%*sLine width: 0x%" PRIx32 "\n", indent, "", values->line_width);
   if (agx_polygon_mode_as_str(values->polygon_mode))
     fprintf(fp, "%*sPolygon mode: %s\n", indent, "", agx_polygon_mode_as_str(values->polygon_mode));
   else
     fprintf(fp, "%*sPolygon mode: unknown %X (XXX)\n", indent, "", values->polygon_mode);
   fprintf(fp, "%*sDisable depth write: %s\n", indent, "", values->disable_depth_write ? "true" : "false");
   fprintf(fp, "%*sUnk - visibility test internal: 0x%" PRIx32 "\n", indent, "", values->unk___visibility_test_internal);
   if (agx_zs_func_as_str(values->depth_function))
     fprintf(fp, "%*sDepth function: %s\n", indent, "", agx_zs_func_as_str(values->depth_function));
   else
     fprintf(fp, "%*sDepth function: unknown %X (XXX)\n", indent, "", values->depth_function);
}
#endif

enum agx_conservative_depth {
   AGX_CONSERVATIVE_DEPTH_ANY = 0,
   AGX_CONSERVATIVE_DEPTH_GREATER = 1,
   AGX_CONSERVATIVE_DEPTH_LESS = 2,
   AGX_CONSERVATIVE_DEPTH_UNCHANGED = 3,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_conservative_depth_as_str(enum agx_conservative_depth imm)
{
    switch (imm) {
    case AGX_CONSERVATIVE_DEPTH_ANY: return "Any";
    case AGX_CONSERVATIVE_DEPTH_GREATER: return "Greater";
    case AGX_CONSERVATIVE_DEPTH_LESS: return "Less";
    case AGX_CONSERVATIVE_DEPTH_UNCHANGED: return "Unchanged";
    default: return NULL;
    }
}
#endif

struct AGX_FRAGMENT_FACE_2 {
   bool                                 disable_depth_write;
   enum agx_conservative_depth          conservative_depth;
   enum agx_zs_func                     depth_function;
   enum agx_object_type                 object_type;
};

#define AGX_FRAGMENT_FACE_2_header 0

static inline void
AGX_FRAGMENT_FACE_2_pack(GLOBAL uint32_t * restrict cl,
                         const struct AGX_FRAGMENT_FACE_2 * restrict values)
{
   agx_genxml_validate_bounds("Fragment face 2::disable_depth_write", values->disable_depth_write, 0x2ull);
   agx_genxml_validate_bounds("Fragment face 2::conservative_depth", values->conservative_depth, 0x4ull);
   agx_genxml_validate_bounds("Fragment face 2::depth_function", values->depth_function, 0x8ull);
   agx_genxml_validate_bounds("Fragment face 2::object_type", values->object_type, 0x10ull);
   cl[ 0] = util_bitpack_uint(values->disable_depth_write, 21, 21) |
            util_bitpack_uint(values->conservative_depth, 22, 23) |
            util_bitpack_uint(values->depth_function, 24, 26) |
            util_bitpack_uint(values->object_type, 28, 31);
}

#define AGX_FRAGMENT_FACE_2_LENGTH 4
struct agx_fragment_face_2_packed { uint32_t opaque[1];};
static inline bool
AGX_FRAGMENT_FACE_2_unpack(FILE *fp, CONST uint8_t * restrict cl,
                           struct AGX_FRAGMENT_FACE_2 * restrict values)
{
   values->disable_depth_write = __gen_unpack_uint((CONST uint32_t *) cl, 21, 21);
   values->conservative_depth = (enum agx_conservative_depth) __gen_unpack_uint((CONST uint32_t *) cl, 22, 23);
   values->depth_function = (enum agx_zs_func) __gen_unpack_uint((CONST uint32_t *) cl, 24, 26);
   values->object_type = (enum agx_object_type) __gen_unpack_uint((CONST uint32_t *) cl, 28, 31);
   return agx_genxml_validate_mask(fp, "Fragment face 2", cl, 0, 0x81fffff);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_FRAGMENT_FACE_2_print(FILE *fp, const struct AGX_FRAGMENT_FACE_2 * values, unsigned indent)
{
   fprintf(fp, "%*sDisable depth write: %s\n", indent, "", values->disable_depth_write ? "true" : "false");
   if (agx_conservative_depth_as_str(values->conservative_depth))
     fprintf(fp, "%*sConservative depth: %s\n", indent, "", agx_conservative_depth_as_str(values->conservative_depth));
   else
     fprintf(fp, "%*sConservative depth: unknown %X (XXX)\n", indent, "", values->conservative_depth);
   if (agx_zs_func_as_str(values->depth_function))
     fprintf(fp, "%*sDepth function: %s\n", indent, "", agx_zs_func_as_str(values->depth_function));
   else
     fprintf(fp, "%*sDepth function: unknown %X (XXX)\n", indent, "", values->depth_function);
   if (agx_object_type_as_str(values->object_type))
     fprintf(fp, "%*sObject type: %s\n", indent, "", agx_object_type_as_str(values->object_type));
   else
     fprintf(fp, "%*sObject type: unknown %X (XXX)\n", indent, "", values->object_type);
}
#endif

struct AGX_FRAGMENT_STENCIL {
   uint32_t                             write_mask;
   uint32_t                             read_mask;
   enum agx_stencil_op                  depth_pass;
   enum agx_stencil_op                  depth_fail;
   enum agx_stencil_op                  stencil_fail;
   enum agx_zs_func                     compare;
};

#define AGX_FRAGMENT_STENCIL_header 0

static inline void
AGX_FRAGMENT_STENCIL_pack(GLOBAL uint32_t * restrict cl,
                          const struct AGX_FRAGMENT_STENCIL * restrict values)
{
   agx_genxml_validate_bounds("Fragment stencil::write_mask", values->write_mask, 0x100ull);
   agx_genxml_validate_bounds("Fragment stencil::read_mask", values->read_mask, 0x100ull);
   agx_genxml_validate_bounds("Fragment stencil::depth_pass", values->depth_pass, 0x8ull);
   agx_genxml_validate_bounds("Fragment stencil::depth_fail", values->depth_fail, 0x8ull);
   agx_genxml_validate_bounds("Fragment stencil::stencil_fail", values->stencil_fail, 0x8ull);
   agx_genxml_validate_bounds("Fragment stencil::compare", values->compare, 0x8ull);
   cl[ 0] = util_bitpack_uint(values->write_mask, 0, 7) |
            util_bitpack_uint(values->read_mask, 8, 15) |
            util_bitpack_uint(values->depth_pass, 16, 18) |
            util_bitpack_uint(values->depth_fail, 19, 21) |
            util_bitpack_uint(values->stencil_fail, 22, 24) |
            util_bitpack_uint(values->compare, 25, 27);
}

#define AGX_FRAGMENT_STENCIL_LENGTH 4
struct agx_fragment_stencil_packed { uint32_t opaque[1];};
static inline bool
AGX_FRAGMENT_STENCIL_unpack(FILE *fp, CONST uint8_t * restrict cl,
                            struct AGX_FRAGMENT_STENCIL * restrict values)
{
   values->write_mask = __gen_unpack_uint((CONST uint32_t *) cl, 0, 7);
   values->read_mask = __gen_unpack_uint((CONST uint32_t *) cl, 8, 15);
   values->depth_pass = (enum agx_stencil_op) __gen_unpack_uint((CONST uint32_t *) cl, 16, 18);
   values->depth_fail = (enum agx_stencil_op) __gen_unpack_uint((CONST uint32_t *) cl, 19, 21);
   values->stencil_fail = (enum agx_stencil_op) __gen_unpack_uint((CONST uint32_t *) cl, 22, 24);
   values->compare = (enum agx_zs_func) __gen_unpack_uint((CONST uint32_t *) cl, 25, 27);
   return agx_genxml_validate_mask(fp, "Fragment stencil", cl, 0, 0xf0000000);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_FRAGMENT_STENCIL_print(FILE *fp, const struct AGX_FRAGMENT_STENCIL * values, unsigned indent)
{
   fprintf(fp, "%*sWrite mask: 0x%" PRIx32 "\n", indent, "", values->write_mask);
   fprintf(fp, "%*sRead mask: 0x%" PRIx32 "\n", indent, "", values->read_mask);
   if (agx_stencil_op_as_str(values->depth_pass))
     fprintf(fp, "%*sDepth pass: %s\n", indent, "", agx_stencil_op_as_str(values->depth_pass));
   else
     fprintf(fp, "%*sDepth pass: unknown %X (XXX)\n", indent, "", values->depth_pass);
   if (agx_stencil_op_as_str(values->depth_fail))
     fprintf(fp, "%*sDepth fail: %s\n", indent, "", agx_stencil_op_as_str(values->depth_fail));
   else
     fprintf(fp, "%*sDepth fail: unknown %X (XXX)\n", indent, "", values->depth_fail);
   if (agx_stencil_op_as_str(values->stencil_fail))
     fprintf(fp, "%*sStencil fail: %s\n", indent, "", agx_stencil_op_as_str(values->stencil_fail));
   else
     fprintf(fp, "%*sStencil fail: unknown %X (XXX)\n", indent, "", values->stencil_fail);
   if (agx_zs_func_as_str(values->compare))
     fprintf(fp, "%*sCompare: %s\n", indent, "", agx_zs_func_as_str(values->compare));
   else
     fprintf(fp, "%*sCompare: unknown %X (XXX)\n", indent, "", values->compare);
}
#endif

enum agx_pass_type {
   AGX_PASS_TYPE_OPAQUE = 0,
   AGX_PASS_TYPE_TRANSLUCENT = 1,
   AGX_PASS_TYPE_PUNCH_THROUGH = 2,
   AGX_PASS_TYPE_TRANSLUCENT_PUNCH_THROUGH = 3,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_pass_type_as_str(enum agx_pass_type imm)
{
    switch (imm) {
    case AGX_PASS_TYPE_OPAQUE: return "Opaque";
    case AGX_PASS_TYPE_TRANSLUCENT: return "Translucent";
    case AGX_PASS_TYPE_PUNCH_THROUGH: return "Punch through";
    case AGX_PASS_TYPE_TRANSLUCENT_PUNCH_THROUGH: return "Translucent punch through";
    default: return NULL;
    }
}
#endif

struct AGX_FRAGMENT_CONTROL {
   enum agx_visibility_mode             visibility_mode;
   bool                                 scissor_enable;
   bool                                 depth_bias_enable;
   bool                                 stencil_test_enable;
   bool                                 two_sided_stencil;
   bool                                 tag_write_disable;
   bool                                 sample_mask_after_depth_stencil;
   bool                                 disable_tri_merging;
   enum agx_pass_type                   pass_type;
};

#define AGX_FRAGMENT_CONTROL_header 0

static inline void
AGX_FRAGMENT_CONTROL_pack(GLOBAL uint32_t * restrict cl,
                          const struct AGX_FRAGMENT_CONTROL * restrict values)
{
   agx_genxml_validate_bounds("Fragment control::visibility_mode", values->visibility_mode, 0x4ull);
   agx_genxml_validate_bounds("Fragment control::scissor_enable", values->scissor_enable, 0x2ull);
   agx_genxml_validate_bounds("Fragment control::depth_bias_enable", values->depth_bias_enable, 0x2ull);
   agx_genxml_validate_bounds("Fragment control::stencil_test_enable", values->stencil_test_enable, 0x2ull);
   agx_genxml_validate_bounds("Fragment control::two_sided_stencil", values->two_sided_stencil, 0x2ull);
   agx_genxml_validate_bounds("Fragment control::tag_write_disable", values->tag_write_disable, 0x2ull);
   agx_genxml_validate_bounds("Fragment control::sample_mask_after_depth_stencil", values->sample_mask_after_depth_stencil, 0x2ull);
   agx_genxml_validate_bounds("Fragment control::disable_tri_merging", values->disable_tri_merging, 0x2ull);
   agx_genxml_validate_bounds("Fragment control::pass_type", values->pass_type, 0x8ull);
   cl[ 0] = util_bitpack_uint(true, 9, 9) |
            util_bitpack_uint(values->visibility_mode, 14, 15) |
            util_bitpack_uint(values->scissor_enable, 16, 16) |
            util_bitpack_uint(values->depth_bias_enable, 17, 17) |
            util_bitpack_uint(values->stencil_test_enable, 18, 18) |
            util_bitpack_uint(values->two_sided_stencil, 19, 19) |
            util_bitpack_uint(values->tag_write_disable, 21, 21) |
            util_bitpack_uint(values->sample_mask_after_depth_stencil, 25, 25) |
            util_bitpack_uint(values->disable_tri_merging, 26, 26) |
            util_bitpack_uint(values->pass_type, 29, 31);
}

#define AGX_FRAGMENT_CONTROL_LENGTH 4
struct agx_fragment_control_packed { uint32_t opaque[1];};
static inline bool
AGX_FRAGMENT_CONTROL_unpack(FILE *fp, CONST uint8_t * restrict cl,
                            struct AGX_FRAGMENT_CONTROL * restrict values)
{
   values->visibility_mode = (enum agx_visibility_mode) __gen_unpack_uint((CONST uint32_t *) cl, 14, 15);
   values->scissor_enable = __gen_unpack_uint((CONST uint32_t *) cl, 16, 16);
   values->depth_bias_enable = __gen_unpack_uint((CONST uint32_t *) cl, 17, 17);
   values->stencil_test_enable = __gen_unpack_uint((CONST uint32_t *) cl, 18, 18);
   values->two_sided_stencil = __gen_unpack_uint((CONST uint32_t *) cl, 19, 19);
   values->tag_write_disable = __gen_unpack_uint((CONST uint32_t *) cl, 21, 21);
   values->sample_mask_after_depth_stencil = __gen_unpack_uint((CONST uint32_t *) cl, 25, 25);
   values->disable_tri_merging = __gen_unpack_uint((CONST uint32_t *) cl, 26, 26);
   values->pass_type = (enum agx_pass_type) __gen_unpack_uint((CONST uint32_t *) cl, 29, 31);
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "Fragment control", cl, 0, 0x19d03dff);
   valid &= agx_genxml_validate_exact(fp, "Fragment control", __gen_unpack_uint((CONST uint32_t *) cl, 9, 9), true);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_FRAGMENT_CONTROL_print(FILE *fp, const struct AGX_FRAGMENT_CONTROL * values, unsigned indent)
{
   if (agx_visibility_mode_as_str(values->visibility_mode))
     fprintf(fp, "%*sVisibility mode: %s\n", indent, "", agx_visibility_mode_as_str(values->visibility_mode));
   else
     fprintf(fp, "%*sVisibility mode: unknown %X (XXX)\n", indent, "", values->visibility_mode);
   fprintf(fp, "%*sScissor enable: %s\n", indent, "", values->scissor_enable ? "true" : "false");
   fprintf(fp, "%*sDepth bias enable: %s\n", indent, "", values->depth_bias_enable ? "true" : "false");
   fprintf(fp, "%*sStencil test enable: %s\n", indent, "", values->stencil_test_enable ? "true" : "false");
   fprintf(fp, "%*sTwo-sided stencil: %s\n", indent, "", values->two_sided_stencil ? "true" : "false");
   fprintf(fp, "%*sTag write disable: %s\n", indent, "", values->tag_write_disable ? "true" : "false");
   fprintf(fp, "%*sSample mask after depth/stencil: %s\n", indent, "", values->sample_mask_after_depth_stencil ? "true" : "false");
   fprintf(fp, "%*sDisable tri merging: %s\n", indent, "", values->disable_tri_merging ? "true" : "false");
   if (agx_pass_type_as_str(values->pass_type))
     fprintf(fp, "%*sPass type: %s\n", indent, "", agx_pass_type_as_str(values->pass_type));
   else
     fprintf(fp, "%*sPass type: unknown %X (XXX)\n", indent, "", values->pass_type);
}
#endif

struct AGX_FRAGMENT_OCCLUSION_QUERY {
   uint32_t                             index;
};

#define AGX_FRAGMENT_OCCLUSION_QUERY_header 0

static inline void
AGX_FRAGMENT_OCCLUSION_QUERY_pack(GLOBAL uint32_t * restrict cl,
                                  const struct AGX_FRAGMENT_OCCLUSION_QUERY * restrict values)
{
   agx_genxml_validate_bounds("Fragment occlusion query::index", values->index, 0x8000ull);
   cl[ 0] = util_bitpack_uint(values->index, 17, 31);
}

#define AGX_FRAGMENT_OCCLUSION_QUERY_LENGTH 4
struct agx_fragment_occlusion_query_packed { uint32_t opaque[1];};
static inline bool
AGX_FRAGMENT_OCCLUSION_QUERY_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                    struct AGX_FRAGMENT_OCCLUSION_QUERY * restrict values)
{
   values->index = __gen_unpack_uint((CONST uint32_t *) cl, 17, 31);
   return agx_genxml_validate_mask(fp, "Fragment occlusion query", cl, 0, 0x1ffff);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_FRAGMENT_OCCLUSION_QUERY_print(FILE *fp, const struct AGX_FRAGMENT_OCCLUSION_QUERY * values, unsigned indent)
{
   fprintf(fp, "%*sIndex: %u\n", indent, "", values->index);
}
#endif

struct AGX_FRAGMENT_OCCLUSION_QUERY_2 {
   uint32_t                             unknown;
};

#define AGX_FRAGMENT_OCCLUSION_QUERY_2_header 0

static inline void
AGX_FRAGMENT_OCCLUSION_QUERY_2_pack(GLOBAL uint32_t * restrict cl,
                                    const struct AGX_FRAGMENT_OCCLUSION_QUERY_2 * restrict values)
{
   agx_genxml_validate_bounds("Fragment occlusion query 2::unknown", values->unknown, 0x20000ull);
   cl[ 0] = util_bitpack_uint(values->unknown, 0, 16);
}

#define AGX_FRAGMENT_OCCLUSION_QUERY_2_LENGTH 4
struct agx_fragment_occlusion_query_2_packed { uint32_t opaque[1];};
static inline bool
AGX_FRAGMENT_OCCLUSION_QUERY_2_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                      struct AGX_FRAGMENT_OCCLUSION_QUERY_2 * restrict values)
{
   values->unknown = __gen_unpack_uint((CONST uint32_t *) cl, 0, 16);
   return agx_genxml_validate_mask(fp, "Fragment occlusion query 2", cl, 0, 0xfffe0000);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_FRAGMENT_OCCLUSION_QUERY_2_print(FILE *fp, const struct AGX_FRAGMENT_OCCLUSION_QUERY_2 * values, unsigned indent)
{
   fprintf(fp, "%*sUnknown: 0x%" PRIx32 "\n", indent, "", values->unknown);
}
#endif

struct AGX_W_CLAMP {
   float                                w_clamp;
};

#define AGX_W_CLAMP_header 0

static inline void
AGX_W_CLAMP_pack(GLOBAL uint32_t * restrict cl,
                 const struct AGX_W_CLAMP * restrict values)
{
   cl[ 0] = util_bitpack_float(values->w_clamp);
}

#define AGX_W_CLAMP_LENGTH 4
struct agx_w_clamp_packed { uint32_t opaque[1];};
static inline bool
AGX_W_CLAMP_unpack(FILE *fp, CONST uint8_t * restrict cl,
                   struct AGX_W_CLAMP * restrict values)
{
   values->w_clamp = __gen_unpack_float((CONST uint32_t *) cl, 0, 31);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_W_CLAMP_print(FILE *fp, const struct AGX_W_CLAMP * values, unsigned indent)
{
   fprintf(fp, "%*sW Clamp: %f\n", indent, "", values->w_clamp);
}
#endif

enum agx_ppp_vertex {
   AGX_PPP_VERTEX_0 = 1,
   AGX_PPP_VERTEX_1 = 2,
   AGX_PPP_VERTEX_2 = 3,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_ppp_vertex_as_str(enum agx_ppp_vertex imm)
{
    switch (imm) {
    case AGX_PPP_VERTEX_0: return "0";
    case AGX_PPP_VERTEX_1: return "1";
    case AGX_PPP_VERTEX_2: return "2";
    default: return NULL;
    }
}
#endif

struct AGX_CULL {
   bool                                 cull_front;
   bool                                 cull_back;
   enum agx_ppp_vertex                  flat_shading_vertex;
   bool                                 depth_clip;
   bool                                 depth_clamp;
   bool                                 primitive_msaa;
   bool                                 front_face_ccw;
   bool                                 rasterizer_discard;
};

#define AGX_CULL_header 0

static inline void
AGX_CULL_pack(GLOBAL uint32_t * restrict cl,
              const struct AGX_CULL * restrict values)
{
   agx_genxml_validate_bounds("Cull::cull_front", values->cull_front, 0x2ull);
   agx_genxml_validate_bounds("Cull::cull_back", values->cull_back, 0x2ull);
   agx_genxml_validate_bounds("Cull::flat_shading_vertex", values->flat_shading_vertex, 0x4ull);
   agx_genxml_validate_bounds("Cull::depth_clip", values->depth_clip, 0x2ull);
   agx_genxml_validate_bounds("Cull::depth_clamp", values->depth_clamp, 0x2ull);
   agx_genxml_validate_bounds("Cull::primitive_msaa", values->primitive_msaa, 0x2ull);
   agx_genxml_validate_bounds("Cull::front_face_ccw", values->front_face_ccw, 0x2ull);
   agx_genxml_validate_bounds("Cull::rasterizer_discard", values->rasterizer_discard, 0x2ull);
   cl[ 0] = util_bitpack_uint(values->cull_front, 0, 0) |
            util_bitpack_uint(values->cull_back, 1, 1) |
            util_bitpack_uint(values->flat_shading_vertex, 7, 8) |
            util_bitpack_uint(values->depth_clip, 10, 10) |
            util_bitpack_uint(values->depth_clamp, 11, 11) |
            util_bitpack_uint(values->primitive_msaa, 15, 15) |
            util_bitpack_uint(values->front_face_ccw, 16, 16) |
            util_bitpack_uint(values->rasterizer_discard, 17, 17);
}

#define AGX_CULL_LENGTH 4
struct agx_cull_packed { uint32_t opaque[1];};
static inline bool
AGX_CULL_unpack(FILE *fp, CONST uint8_t * restrict cl,
                struct AGX_CULL * restrict values)
{
   values->cull_front = __gen_unpack_uint((CONST uint32_t *) cl, 0, 0);
   values->cull_back = __gen_unpack_uint((CONST uint32_t *) cl, 1, 1);
   values->flat_shading_vertex = (enum agx_ppp_vertex) __gen_unpack_uint((CONST uint32_t *) cl, 7, 8);
   values->depth_clip = __gen_unpack_uint((CONST uint32_t *) cl, 10, 10);
   values->depth_clamp = __gen_unpack_uint((CONST uint32_t *) cl, 11, 11);
   values->primitive_msaa = __gen_unpack_uint((CONST uint32_t *) cl, 15, 15);
   values->front_face_ccw = __gen_unpack_uint((CONST uint32_t *) cl, 16, 16);
   values->rasterizer_discard = __gen_unpack_uint((CONST uint32_t *) cl, 17, 17);
   return agx_genxml_validate_mask(fp, "Cull", cl, 0, 0xfffc727c);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_CULL_print(FILE *fp, const struct AGX_CULL * values, unsigned indent)
{
   fprintf(fp, "%*sCull front: %s\n", indent, "", values->cull_front ? "true" : "false");
   fprintf(fp, "%*sCull back: %s\n", indent, "", values->cull_back ? "true" : "false");
   if (agx_ppp_vertex_as_str(values->flat_shading_vertex))
     fprintf(fp, "%*sFlat shading vertex: %s\n", indent, "", agx_ppp_vertex_as_str(values->flat_shading_vertex));
   else
     fprintf(fp, "%*sFlat shading vertex: unknown %X (XXX)\n", indent, "", values->flat_shading_vertex);
   fprintf(fp, "%*sDepth clip: %s\n", indent, "", values->depth_clip ? "true" : "false");
   fprintf(fp, "%*sDepth clamp: %s\n", indent, "", values->depth_clamp ? "true" : "false");
   fprintf(fp, "%*sPrimitive MSAA: %s\n", indent, "", values->primitive_msaa ? "true" : "false");
   fprintf(fp, "%*sFront face CCW: %s\n", indent, "", values->front_face_ccw ? "true" : "false");
   fprintf(fp, "%*sRasterizer discard: %s\n", indent, "", values->rasterizer_discard ? "true" : "false");
}
#endif

struct AGX_CULL_2 {
   bool                                 clamp_w;
   bool                                 draw_clipped_edges;
   bool                                 needs_primitive_id;
   bool                                 rasterizer_discard;
};

#define AGX_CULL_2_header 0

static inline void
AGX_CULL_2_pack(GLOBAL uint32_t * restrict cl,
                const struct AGX_CULL_2 * restrict values)
{
   agx_genxml_validate_bounds("Cull 2::clamp_w", values->clamp_w, 0x2ull);
   agx_genxml_validate_bounds("Cull 2::draw_clipped_edges", values->draw_clipped_edges, 0x2ull);
   agx_genxml_validate_bounds("Cull 2::needs_primitive_id", values->needs_primitive_id, 0x2ull);
   agx_genxml_validate_bounds("Cull 2::rasterizer_discard", values->rasterizer_discard, 0x2ull);
   cl[ 0] = util_bitpack_uint(values->clamp_w, 5, 5) |
            util_bitpack_uint(values->draw_clipped_edges, 9, 9) |
            util_bitpack_uint(values->needs_primitive_id, 12, 12) |
            util_bitpack_uint(values->rasterizer_discard, 17, 17);
}

#define AGX_CULL_2_LENGTH 4
struct agx_cull_2_packed { uint32_t opaque[1];};
static inline bool
AGX_CULL_2_unpack(FILE *fp, CONST uint8_t * restrict cl,
                  struct AGX_CULL_2 * restrict values)
{
   values->clamp_w = __gen_unpack_uint((CONST uint32_t *) cl, 5, 5);
   values->draw_clipped_edges = __gen_unpack_uint((CONST uint32_t *) cl, 9, 9);
   values->needs_primitive_id = __gen_unpack_uint((CONST uint32_t *) cl, 12, 12);
   values->rasterizer_discard = __gen_unpack_uint((CONST uint32_t *) cl, 17, 17);
   return agx_genxml_validate_mask(fp, "Cull 2", cl, 0, 0xfffdeddf);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_CULL_2_print(FILE *fp, const struct AGX_CULL_2 * values, unsigned indent)
{
   fprintf(fp, "%*sClamp W: %s\n", indent, "", values->clamp_w ? "true" : "false");
   fprintf(fp, "%*sDraw clipped edges: %s\n", indent, "", values->draw_clipped_edges ? "true" : "false");
   fprintf(fp, "%*sNeeds Primitive ID: %s\n", indent, "", values->needs_primitive_id ? "true" : "false");
   fprintf(fp, "%*sRasterizer discard: %s\n", indent, "", values->rasterizer_discard ? "true" : "false");
}
#endif

struct AGX_VARYING_COUNTS {
   uint32_t                             smooth;
   uint32_t                             flat;
   uint32_t                             linear;
};

#define AGX_VARYING_COUNTS_header 0

static inline void
AGX_VARYING_COUNTS_pack(GLOBAL uint32_t * restrict cl,
                        const struct AGX_VARYING_COUNTS * restrict values)
{
   agx_genxml_validate_bounds("Varying Counts::smooth", values->smooth, 0x100ull);
   agx_genxml_validate_bounds("Varying Counts::flat", values->flat, 0x100ull);
   agx_genxml_validate_bounds("Varying Counts::linear", values->linear, 0x100ull);
   cl[ 0] = util_bitpack_uint(values->smooth, 0, 7) |
            util_bitpack_uint(values->flat, 8, 15) |
            util_bitpack_uint(values->linear, 16, 23);
}

#define AGX_VARYING_COUNTS_LENGTH 4
struct agx_varying_counts_packed { uint32_t opaque[1];};
static inline bool
AGX_VARYING_COUNTS_unpack(FILE *fp, CONST uint8_t * restrict cl,
                          struct AGX_VARYING_COUNTS * restrict values)
{
   values->smooth = __gen_unpack_uint((CONST uint32_t *) cl, 0, 7);
   values->flat = __gen_unpack_uint((CONST uint32_t *) cl, 8, 15);
   values->linear = __gen_unpack_uint((CONST uint32_t *) cl, 16, 23);
   return agx_genxml_validate_mask(fp, "Varying Counts", cl, 0, 0xff000000);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VARYING_COUNTS_print(FILE *fp, const struct AGX_VARYING_COUNTS * values, unsigned indent)
{
   fprintf(fp, "%*sSmooth: %u\n", indent, "", values->smooth);
   fprintf(fp, "%*sFlat: %u\n", indent, "", values->flat);
   fprintf(fp, "%*sLinear: %u\n", indent, "", values->linear);
}
#endif

struct AGX_VARYING_2 {
   int dummy;
};

#define AGX_VARYING_2_header 0

static inline void
AGX_VARYING_2_pack(GLOBAL uint32_t * restrict cl,
                   const struct AGX_VARYING_2 * restrict values)
{
   cl[ 0] = 0;
   cl[ 1] = 0;
}

#define AGX_VARYING_2_LENGTH 8
struct agx_varying_2_packed { uint32_t opaque[2];};
static inline bool
AGX_VARYING_2_unpack(FILE *fp, CONST uint8_t * restrict cl,
                     struct AGX_VARYING_2 * restrict values)
{
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "Varying 2", cl, 0, 0xffffffff);
   valid &= agx_genxml_validate_mask(fp, "Varying 2", cl, 1, 0xffffffff);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VARYING_2_print(FILE *fp, const struct AGX_VARYING_2 * values, unsigned indent)
{
}
#endif

struct AGX_OUTPUT_SELECT {
   bool                                 clip_distance_plane_0;
   bool                                 clip_distance_plane_1;
   bool                                 clip_distance_plane_2;
   bool                                 clip_distance_plane_3;
   bool                                 clip_distance_plane_4;
   bool                                 clip_distance_plane_5;
   bool                                 clip_distance_plane_6;
   bool                                 clip_distance_plane_7;
   bool                                 clip_distance_plane_8;
   bool                                 clip_distance_plane_9;
   bool                                 clip_distance_plane_10;
   bool                                 clip_distance_plane_11;
   bool                                 clip_distance_plane_12;
   bool                                 clip_distance_plane_13;
   bool                                 clip_distance_plane_14;
   bool                                 clip_distance_plane_15;
   bool                                 varyings;
   bool                                 point_size;
   bool                                 viewport_target;
   bool                                 render_target;
   bool                                 frag_coord_z;
   bool                                 barycentric_coordinates;
};

#define AGX_OUTPUT_SELECT_header 0

static inline void
AGX_OUTPUT_SELECT_pack(GLOBAL uint32_t * restrict cl,
                       const struct AGX_OUTPUT_SELECT * restrict values)
{
   agx_genxml_validate_bounds("Output Select::clip_distance_plane_0", values->clip_distance_plane_0, 0x2ull);
   agx_genxml_validate_bounds("Output Select::clip_distance_plane_1", values->clip_distance_plane_1, 0x2ull);
   agx_genxml_validate_bounds("Output Select::clip_distance_plane_2", values->clip_distance_plane_2, 0x2ull);
   agx_genxml_validate_bounds("Output Select::clip_distance_plane_3", values->clip_distance_plane_3, 0x2ull);
   agx_genxml_validate_bounds("Output Select::clip_distance_plane_4", values->clip_distance_plane_4, 0x2ull);
   agx_genxml_validate_bounds("Output Select::clip_distance_plane_5", values->clip_distance_plane_5, 0x2ull);
   agx_genxml_validate_bounds("Output Select::clip_distance_plane_6", values->clip_distance_plane_6, 0x2ull);
   agx_genxml_validate_bounds("Output Select::clip_distance_plane_7", values->clip_distance_plane_7, 0x2ull);
   agx_genxml_validate_bounds("Output Select::clip_distance_plane_8", values->clip_distance_plane_8, 0x2ull);
   agx_genxml_validate_bounds("Output Select::clip_distance_plane_9", values->clip_distance_plane_9, 0x2ull);
   agx_genxml_validate_bounds("Output Select::clip_distance_plane_10", values->clip_distance_plane_10, 0x2ull);
   agx_genxml_validate_bounds("Output Select::clip_distance_plane_11", values->clip_distance_plane_11, 0x2ull);
   agx_genxml_validate_bounds("Output Select::clip_distance_plane_12", values->clip_distance_plane_12, 0x2ull);
   agx_genxml_validate_bounds("Output Select::clip_distance_plane_13", values->clip_distance_plane_13, 0x2ull);
   agx_genxml_validate_bounds("Output Select::clip_distance_plane_14", values->clip_distance_plane_14, 0x2ull);
   agx_genxml_validate_bounds("Output Select::clip_distance_plane_15", values->clip_distance_plane_15, 0x2ull);
   agx_genxml_validate_bounds("Output Select::varyings", values->varyings, 0x2ull);
   agx_genxml_validate_bounds("Output Select::point_size", values->point_size, 0x2ull);
   agx_genxml_validate_bounds("Output Select::viewport_target", values->viewport_target, 0x2ull);
   agx_genxml_validate_bounds("Output Select::render_target", values->render_target, 0x2ull);
   agx_genxml_validate_bounds("Output Select::frag_coord_z", values->frag_coord_z, 0x2ull);
   agx_genxml_validate_bounds("Output Select::barycentric_coordinates", values->barycentric_coordinates, 0x2ull);
   cl[ 0] = util_bitpack_uint(values->clip_distance_plane_0, 0, 0) |
            util_bitpack_uint(values->clip_distance_plane_1, 1, 1) |
            util_bitpack_uint(values->clip_distance_plane_2, 2, 2) |
            util_bitpack_uint(values->clip_distance_plane_3, 3, 3) |
            util_bitpack_uint(values->clip_distance_plane_4, 4, 4) |
            util_bitpack_uint(values->clip_distance_plane_5, 5, 5) |
            util_bitpack_uint(values->clip_distance_plane_6, 6, 6) |
            util_bitpack_uint(values->clip_distance_plane_7, 7, 7) |
            util_bitpack_uint(values->clip_distance_plane_8, 8, 8) |
            util_bitpack_uint(values->clip_distance_plane_9, 9, 9) |
            util_bitpack_uint(values->clip_distance_plane_10, 10, 10) |
            util_bitpack_uint(values->clip_distance_plane_11, 11, 11) |
            util_bitpack_uint(values->clip_distance_plane_12, 12, 12) |
            util_bitpack_uint(values->clip_distance_plane_13, 13, 13) |
            util_bitpack_uint(values->clip_distance_plane_14, 14, 14) |
            util_bitpack_uint(values->clip_distance_plane_15, 15, 15) |
            util_bitpack_uint(values->varyings, 16, 16) |
            util_bitpack_uint(values->point_size, 18, 18) |
            util_bitpack_uint(values->viewport_target, 19, 19) |
            util_bitpack_uint(values->render_target, 20, 20) |
            util_bitpack_uint(values->frag_coord_z, 21, 21) |
            util_bitpack_uint(values->barycentric_coordinates, 22, 22);
}

#define AGX_OUTPUT_SELECT_LENGTH 4
struct agx_output_select_packed { uint32_t opaque[1];};
static inline bool
AGX_OUTPUT_SELECT_unpack(FILE *fp, CONST uint8_t * restrict cl,
                         struct AGX_OUTPUT_SELECT * restrict values)
{
   values->clip_distance_plane_0 = __gen_unpack_uint((CONST uint32_t *) cl, 0, 0);
   values->clip_distance_plane_1 = __gen_unpack_uint((CONST uint32_t *) cl, 1, 1);
   values->clip_distance_plane_2 = __gen_unpack_uint((CONST uint32_t *) cl, 2, 2);
   values->clip_distance_plane_3 = __gen_unpack_uint((CONST uint32_t *) cl, 3, 3);
   values->clip_distance_plane_4 = __gen_unpack_uint((CONST uint32_t *) cl, 4, 4);
   values->clip_distance_plane_5 = __gen_unpack_uint((CONST uint32_t *) cl, 5, 5);
   values->clip_distance_plane_6 = __gen_unpack_uint((CONST uint32_t *) cl, 6, 6);
   values->clip_distance_plane_7 = __gen_unpack_uint((CONST uint32_t *) cl, 7, 7);
   values->clip_distance_plane_8 = __gen_unpack_uint((CONST uint32_t *) cl, 8, 8);
   values->clip_distance_plane_9 = __gen_unpack_uint((CONST uint32_t *) cl, 9, 9);
   values->clip_distance_plane_10 = __gen_unpack_uint((CONST uint32_t *) cl, 10, 10);
   values->clip_distance_plane_11 = __gen_unpack_uint((CONST uint32_t *) cl, 11, 11);
   values->clip_distance_plane_12 = __gen_unpack_uint((CONST uint32_t *) cl, 12, 12);
   values->clip_distance_plane_13 = __gen_unpack_uint((CONST uint32_t *) cl, 13, 13);
   values->clip_distance_plane_14 = __gen_unpack_uint((CONST uint32_t *) cl, 14, 14);
   values->clip_distance_plane_15 = __gen_unpack_uint((CONST uint32_t *) cl, 15, 15);
   values->varyings = __gen_unpack_uint((CONST uint32_t *) cl, 16, 16);
   values->point_size = __gen_unpack_uint((CONST uint32_t *) cl, 18, 18);
   values->viewport_target = __gen_unpack_uint((CONST uint32_t *) cl, 19, 19);
   values->render_target = __gen_unpack_uint((CONST uint32_t *) cl, 20, 20);
   values->frag_coord_z = __gen_unpack_uint((CONST uint32_t *) cl, 21, 21);
   values->barycentric_coordinates = __gen_unpack_uint((CONST uint32_t *) cl, 22, 22);
   return agx_genxml_validate_mask(fp, "Output Select", cl, 0, 0xff820000);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_OUTPUT_SELECT_print(FILE *fp, const struct AGX_OUTPUT_SELECT * values, unsigned indent)
{
   fprintf(fp, "%*sClip distance plane 0: %s\n", indent, "", values->clip_distance_plane_0 ? "true" : "false");
   fprintf(fp, "%*sClip distance plane 1: %s\n", indent, "", values->clip_distance_plane_1 ? "true" : "false");
   fprintf(fp, "%*sClip distance plane 2: %s\n", indent, "", values->clip_distance_plane_2 ? "true" : "false");
   fprintf(fp, "%*sClip distance plane 3: %s\n", indent, "", values->clip_distance_plane_3 ? "true" : "false");
   fprintf(fp, "%*sClip distance plane 4: %s\n", indent, "", values->clip_distance_plane_4 ? "true" : "false");
   fprintf(fp, "%*sClip distance plane 5: %s\n", indent, "", values->clip_distance_plane_5 ? "true" : "false");
   fprintf(fp, "%*sClip distance plane 6: %s\n", indent, "", values->clip_distance_plane_6 ? "true" : "false");
   fprintf(fp, "%*sClip distance plane 7: %s\n", indent, "", values->clip_distance_plane_7 ? "true" : "false");
   fprintf(fp, "%*sClip distance plane 8: %s\n", indent, "", values->clip_distance_plane_8 ? "true" : "false");
   fprintf(fp, "%*sClip distance plane 9: %s\n", indent, "", values->clip_distance_plane_9 ? "true" : "false");
   fprintf(fp, "%*sClip distance plane 10: %s\n", indent, "", values->clip_distance_plane_10 ? "true" : "false");
   fprintf(fp, "%*sClip distance plane 11: %s\n", indent, "", values->clip_distance_plane_11 ? "true" : "false");
   fprintf(fp, "%*sClip distance plane 12: %s\n", indent, "", values->clip_distance_plane_12 ? "true" : "false");
   fprintf(fp, "%*sClip distance plane 13: %s\n", indent, "", values->clip_distance_plane_13 ? "true" : "false");
   fprintf(fp, "%*sClip distance plane 14: %s\n", indent, "", values->clip_distance_plane_14 ? "true" : "false");
   fprintf(fp, "%*sClip distance plane 15: %s\n", indent, "", values->clip_distance_plane_15 ? "true" : "false");
   fprintf(fp, "%*sVaryings: %s\n", indent, "", values->varyings ? "true" : "false");
   fprintf(fp, "%*sPoint size: %s\n", indent, "", values->point_size ? "true" : "false");
   fprintf(fp, "%*sViewport target: %s\n", indent, "", values->viewport_target ? "true" : "false");
   fprintf(fp, "%*sRender target: %s\n", indent, "", values->render_target ? "true" : "false");
   fprintf(fp, "%*sFrag coord Z: %s\n", indent, "", values->frag_coord_z ? "true" : "false");
   fprintf(fp, "%*sBarycentric coordinates: %s\n", indent, "", values->barycentric_coordinates ? "true" : "false");
}
#endif

struct AGX_OUTPUT_UNKNOWN {
   int dummy;
};

#define AGX_OUTPUT_UNKNOWN_header 0

static inline void
AGX_OUTPUT_UNKNOWN_pack(GLOBAL uint32_t * restrict cl,
                        const struct AGX_OUTPUT_UNKNOWN * restrict values)
{
   cl[ 0] = 0;
}

#define AGX_OUTPUT_UNKNOWN_LENGTH 4
struct agx_output_unknown_packed { uint32_t opaque[1];};
static inline bool
AGX_OUTPUT_UNKNOWN_unpack(FILE *fp, CONST uint8_t * restrict cl,
                          struct AGX_OUTPUT_UNKNOWN * restrict values)
{
   return agx_genxml_validate_mask(fp, "Output Unknown", cl, 0, 0xffffffff);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_OUTPUT_UNKNOWN_print(FILE *fp, const struct AGX_OUTPUT_UNKNOWN * values, unsigned indent)
{
}
#endif

struct AGX_OUTPUT_SIZE {
   uint32_t                             count;
};

#define AGX_OUTPUT_SIZE_header 0

static inline void
AGX_OUTPUT_SIZE_pack(GLOBAL uint32_t * restrict cl,
                     const struct AGX_OUTPUT_SIZE * restrict values)
{
   agx_genxml_validate_bounds("Output Size::count", values->count, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->count, 0, 31);
}

#define AGX_OUTPUT_SIZE_LENGTH 4
struct agx_output_size_packed { uint32_t opaque[1];};
static inline bool
AGX_OUTPUT_SIZE_unpack(FILE *fp, CONST uint8_t * restrict cl,
                       struct AGX_OUTPUT_SIZE * restrict values)
{
   values->count = __gen_unpack_uint((CONST uint32_t *) cl, 0, 31);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_OUTPUT_SIZE_print(FILE *fp, const struct AGX_OUTPUT_SIZE * values, unsigned indent)
{
   fprintf(fp, "%*sCount: %u\n", indent, "", values->count);
}
#endif

struct AGX_DEPTH_BIAS_SCISSOR {
   uint32_t                             scissor;
   uint32_t                             depth_bias;
};

#define AGX_DEPTH_BIAS_SCISSOR_header 0

static inline void
AGX_DEPTH_BIAS_SCISSOR_pack(GLOBAL uint32_t * restrict cl,
                            const struct AGX_DEPTH_BIAS_SCISSOR * restrict values)
{
   agx_genxml_validate_bounds("Depth bias/Scissor::scissor", values->scissor, 0x10000ull);
   agx_genxml_validate_bounds("Depth bias/Scissor::depth_bias", values->depth_bias, 0x10000ull);
   cl[ 0] = util_bitpack_uint(values->scissor, 0, 15) |
            util_bitpack_uint(values->depth_bias, 16, 31);
}

#define AGX_DEPTH_BIAS_SCISSOR_LENGTH 4
struct agx_depth_bias_scissor_packed { uint32_t opaque[1];};
static inline bool
AGX_DEPTH_BIAS_SCISSOR_unpack(FILE *fp, CONST uint8_t * restrict cl,
                              struct AGX_DEPTH_BIAS_SCISSOR * restrict values)
{
   values->scissor = __gen_unpack_uint((CONST uint32_t *) cl, 0, 15);
   values->depth_bias = __gen_unpack_uint((CONST uint32_t *) cl, 16, 31);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_DEPTH_BIAS_SCISSOR_print(FILE *fp, const struct AGX_DEPTH_BIAS_SCISSOR * values, unsigned indent)
{
   fprintf(fp, "%*sScissor: %u\n", indent, "", values->scissor);
   fprintf(fp, "%*sDepth bias: %u\n", indent, "", values->depth_bias);
}
#endif

struct AGX_FRAGMENT_SHADER_WORD_0 {
   uint32_t                             unknown_0;
   uint32_t                             uniform_register_count;
   uint32_t                             texture_state_register_count;
   enum agx_sampler_states              sampler_state_register_count;
   uint32_t                             preshader_register_count;
   uint32_t                             cf_binding_count;
};

#define AGX_FRAGMENT_SHADER_WORD_0_header       \
   .uniform_register_count = 512,  \
   .texture_state_register_count = 256,  \
   .preshader_register_count = 256

static inline void
AGX_FRAGMENT_SHADER_WORD_0_pack(GLOBAL uint32_t * restrict cl,
                                const struct AGX_FRAGMENT_SHADER_WORD_0 * restrict values)
{
   agx_genxml_validate_bounds("Fragment Shader Word 0::unknown_0", values->unknown_0, 0x2ull);
   agx_genxml_validate_bounds("Fragment Shader Word 0::uniform_register_count", __gen_to_groups(values->uniform_register_count, 64, 3), 0x8ull);
   agx_genxml_validate_bounds("Fragment Shader Word 0::texture_state_register_count", __gen_to_groups(values->texture_state_register_count, 8, 5), 0x20ull);
   agx_genxml_validate_bounds("Fragment Shader Word 0::sampler_state_register_count", values->sampler_state_register_count, 0x8ull);
   agx_genxml_validate_bounds("Fragment Shader Word 0::preshader_register_count", __gen_to_groups(values->preshader_register_count, 16, 4), 0x10ull);
   agx_genxml_validate_bounds("Fragment Shader Word 0::cf_binding_count", values->cf_binding_count, 0x80ull);
   cl[ 0] = util_bitpack_uint(values->unknown_0, 0, 0) |
            util_bitpack_uint(__gen_to_groups(values->uniform_register_count, 64, 3), 1, 3) |
            util_bitpack_uint(__gen_to_groups(values->texture_state_register_count, 8, 5), 4, 8) |
            util_bitpack_uint(values->sampler_state_register_count, 9, 11) |
            util_bitpack_uint(__gen_to_groups(values->preshader_register_count, 16, 4), 12, 15) |
            util_bitpack_uint(values->cf_binding_count, 16, 22);
}

#define AGX_FRAGMENT_SHADER_WORD_0_LENGTH 4
struct agx_fragment_shader_word_0_packed { uint32_t opaque[1];};
static inline bool
AGX_FRAGMENT_SHADER_WORD_0_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                  struct AGX_FRAGMENT_SHADER_WORD_0 * restrict values)
{
   values->unknown_0 = __gen_unpack_uint((CONST uint32_t *) cl, 0, 0);
   values->uniform_register_count = __gen_from_groups(__gen_unpack_uint((CONST uint32_t *) cl, 1, 3), 64, 3);
   values->texture_state_register_count = __gen_from_groups(__gen_unpack_uint((CONST uint32_t *) cl, 4, 8), 8, 5);
   values->sampler_state_register_count = (enum agx_sampler_states) __gen_unpack_uint((CONST uint32_t *) cl, 9, 11);
   values->preshader_register_count = __gen_from_groups(__gen_unpack_uint((CONST uint32_t *) cl, 12, 15), 16, 4);
   values->cf_binding_count = __gen_unpack_uint((CONST uint32_t *) cl, 16, 22);
   return agx_genxml_validate_mask(fp, "Fragment Shader Word 0", cl, 0, 0xff800000);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_FRAGMENT_SHADER_WORD_0_print(FILE *fp, const struct AGX_FRAGMENT_SHADER_WORD_0 * values, unsigned indent)
{
   fprintf(fp, "%*sUnknown 0: 0x%" PRIx32 "\n", indent, "", values->unknown_0);
   fprintf(fp, "%*sUniform register count: %u\n", indent, "", values->uniform_register_count);
   fprintf(fp, "%*sTexture state register count: %u\n", indent, "", values->texture_state_register_count);
   if (agx_sampler_states_as_str(values->sampler_state_register_count))
     fprintf(fp, "%*sSampler state register count: %s\n", indent, "", agx_sampler_states_as_str(values->sampler_state_register_count));
   else
     fprintf(fp, "%*sSampler state register count: unknown %X (XXX)\n", indent, "", values->sampler_state_register_count);
   fprintf(fp, "%*sPreshader register count: %u\n", indent, "", values->preshader_register_count);
   fprintf(fp, "%*sCF binding count: %u\n", indent, "", values->cf_binding_count);
}
#endif

struct AGX_FRAGMENT_SHADER_WORD_1 {
   uint32_t                             unknown_10;
   uint64_t                             pipeline;
};

#define AGX_FRAGMENT_SHADER_WORD_1_header 0

static inline void
AGX_FRAGMENT_SHADER_WORD_1_pack(GLOBAL uint32_t * restrict cl,
                                const struct AGX_FRAGMENT_SHADER_WORD_1 * restrict values)
{
   assert((values->pipeline & 0x3f) == 0);
   agx_genxml_validate_bounds("Fragment Shader Word 1::unknown_10", values->unknown_10, 0x4ull);
   agx_genxml_validate_bounds("Fragment Shader Word 1::pipeline", values->pipeline, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->unknown_10, 0, 1) |
            util_bitpack_uint(values->pipeline, 0, 31);
}

#define AGX_FRAGMENT_SHADER_WORD_1_LENGTH 4
struct agx_fragment_shader_word_1_packed { uint32_t opaque[1];};
static inline bool
AGX_FRAGMENT_SHADER_WORD_1_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                  struct AGX_FRAGMENT_SHADER_WORD_1 * restrict values)
{
   values->unknown_10 = __gen_unpack_uint((CONST uint32_t *) cl, 0, 1);
   values->pipeline = __gen_unpack_uint((CONST uint32_t *) cl, 6, 31) << 6;
   return agx_genxml_validate_mask(fp, "Fragment Shader Word 1", cl, 0, 0x3c);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_FRAGMENT_SHADER_WORD_1_print(FILE *fp, const struct AGX_FRAGMENT_SHADER_WORD_1 * values, unsigned indent)
{
   fprintf(fp, "%*sUnknown 1:0: 0x%" PRIx32 "\n", indent, "", values->unknown_10);
   fprintf(fp, "%*sPipeline: 0x%" PRIx64 "\n", indent, "", values->pipeline);
}
#endif

struct AGX_FRAGMENT_SHADER_WORD_2 {
   uint64_t                             cf_bindings;
};

#define AGX_FRAGMENT_SHADER_WORD_2_header 0

static inline void
AGX_FRAGMENT_SHADER_WORD_2_pack(GLOBAL uint32_t * restrict cl,
                                const struct AGX_FRAGMENT_SHADER_WORD_2 * restrict values)
{
   assert((values->cf_bindings & 0x3) == 0);
   agx_genxml_validate_bounds("Fragment Shader Word 2::cf_bindings", values->cf_bindings, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->cf_bindings, 0, 31);
}

#define AGX_FRAGMENT_SHADER_WORD_2_LENGTH 4
struct agx_fragment_shader_word_2_packed { uint32_t opaque[1];};
static inline bool
AGX_FRAGMENT_SHADER_WORD_2_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                  struct AGX_FRAGMENT_SHADER_WORD_2 * restrict values)
{
   values->cf_bindings = __gen_unpack_uint((CONST uint32_t *) cl, 2, 31) << 2;
   return agx_genxml_validate_mask(fp, "Fragment Shader Word 2", cl, 0, 0x3);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_FRAGMENT_SHADER_WORD_2_print(FILE *fp, const struct AGX_FRAGMENT_SHADER_WORD_2 * values, unsigned indent)
{
   fprintf(fp, "%*sCF bindings: 0x%" PRIx64 "\n", indent, "", values->cf_bindings);
}
#endif

struct AGX_FRAGMENT_SHADER_WORD_3 {
   uint32_t                             unknown;
};

#define AGX_FRAGMENT_SHADER_WORD_3_header 0

static inline void
AGX_FRAGMENT_SHADER_WORD_3_pack(GLOBAL uint32_t * restrict cl,
                                const struct AGX_FRAGMENT_SHADER_WORD_3 * restrict values)
{
   agx_genxml_validate_bounds("Fragment Shader Word 3::unknown", values->unknown, 0x10ull);
   cl[ 0] = util_bitpack_uint(values->unknown, 0, 3);
}

#define AGX_FRAGMENT_SHADER_WORD_3_LENGTH 4
struct agx_fragment_shader_word_3_packed { uint32_t opaque[1];};
static inline bool
AGX_FRAGMENT_SHADER_WORD_3_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                  struct AGX_FRAGMENT_SHADER_WORD_3 * restrict values)
{
   values->unknown = __gen_unpack_uint((CONST uint32_t *) cl, 0, 3);
   return agx_genxml_validate_mask(fp, "Fragment Shader Word 3", cl, 0, 0xfffffff0);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_FRAGMENT_SHADER_WORD_3_print(FILE *fp, const struct AGX_FRAGMENT_SHADER_WORD_3 * values, unsigned indent)
{
   fprintf(fp, "%*sUnknown: 0x%" PRIx32 "\n", indent, "", values->unknown);
}
#endif

enum agx_usc_control {
   AGX_USC_CONTROL_PRESHADER = 56,
   AGX_USC_CONTROL_FRAGMENT_PROPERTIES = 88,
   AGX_USC_CONTROL_NO_PRESHADER = 136,
   AGX_USC_CONTROL_SHADER = 13,
   AGX_USC_CONTROL_UNIFORM = 29,
   AGX_USC_CONTROL_UNIFORM_HIGH = 61,
   AGX_USC_CONTROL_SHARED = 77,
   AGX_USC_CONTROL_REGISTERS = 141,
   AGX_USC_CONTROL_SAMPLER = 157,
   AGX_USC_CONTROL_TEXTURE = 221,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_usc_control_as_str(enum agx_usc_control imm)
{
    switch (imm) {
    case AGX_USC_CONTROL_PRESHADER: return "Preshader";
    case AGX_USC_CONTROL_FRAGMENT_PROPERTIES: return "Fragment properties";
    case AGX_USC_CONTROL_NO_PRESHADER: return "No preshader";
    case AGX_USC_CONTROL_SHADER: return "Shader";
    case AGX_USC_CONTROL_UNIFORM: return "Uniform";
    case AGX_USC_CONTROL_UNIFORM_HIGH: return "Uniform high";
    case AGX_USC_CONTROL_SHARED: return "Shared";
    case AGX_USC_CONTROL_REGISTERS: return "Registers";
    case AGX_USC_CONTROL_SAMPLER: return "Sampler";
    case AGX_USC_CONTROL_TEXTURE: return "Texture";
    default: return NULL;
    }
}
#endif

struct AGX_USC_UNIFORM {
   uint32_t                             start_halfs;
   uint32_t                             size_halfs;
   uint64_t                             buffer;
};

#define AGX_USC_UNIFORM_header                  \
   .size_halfs = 64

static inline void
AGX_USC_UNIFORM_pack(GLOBAL uint32_t * restrict cl,
                     const struct AGX_USC_UNIFORM * restrict values)
{
   assert((values->buffer & 0x3) == 0);
   agx_genxml_validate_bounds("USC Uniform::start_halfs", values->start_halfs, 0x100ull);
   agx_genxml_validate_bounds("USC Uniform::size_halfs", __gen_to_groups(values->size_halfs, 1, 6), 0x40ull);
   agx_genxml_validate_bounds("USC Uniform::buffer", values->buffer, 0x10000000000ull);
   cl[ 0] = util_bitpack_uint(AGX_USC_CONTROL_UNIFORM, 0, 7) |
            util_bitpack_uint(values->start_halfs, 8, 15) |
            util_bitpack_uint(__gen_to_groups(values->size_halfs, 1, 6), 20, 25) |
            util_bitpack_uint(values->buffer, 24, 63);
   agx_genxml_validate_bounds("USC Uniform::buffer", values->buffer, 0x10000000000ull);
   cl[ 1] = util_bitpack_uint(values->buffer, 24, 63) >> 32;
}

#define AGX_USC_UNIFORM_LENGTH 8
struct agx_usc_uniform_packed { uint32_t opaque[2];};
static inline bool
AGX_USC_UNIFORM_unpack(FILE *fp, CONST uint8_t * restrict cl,
                       struct AGX_USC_UNIFORM * restrict values)
{
   values->start_halfs = __gen_unpack_uint((CONST uint32_t *) cl, 8, 15);
   values->size_halfs = __gen_from_groups(__gen_unpack_uint((CONST uint32_t *) cl, 20, 25), 1, 6);
   values->buffer = __gen_unpack_uint((CONST uint32_t *) cl, 26, 63) << 2;
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "USC Uniform", cl, 0, 0xf0000);
   valid &= agx_genxml_validate_exact(fp, "USC Uniform", __gen_unpack_uint((CONST uint32_t *) cl, 0, 7), AGX_USC_CONTROL_UNIFORM);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_USC_UNIFORM_print(FILE *fp, const struct AGX_USC_UNIFORM * values, unsigned indent)
{
   fprintf(fp, "%*sStart (halfs): %u\n", indent, "", values->start_halfs);
   fprintf(fp, "%*sSize (halfs): %u\n", indent, "", values->size_halfs);
   fprintf(fp, "%*sBuffer: 0x%" PRIx64 "\n", indent, "", values->buffer);
}
#endif

struct AGX_USC_UNIFORM_HIGH {
   uint32_t                             start_halfs;
   uint32_t                             size_halfs;
   uint64_t                             buffer;
};

#define AGX_USC_UNIFORM_HIGH_header             \
   .size_halfs = 64

static inline void
AGX_USC_UNIFORM_HIGH_pack(GLOBAL uint32_t * restrict cl,
                          const struct AGX_USC_UNIFORM_HIGH * restrict values)
{
   assert((values->buffer & 0x3) == 0);
   agx_genxml_validate_bounds("USC Uniform High::start_halfs", values->start_halfs, 0x100ull);
   agx_genxml_validate_bounds("USC Uniform High::size_halfs", __gen_to_groups(values->size_halfs, 1, 6), 0x40ull);
   agx_genxml_validate_bounds("USC Uniform High::buffer", values->buffer, 0x10000000000ull);
   cl[ 0] = util_bitpack_uint(AGX_USC_CONTROL_UNIFORM_HIGH, 0, 7) |
            util_bitpack_uint(values->start_halfs, 8, 15) |
            util_bitpack_uint(__gen_to_groups(values->size_halfs, 1, 6), 20, 25) |
            util_bitpack_uint(values->buffer, 24, 63);
   agx_genxml_validate_bounds("USC Uniform High::buffer", values->buffer, 0x10000000000ull);
   cl[ 1] = util_bitpack_uint(values->buffer, 24, 63) >> 32;
}

#define AGX_USC_UNIFORM_HIGH_LENGTH 8
struct agx_usc_uniform_high_packed { uint32_t opaque[2];};
static inline bool
AGX_USC_UNIFORM_HIGH_unpack(FILE *fp, CONST uint8_t * restrict cl,
                            struct AGX_USC_UNIFORM_HIGH * restrict values)
{
   values->start_halfs = __gen_unpack_uint((CONST uint32_t *) cl, 8, 15);
   values->size_halfs = __gen_from_groups(__gen_unpack_uint((CONST uint32_t *) cl, 20, 25), 1, 6);
   values->buffer = __gen_unpack_uint((CONST uint32_t *) cl, 26, 63) << 2;
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "USC Uniform High", cl, 0, 0xf0000);
   valid &= agx_genxml_validate_exact(fp, "USC Uniform High", __gen_unpack_uint((CONST uint32_t *) cl, 0, 7), AGX_USC_CONTROL_UNIFORM_HIGH);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_USC_UNIFORM_HIGH_print(FILE *fp, const struct AGX_USC_UNIFORM_HIGH * values, unsigned indent)
{
   fprintf(fp, "%*sStart (halfs): %u\n", indent, "", values->start_halfs);
   fprintf(fp, "%*sSize (halfs): %u\n", indent, "", values->size_halfs);
   fprintf(fp, "%*sBuffer: 0x%" PRIx64 "\n", indent, "", values->buffer);
}
#endif

struct AGX_USC_TEXTURE {
   uint32_t                             start;
   uint32_t                             count;
   uint64_t                             buffer;
};

#define AGX_USC_TEXTURE_header 0

static inline void
AGX_USC_TEXTURE_pack(GLOBAL uint32_t * restrict cl,
                     const struct AGX_USC_TEXTURE * restrict values)
{
   assert((values->buffer & 0x7) == 0);
   agx_genxml_validate_bounds("USC Texture::start", values->start, 0x100ull);
   agx_genxml_validate_bounds("USC Texture::count", values->count, 0x80ull);
   agx_genxml_validate_bounds("USC Texture::buffer", values->buffer, 0x8000000000ull);
   cl[ 0] = util_bitpack_uint(AGX_USC_CONTROL_TEXTURE, 0, 7) |
            util_bitpack_uint(values->start, 8, 15) |
            util_bitpack_uint(values->count, 20, 26) |
            util_bitpack_uint(values->buffer, 24, 62);
   agx_genxml_validate_bounds("USC Texture::buffer", values->buffer, 0x8000000000ull);
   cl[ 1] = util_bitpack_uint(values->buffer, 24, 62) >> 32;
}

#define AGX_USC_TEXTURE_LENGTH 8
struct agx_usc_texture_packed { uint32_t opaque[2];};
static inline bool
AGX_USC_TEXTURE_unpack(FILE *fp, CONST uint8_t * restrict cl,
                       struct AGX_USC_TEXTURE * restrict values)
{
   values->start = __gen_unpack_uint((CONST uint32_t *) cl, 8, 15);
   values->count = __gen_unpack_uint((CONST uint32_t *) cl, 20, 26);
   values->buffer = __gen_unpack_uint((CONST uint32_t *) cl, 27, 62) << 3;
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "USC Texture", cl, 0, 0xf0000);
   valid &= agx_genxml_validate_mask(fp, "USC Texture", cl, 1, 0x80000000);
   valid &= agx_genxml_validate_exact(fp, "USC Texture", __gen_unpack_uint((CONST uint32_t *) cl, 0, 7), AGX_USC_CONTROL_TEXTURE);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_USC_TEXTURE_print(FILE *fp, const struct AGX_USC_TEXTURE * values, unsigned indent)
{
   fprintf(fp, "%*sStart: %u\n", indent, "", values->start);
   fprintf(fp, "%*sCount: %u\n", indent, "", values->count);
   fprintf(fp, "%*sBuffer: 0x%" PRIx64 "\n", indent, "", values->buffer);
}
#endif

struct AGX_USC_SAMPLER {
   uint32_t                             start;
   uint32_t                             count;
   uint64_t                             buffer;
};

#define AGX_USC_SAMPLER_header 0

static inline void
AGX_USC_SAMPLER_pack(GLOBAL uint32_t * restrict cl,
                     const struct AGX_USC_SAMPLER * restrict values)
{
   assert((values->buffer & 0x7) == 0);
   agx_genxml_validate_bounds("USC Sampler::start", values->start, 0x100ull);
   agx_genxml_validate_bounds("USC Sampler::count", values->count, 0x80ull);
   agx_genxml_validate_bounds("USC Sampler::buffer", values->buffer, 0x8000000000ull);
   cl[ 0] = util_bitpack_uint(AGX_USC_CONTROL_SAMPLER, 0, 7) |
            util_bitpack_uint(values->start, 8, 15) |
            util_bitpack_uint(values->count, 20, 26) |
            util_bitpack_uint(values->buffer, 24, 62);
   agx_genxml_validate_bounds("USC Sampler::buffer", values->buffer, 0x8000000000ull);
   cl[ 1] = util_bitpack_uint(values->buffer, 24, 62) >> 32;
}

#define AGX_USC_SAMPLER_LENGTH 8
struct agx_usc_sampler_packed { uint32_t opaque[2];};
static inline bool
AGX_USC_SAMPLER_unpack(FILE *fp, CONST uint8_t * restrict cl,
                       struct AGX_USC_SAMPLER * restrict values)
{
   values->start = __gen_unpack_uint((CONST uint32_t *) cl, 8, 15);
   values->count = __gen_unpack_uint((CONST uint32_t *) cl, 20, 26);
   values->buffer = __gen_unpack_uint((CONST uint32_t *) cl, 27, 62) << 3;
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "USC Sampler", cl, 0, 0xf0000);
   valid &= agx_genxml_validate_mask(fp, "USC Sampler", cl, 1, 0x80000000);
   valid &= agx_genxml_validate_exact(fp, "USC Sampler", __gen_unpack_uint((CONST uint32_t *) cl, 0, 7), AGX_USC_CONTROL_SAMPLER);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_USC_SAMPLER_print(FILE *fp, const struct AGX_USC_SAMPLER * values, unsigned indent)
{
   fprintf(fp, "%*sStart: %u\n", indent, "", values->start);
   fprintf(fp, "%*sCount: %u\n", indent, "", values->count);
   fprintf(fp, "%*sBuffer: 0x%" PRIx64 "\n", indent, "", values->buffer);
}
#endif

enum agx_shared_layout {
   AGX_SHARED_LAYOUT_VERTEX_COMPUTE = 36,
   AGX_SHARED_LAYOUT_32X32 = 47,
   AGX_SHARED_LAYOUT_32X16 = 63,
   AGX_SHARED_LAYOUT_16X16 = 54,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_shared_layout_as_str(enum agx_shared_layout imm)
{
    switch (imm) {
    case AGX_SHARED_LAYOUT_VERTEX_COMPUTE: return "Vertex/compute";
    case AGX_SHARED_LAYOUT_32X32: return "32x32";
    case AGX_SHARED_LAYOUT_32X16: return "32x16";
    case AGX_SHARED_LAYOUT_16X16: return "16x16";
    default: return NULL;
    }
}
#endif

struct AGX_USC_SHARED {
   bool                                 uses_shared_memory;
   enum agx_shared_layout               layout;
   uint32_t                             sample_count;
   uint32_t                             sample_stride_in_8_bytes;
   uint32_t                             bytes_per_threadgroup;
};

#define AGX_USC_SHARED_header                   \
   .sample_count = 1,  \
   .bytes_per_threadgroup = 65536

static inline void
AGX_USC_SHARED_pack(GLOBAL uint32_t * restrict cl,
                    const struct AGX_USC_SHARED * restrict values)
{
   assert(IS_POT_NONZERO(values->sample_count));
   agx_genxml_validate_bounds("USC Shared::uses_shared_memory", values->uses_shared_memory, 0x2ull);
   agx_genxml_validate_bounds("USC Shared::layout", values->layout, 0x40ull);
   agx_genxml_validate_bounds("USC Shared::sample_count", util_logbase2(values->sample_count), 0x4ull);
   agx_genxml_validate_bounds("USC Shared::sample_stride_in_8_bytes", values->sample_stride_in_8_bytes, 0x10ull);
   agx_genxml_validate_bounds("USC Shared::bytes_per_threadgroup", __gen_to_groups(values->bytes_per_threadgroup, 256, 8), 0x100ull);
   cl[ 0] = util_bitpack_uint(AGX_USC_CONTROL_SHARED, 0, 7) |
            util_bitpack_uint(values->uses_shared_memory, 8, 8) |
            util_bitpack_uint(values->layout, 10, 15) |
            util_bitpack_uint(util_logbase2(values->sample_count), 16, 17) |
            util_bitpack_uint(values->sample_stride_in_8_bytes, 20, 23) |
            util_bitpack_uint(__gen_to_groups(values->bytes_per_threadgroup, 256, 8), 24, 31);
}

#define AGX_USC_SHARED_LENGTH 4
struct agx_usc_shared_packed { uint32_t opaque[1];};
static inline bool
AGX_USC_SHARED_unpack(FILE *fp, CONST uint8_t * restrict cl,
                      struct AGX_USC_SHARED * restrict values)
{
   values->uses_shared_memory = __gen_unpack_uint((CONST uint32_t *) cl, 8, 8);
   values->layout = (enum agx_shared_layout) __gen_unpack_uint((CONST uint32_t *) cl, 10, 15);
   values->sample_count = 1 << __gen_unpack_uint((CONST uint32_t *) cl, 16, 17);
   values->sample_stride_in_8_bytes = __gen_unpack_uint((CONST uint32_t *) cl, 20, 23);
   values->bytes_per_threadgroup = __gen_from_groups(__gen_unpack_uint((CONST uint32_t *) cl, 24, 31), 256, 8);
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "USC Shared", cl, 0, 0xc0200);
   valid &= agx_genxml_validate_exact(fp, "USC Shared", __gen_unpack_uint((CONST uint32_t *) cl, 0, 7), AGX_USC_CONTROL_SHARED);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_USC_SHARED_print(FILE *fp, const struct AGX_USC_SHARED * values, unsigned indent)
{
   fprintf(fp, "%*sUses shared memory: %s\n", indent, "", values->uses_shared_memory ? "true" : "false");
   if (agx_shared_layout_as_str(values->layout))
     fprintf(fp, "%*sLayout: %s\n", indent, "", agx_shared_layout_as_str(values->layout));
   else
     fprintf(fp, "%*sLayout: unknown %X (XXX)\n", indent, "", values->layout);
   fprintf(fp, "%*sSample count: %u\n", indent, "", values->sample_count);
   fprintf(fp, "%*sSample stride in 8 bytes: %u\n", indent, "", values->sample_stride_in_8_bytes);
   fprintf(fp, "%*sBytes per threadgroup: %u\n", indent, "", values->bytes_per_threadgroup);
}
#endif

struct AGX_USC_SHADER {
   bool                                 loads_varyings;
   bool                                 unk_1;
   uint32_t                             unk_2;
   uint64_t                             code;
};

#define AGX_USC_SHADER_header 0

static inline void
AGX_USC_SHADER_pack(GLOBAL uint32_t * restrict cl,
                    const struct AGX_USC_SHADER * restrict values)
{
   agx_genxml_validate_bounds("USC Shader::loads_varyings", values->loads_varyings, 0x2ull);
   agx_genxml_validate_bounds("USC Shader::unk_1", values->unk_1, 0x2ull);
   agx_genxml_validate_bounds("USC Shader::unk_2", values->unk_2, 0x40ull);
   agx_genxml_validate_bounds("USC Shader::code", values->code, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(AGX_USC_CONTROL_SHADER, 0, 7) |
            util_bitpack_uint(values->loads_varyings, 8, 8) |
            util_bitpack_uint(values->unk_1, 9, 9) |
            util_bitpack_uint(values->unk_2, 10, 15) |
            util_bitpack_uint(values->code, 16, 47);
   agx_genxml_validate_bounds("USC Shader::code", values->code, 0x100000000ull);
   cl[ 1] = util_bitpack_uint(values->code, 16, 47) >> 32;
}

#define AGX_USC_SHADER_LENGTH 6
struct agx_usc_shader_packed { uint32_t opaque[2];};
static inline bool
AGX_USC_SHADER_unpack(FILE *fp, CONST uint8_t * restrict cl,
                      struct AGX_USC_SHADER * restrict values)
{
   values->loads_varyings = __gen_unpack_uint((CONST uint32_t *) cl, 8, 8);
   values->unk_1 = __gen_unpack_uint((CONST uint32_t *) cl, 9, 9);
   values->unk_2 = __gen_unpack_uint((CONST uint32_t *) cl, 10, 15);
   values->code = __gen_unpack_uint((CONST uint32_t *) cl, 16, 47);
   return agx_genxml_validate_exact(fp, "USC Shader", __gen_unpack_uint((CONST uint32_t *) cl, 0, 7), AGX_USC_CONTROL_SHADER);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_USC_SHADER_print(FILE *fp, const struct AGX_USC_SHADER * values, unsigned indent)
{
   fprintf(fp, "%*sLoads varyings: %s\n", indent, "", values->loads_varyings ? "true" : "false");
   fprintf(fp, "%*sUnk 1: %s\n", indent, "", values->unk_1 ? "true" : "false");
   fprintf(fp, "%*sUnk 2: %u\n", indent, "", values->unk_2);
   fprintf(fp, "%*sCode: 0x%" PRIx64 "\n", indent, "", values->code);
}
#endif

struct AGX_USC_REGISTERS {
   uint32_t                             register_count;
   bool                                 unk_1;
   uint32_t                             spill_size;
   uint32_t                             unk_4;
};

#define AGX_USC_REGISTERS_header                \
   .register_count = 256

static inline void
AGX_USC_REGISTERS_pack(GLOBAL uint32_t * restrict cl,
                       const struct AGX_USC_REGISTERS * restrict values)
{
   agx_genxml_validate_bounds("USC Registers::register_count", __gen_to_groups(values->register_count, 8, 5), 0x20ull);
   agx_genxml_validate_bounds("USC Registers::unk_1", values->unk_1, 0x2ull);
   agx_genxml_validate_bounds("USC Registers::spill_size", values->spill_size, 0x10ull);
   agx_genxml_validate_bounds("USC Registers::unk_4", values->unk_4, 0x100ull);
   cl[ 0] = util_bitpack_uint(AGX_USC_CONTROL_REGISTERS, 0, 7) |
            util_bitpack_uint(__gen_to_groups(values->register_count, 8, 5), 8, 12) |
            util_bitpack_uint(values->unk_1, 13, 13) |
            util_bitpack_uint(values->spill_size, 18, 21) |
            util_bitpack_uint(values->unk_4, 24, 31);
}

#define AGX_USC_REGISTERS_LENGTH 4
struct agx_usc_registers_packed { uint32_t opaque[1];};
static inline bool
AGX_USC_REGISTERS_unpack(FILE *fp, CONST uint8_t * restrict cl,
                         struct AGX_USC_REGISTERS * restrict values)
{
   values->register_count = __gen_from_groups(__gen_unpack_uint((CONST uint32_t *) cl, 8, 12), 8, 5);
   values->unk_1 = __gen_unpack_uint((CONST uint32_t *) cl, 13, 13);
   values->spill_size = __gen_unpack_uint((CONST uint32_t *) cl, 18, 21);
   values->unk_4 = __gen_unpack_uint((CONST uint32_t *) cl, 24, 31);
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "USC Registers", cl, 0, 0xc3c000);
   valid &= agx_genxml_validate_exact(fp, "USC Registers", __gen_unpack_uint((CONST uint32_t *) cl, 0, 7), AGX_USC_CONTROL_REGISTERS);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_USC_REGISTERS_print(FILE *fp, const struct AGX_USC_REGISTERS * values, unsigned indent)
{
   fprintf(fp, "%*sRegister count: %u\n", indent, "", values->register_count);
   fprintf(fp, "%*sUnk 1: %s\n", indent, "", values->unk_1 ? "true" : "false");
   fprintf(fp, "%*sSpill size: 0x%" PRIx32 "\n", indent, "", values->spill_size);
   fprintf(fp, "%*sUnk 4: 0x%" PRIx32 "\n", indent, "", values->unk_4);
}
#endif

struct AGX_USC_NO_PRESHADER {
   int dummy;
};

#define AGX_USC_NO_PRESHADER_header 0

static inline void
AGX_USC_NO_PRESHADER_pack(GLOBAL uint32_t * restrict cl,
                          const struct AGX_USC_NO_PRESHADER * restrict values)
{
   cl[ 0] = util_bitpack_uint(AGX_USC_CONTROL_NO_PRESHADER, 0, 7);
}

#define AGX_USC_NO_PRESHADER_LENGTH 2
struct agx_usc_no_preshader_packed { uint32_t opaque[1];};
static inline bool
AGX_USC_NO_PRESHADER_unpack(FILE *fp, CONST uint8_t * restrict cl,
                            struct AGX_USC_NO_PRESHADER * restrict values)
{
   return agx_genxml_validate_exact(fp, "USC No Preshader", __gen_unpack_uint((CONST uint32_t *) cl, 0, 7), AGX_USC_CONTROL_NO_PRESHADER);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_USC_NO_PRESHADER_print(FILE *fp, const struct AGX_USC_NO_PRESHADER * values, unsigned indent)
{
}
#endif

struct AGX_USC_PRESHADER {
   uint64_t                             code;
};

#define AGX_USC_PRESHADER_header 0

static inline void
AGX_USC_PRESHADER_pack(GLOBAL uint32_t * restrict cl,
                       const struct AGX_USC_PRESHADER * restrict values)
{
   cl[ 0] = util_bitpack_uint(AGX_USC_CONTROL_PRESHADER, 0, 7) |
            util_bitpack_uint(0xc08000, 8, 31);
   agx_genxml_validate_bounds("USC Preshader::code", values->code, 0x100000000ull);
   cl[ 1] = util_bitpack_uint(values->code, 0, 31);
}

#define AGX_USC_PRESHADER_LENGTH 8
struct agx_usc_preshader_packed { uint32_t opaque[2];};
static inline bool
AGX_USC_PRESHADER_unpack(FILE *fp, CONST uint8_t * restrict cl,
                         struct AGX_USC_PRESHADER * restrict values)
{
   values->code = __gen_unpack_uint((CONST uint32_t *) cl, 32, 63);
   bool valid = true;
   valid &= agx_genxml_validate_exact(fp, "USC Preshader", __gen_unpack_uint((CONST uint32_t *) cl, 0, 7), AGX_USC_CONTROL_PRESHADER);
   valid &= agx_genxml_validate_exact(fp, "USC Preshader", __gen_unpack_uint((CONST uint32_t *) cl, 8, 31), 0xc08000);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_USC_PRESHADER_print(FILE *fp, const struct AGX_USC_PRESHADER * values, unsigned indent)
{
   fprintf(fp, "%*sCode: 0x%" PRIx64 "\n", indent, "", values->code);
}
#endif

struct AGX_USC_FRAGMENT_PROPERTIES {
   bool                                 early_z_testing;
   bool                                 unk_2;
   bool                                 unconditional_discard_1;
   bool                                 unconditional_discard_2;
   uint32_t                             unk_3;
   uint32_t                             unk_4;
   uint32_t                             unk_5;
};

#define AGX_USC_FRAGMENT_PROPERTIES_header 0

static inline void
AGX_USC_FRAGMENT_PROPERTIES_pack(GLOBAL uint32_t * restrict cl,
                                 const struct AGX_USC_FRAGMENT_PROPERTIES * restrict values)
{
   agx_genxml_validate_bounds("USC Fragment Properties::early_z_testing", values->early_z_testing, 0x2ull);
   agx_genxml_validate_bounds("USC Fragment Properties::unk_2", values->unk_2, 0x2ull);
   agx_genxml_validate_bounds("USC Fragment Properties::unconditional_discard_1", values->unconditional_discard_1, 0x2ull);
   agx_genxml_validate_bounds("USC Fragment Properties::unconditional_discard_2", values->unconditional_discard_2, 0x2ull);
   agx_genxml_validate_bounds("USC Fragment Properties::unk_3", values->unk_3, 0x10ull);
   agx_genxml_validate_bounds("USC Fragment Properties::unk_4", values->unk_4, 0x100ull);
   agx_genxml_validate_bounds("USC Fragment Properties::unk_5", values->unk_5, 0x100ull);
   cl[ 0] = util_bitpack_uint(AGX_USC_CONTROL_FRAGMENT_PROPERTIES, 0, 7) |
            util_bitpack_uint(values->early_z_testing, 8, 8) |
            util_bitpack_uint(values->unk_2, 9, 9) |
            util_bitpack_uint(values->unconditional_discard_1, 10, 10) |
            util_bitpack_uint(values->unconditional_discard_2, 11, 11) |
            util_bitpack_uint(values->unk_3, 12, 15) |
            util_bitpack_uint(values->unk_4, 16, 23) |
            util_bitpack_uint(values->unk_5, 24, 31);
}

#define AGX_USC_FRAGMENT_PROPERTIES_LENGTH 4
struct agx_usc_fragment_properties_packed { uint32_t opaque[1];};
static inline bool
AGX_USC_FRAGMENT_PROPERTIES_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                   struct AGX_USC_FRAGMENT_PROPERTIES * restrict values)
{
   values->early_z_testing = __gen_unpack_uint((CONST uint32_t *) cl, 8, 8);
   values->unk_2 = __gen_unpack_uint((CONST uint32_t *) cl, 9, 9);
   values->unconditional_discard_1 = __gen_unpack_uint((CONST uint32_t *) cl, 10, 10);
   values->unconditional_discard_2 = __gen_unpack_uint((CONST uint32_t *) cl, 11, 11);
   values->unk_3 = __gen_unpack_uint((CONST uint32_t *) cl, 12, 15);
   values->unk_4 = __gen_unpack_uint((CONST uint32_t *) cl, 16, 23);
   values->unk_5 = __gen_unpack_uint((CONST uint32_t *) cl, 24, 31);
   return agx_genxml_validate_exact(fp, "USC Fragment Properties", __gen_unpack_uint((CONST uint32_t *) cl, 0, 7), AGX_USC_CONTROL_FRAGMENT_PROPERTIES);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_USC_FRAGMENT_PROPERTIES_print(FILE *fp, const struct AGX_USC_FRAGMENT_PROPERTIES * values, unsigned indent)
{
   fprintf(fp, "%*sEarly-z testing: %s\n", indent, "", values->early_z_testing ? "true" : "false");
   fprintf(fp, "%*sUnk 2: %s\n", indent, "", values->unk_2 ? "true" : "false");
   fprintf(fp, "%*sUnconditional discard 1: %s\n", indent, "", values->unconditional_discard_1 ? "true" : "false");
   fprintf(fp, "%*sUnconditional discard 2: %s\n", indent, "", values->unconditional_discard_2 ? "true" : "false");
   fprintf(fp, "%*sUnk 3: 0x%" PRIx32 "\n", indent, "", values->unk_3);
   fprintf(fp, "%*sUnk 4: 0x%" PRIx32 "\n", indent, "", values->unk_4);
   fprintf(fp, "%*sUnk 5: 0x%" PRIx32 "\n", indent, "", values->unk_5);
}
#endif

enum agx_vdm_block_type {
   AGX_VDM_BLOCK_TYPE_PPP_STATE_UPDATE = 0,
   AGX_VDM_BLOCK_TYPE_BARRIER = 1,
   AGX_VDM_BLOCK_TYPE_VDM_STATE_UPDATE = 2,
   AGX_VDM_BLOCK_TYPE_INDEX_LIST = 3,
   AGX_VDM_BLOCK_TYPE_STREAM_LINK = 4,
   AGX_VDM_BLOCK_TYPE_TESSELLATE = 5,
   AGX_VDM_BLOCK_TYPE_STREAM_TERMINATE = 6,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_vdm_block_type_as_str(enum agx_vdm_block_type imm)
{
    switch (imm) {
    case AGX_VDM_BLOCK_TYPE_PPP_STATE_UPDATE: return "PPP State Update";
    case AGX_VDM_BLOCK_TYPE_BARRIER: return "Barrier";
    case AGX_VDM_BLOCK_TYPE_VDM_STATE_UPDATE: return "VDM State Update";
    case AGX_VDM_BLOCK_TYPE_INDEX_LIST: return "Index List";
    case AGX_VDM_BLOCK_TYPE_STREAM_LINK: return "Stream Link";
    case AGX_VDM_BLOCK_TYPE_TESSELLATE: return "Tessellate";
    case AGX_VDM_BLOCK_TYPE_STREAM_TERMINATE: return "Stream terminate";
    default: return NULL;
    }
}
#endif

struct AGX_PPP_STATE {
   uint32_t                             pointer_hi;
   uint32_t                             size_words;
   uint64_t                             pointer_lo;
};

#define AGX_PPP_STATE_header 0

static inline void
AGX_PPP_STATE_pack(GLOBAL uint32_t * restrict cl,
                   const struct AGX_PPP_STATE * restrict values)
{
   agx_genxml_validate_bounds("PPP State::pointer_hi", values->pointer_hi, 0x100ull);
   agx_genxml_validate_bounds("PPP State::size_words", values->size_words, 0x100ull);
   cl[ 0] = util_bitpack_uint(values->pointer_hi, 0, 7) |
            util_bitpack_uint(values->size_words, 8, 15) |
            util_bitpack_uint(AGX_VDM_BLOCK_TYPE_PPP_STATE_UPDATE, 29, 31);
   agx_genxml_validate_bounds("PPP State::pointer_lo", values->pointer_lo, 0x100000000ull);
   cl[ 1] = util_bitpack_uint(values->pointer_lo, 0, 31);
}

#define AGX_PPP_STATE_LENGTH 8
struct agx_ppp_state_packed { uint32_t opaque[2];};
static inline bool
AGX_PPP_STATE_unpack(FILE *fp, CONST uint8_t * restrict cl,
                     struct AGX_PPP_STATE * restrict values)
{
   values->pointer_hi = __gen_unpack_uint((CONST uint32_t *) cl, 0, 7);
   values->size_words = __gen_unpack_uint((CONST uint32_t *) cl, 8, 15);
   values->pointer_lo = __gen_unpack_uint((CONST uint32_t *) cl, 32, 63);
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "PPP State", cl, 0, 0x1fff0000);
   valid &= agx_genxml_validate_exact(fp, "PPP State", __gen_unpack_uint((CONST uint32_t *) cl, 29, 31), AGX_VDM_BLOCK_TYPE_PPP_STATE_UPDATE);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_PPP_STATE_print(FILE *fp, const struct AGX_PPP_STATE * values, unsigned indent)
{
   fprintf(fp, "%*sPointer (hi): 0x%" PRIx32 "\n", indent, "", values->pointer_hi);
   fprintf(fp, "%*sSize (words): %u\n", indent, "", values->size_words);
   fprintf(fp, "%*sPointer (lo): 0x%" PRIx64 "\n", indent, "", values->pointer_lo);
}
#endif

struct AGX_VDM_BARRIER {
   bool                                 usc_cache_inval;
   bool                                 unk_4;
   bool                                 unk_5;
   bool                                 unk_6;
   bool                                 unk_8;
   bool                                 unk_11;
   bool                                 unk_20;
   bool                                 unk_24;
   bool                                 unk_26;
   bool                                 returns;
};

#define AGX_VDM_BARRIER_header 0

static inline void
AGX_VDM_BARRIER_pack(GLOBAL uint32_t * restrict cl,
                     const struct AGX_VDM_BARRIER * restrict values)
{
   agx_genxml_validate_bounds("VDM Barrier::usc_cache_inval", values->usc_cache_inval, 0x2ull);
   agx_genxml_validate_bounds("VDM Barrier::unk_4", values->unk_4, 0x2ull);
   agx_genxml_validate_bounds("VDM Barrier::unk_5", values->unk_5, 0x2ull);
   agx_genxml_validate_bounds("VDM Barrier::unk_6", values->unk_6, 0x2ull);
   agx_genxml_validate_bounds("VDM Barrier::unk_8", values->unk_8, 0x2ull);
   agx_genxml_validate_bounds("VDM Barrier::unk_11", values->unk_11, 0x2ull);
   agx_genxml_validate_bounds("VDM Barrier::unk_20", values->unk_20, 0x2ull);
   agx_genxml_validate_bounds("VDM Barrier::unk_24", values->unk_24, 0x2ull);
   agx_genxml_validate_bounds("VDM Barrier::unk_26", values->unk_26, 0x2ull);
   agx_genxml_validate_bounds("VDM Barrier::returns", values->returns, 0x2ull);
   cl[ 0] = util_bitpack_uint(values->usc_cache_inval, 3, 3) |
            util_bitpack_uint(values->unk_4, 4, 4) |
            util_bitpack_uint(values->unk_5, 5, 5) |
            util_bitpack_uint(values->unk_6, 6, 6) |
            util_bitpack_uint(values->unk_8, 8, 8) |
            util_bitpack_uint(values->unk_11, 11, 11) |
            util_bitpack_uint(values->unk_20, 20, 20) |
            util_bitpack_uint(values->unk_24, 24, 24) |
            util_bitpack_uint(values->unk_26, 26, 26) |
            util_bitpack_uint(values->returns, 27, 27) |
            util_bitpack_uint(AGX_VDM_BLOCK_TYPE_BARRIER, 29, 31);
}

#define AGX_VDM_BARRIER_LENGTH 4
struct agx_vdm_barrier_packed { uint32_t opaque[1];};
static inline bool
AGX_VDM_BARRIER_unpack(FILE *fp, CONST uint8_t * restrict cl,
                       struct AGX_VDM_BARRIER * restrict values)
{
   values->usc_cache_inval = __gen_unpack_uint((CONST uint32_t *) cl, 3, 3);
   values->unk_4 = __gen_unpack_uint((CONST uint32_t *) cl, 4, 4);
   values->unk_5 = __gen_unpack_uint((CONST uint32_t *) cl, 5, 5);
   values->unk_6 = __gen_unpack_uint((CONST uint32_t *) cl, 6, 6);
   values->unk_8 = __gen_unpack_uint((CONST uint32_t *) cl, 8, 8);
   values->unk_11 = __gen_unpack_uint((CONST uint32_t *) cl, 11, 11);
   values->unk_20 = __gen_unpack_uint((CONST uint32_t *) cl, 20, 20);
   values->unk_24 = __gen_unpack_uint((CONST uint32_t *) cl, 24, 24);
   values->unk_26 = __gen_unpack_uint((CONST uint32_t *) cl, 26, 26);
   values->returns = __gen_unpack_uint((CONST uint32_t *) cl, 27, 27);
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "VDM Barrier", cl, 0, 0x12eff687);
   valid &= agx_genxml_validate_exact(fp, "VDM Barrier", __gen_unpack_uint((CONST uint32_t *) cl, 29, 31), AGX_VDM_BLOCK_TYPE_BARRIER);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VDM_BARRIER_print(FILE *fp, const struct AGX_VDM_BARRIER * values, unsigned indent)
{
   fprintf(fp, "%*sUSC cache inval: %s\n", indent, "", values->usc_cache_inval ? "true" : "false");
   fprintf(fp, "%*sUnk 4: %s\n", indent, "", values->unk_4 ? "true" : "false");
   fprintf(fp, "%*sUnk 5: %s\n", indent, "", values->unk_5 ? "true" : "false");
   fprintf(fp, "%*sUnk 6: %s\n", indent, "", values->unk_6 ? "true" : "false");
   fprintf(fp, "%*sUnk 8: %s\n", indent, "", values->unk_8 ? "true" : "false");
   fprintf(fp, "%*sUnk 11: %s\n", indent, "", values->unk_11 ? "true" : "false");
   fprintf(fp, "%*sUnk 20: %s\n", indent, "", values->unk_20 ? "true" : "false");
   fprintf(fp, "%*sUnk 24: %s\n", indent, "", values->unk_24 ? "true" : "false");
   fprintf(fp, "%*sUnk 26: %s\n", indent, "", values->unk_26 ? "true" : "false");
   fprintf(fp, "%*sReturns: %s\n", indent, "", values->returns ? "true" : "false");
}
#endif

enum agx_index_size {
   AGX_INDEX_SIZE_U8 = 0,
   AGX_INDEX_SIZE_U16 = 1,
   AGX_INDEX_SIZE_U32 = 2,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_index_size_as_str(enum agx_index_size imm)
{
    switch (imm) {
    case AGX_INDEX_SIZE_U8: return "U8";
    case AGX_INDEX_SIZE_U16: return "U16";
    case AGX_INDEX_SIZE_U32: return "U32";
    default: return NULL;
    }
}
#endif

struct AGX_VDM_STATE {
   bool                                 restart_index_present;
   bool                                 vertex_shader_word_0_present;
   bool                                 vertex_shader_word_1_present;
   bool                                 vertex_outputs_present;
   bool                                 tessellation_present;
   bool                                 vertex_unknown_present;
   bool                                 tessellation_scale_present;
};

#define AGX_VDM_STATE_header 0

static inline void
AGX_VDM_STATE_pack(GLOBAL uint32_t * restrict cl,
                   const struct AGX_VDM_STATE * restrict values)
{
   agx_genxml_validate_bounds("VDM State::restart_index_present", values->restart_index_present, 0x2ull);
   agx_genxml_validate_bounds("VDM State::vertex_shader_word_0_present", values->vertex_shader_word_0_present, 0x2ull);
   agx_genxml_validate_bounds("VDM State::vertex_shader_word_1_present", values->vertex_shader_word_1_present, 0x2ull);
   agx_genxml_validate_bounds("VDM State::vertex_outputs_present", values->vertex_outputs_present, 0x2ull);
   agx_genxml_validate_bounds("VDM State::tessellation_present", values->tessellation_present, 0x2ull);
   agx_genxml_validate_bounds("VDM State::vertex_unknown_present", values->vertex_unknown_present, 0x2ull);
   agx_genxml_validate_bounds("VDM State::tessellation_scale_present", values->tessellation_scale_present, 0x2ull);
   cl[ 0] = util_bitpack_uint(values->restart_index_present, 0, 0) |
            util_bitpack_uint(values->vertex_shader_word_0_present, 1, 1) |
            util_bitpack_uint(values->vertex_shader_word_1_present, 2, 2) |
            util_bitpack_uint(values->vertex_outputs_present, 3, 3) |
            util_bitpack_uint(values->tessellation_present, 4, 4) |
            util_bitpack_uint(values->vertex_unknown_present, 5, 5) |
            util_bitpack_uint(values->tessellation_scale_present, 7, 7) |
            util_bitpack_uint(AGX_VDM_BLOCK_TYPE_VDM_STATE_UPDATE, 29, 31);
}

#define AGX_VDM_STATE_LENGTH 4
struct agx_vdm_state_packed { uint32_t opaque[1];};
static inline bool
AGX_VDM_STATE_unpack(FILE *fp, CONST uint8_t * restrict cl,
                     struct AGX_VDM_STATE * restrict values)
{
   values->restart_index_present = __gen_unpack_uint((CONST uint32_t *) cl, 0, 0);
   values->vertex_shader_word_0_present = __gen_unpack_uint((CONST uint32_t *) cl, 1, 1);
   values->vertex_shader_word_1_present = __gen_unpack_uint((CONST uint32_t *) cl, 2, 2);
   values->vertex_outputs_present = __gen_unpack_uint((CONST uint32_t *) cl, 3, 3);
   values->tessellation_present = __gen_unpack_uint((CONST uint32_t *) cl, 4, 4);
   values->vertex_unknown_present = __gen_unpack_uint((CONST uint32_t *) cl, 5, 5);
   values->tessellation_scale_present = __gen_unpack_uint((CONST uint32_t *) cl, 7, 7);
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "VDM State", cl, 0, 0x1fffff40);
   valid &= agx_genxml_validate_exact(fp, "VDM State", __gen_unpack_uint((CONST uint32_t *) cl, 29, 31), AGX_VDM_BLOCK_TYPE_VDM_STATE_UPDATE);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VDM_STATE_print(FILE *fp, const struct AGX_VDM_STATE * values, unsigned indent)
{
   fprintf(fp, "%*sRestart index present: %s\n", indent, "", values->restart_index_present ? "true" : "false");
   fprintf(fp, "%*sVertex shader word 0 present: %s\n", indent, "", values->vertex_shader_word_0_present ? "true" : "false");
   fprintf(fp, "%*sVertex shader word 1 present: %s\n", indent, "", values->vertex_shader_word_1_present ? "true" : "false");
   fprintf(fp, "%*sVertex outputs present: %s\n", indent, "", values->vertex_outputs_present ? "true" : "false");
   fprintf(fp, "%*sTessellation present: %s\n", indent, "", values->tessellation_present ? "true" : "false");
   fprintf(fp, "%*sVertex unknown present: %s\n", indent, "", values->vertex_unknown_present ? "true" : "false");
   fprintf(fp, "%*sTessellation scale present: %s\n", indent, "", values->tessellation_scale_present ? "true" : "false");
}
#endif

struct AGX_VDM_STATE_RESTART_INDEX {
   uint32_t                             value;
};

#define AGX_VDM_STATE_RESTART_INDEX_header 0

static inline void
AGX_VDM_STATE_RESTART_INDEX_pack(GLOBAL uint32_t * restrict cl,
                                 const struct AGX_VDM_STATE_RESTART_INDEX * restrict values)
{
   agx_genxml_validate_bounds("VDM State Restart Index::value", values->value, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->value, 0, 31);
}

#define AGX_VDM_STATE_RESTART_INDEX_LENGTH 4
struct agx_vdm_state_restart_index_packed { uint32_t opaque[1];};
static inline bool
AGX_VDM_STATE_RESTART_INDEX_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                   struct AGX_VDM_STATE_RESTART_INDEX * restrict values)
{
   values->value = __gen_unpack_uint((CONST uint32_t *) cl, 0, 31);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VDM_STATE_RESTART_INDEX_print(FILE *fp, const struct AGX_VDM_STATE_RESTART_INDEX * values, unsigned indent)
{
   fprintf(fp, "%*sValue: 0x%" PRIx32 "\n", indent, "", values->value);
}
#endif

struct AGX_VDM_STATE_VERTEX_SHADER_WORD_0 {
   uint32_t                             unknown_0;
   uint32_t                             uniform_register_count;
   uint32_t                             texture_state_register_count;
   enum agx_sampler_states              sampler_state_register_count;
   uint32_t                             preshader_register_count;
};

#define AGX_VDM_STATE_VERTEX_SHADER_WORD_0_header\
   .uniform_register_count = 512,  \
   .texture_state_register_count = 256,  \
   .preshader_register_count = 256

static inline void
AGX_VDM_STATE_VERTEX_SHADER_WORD_0_pack(GLOBAL uint32_t * restrict cl,
                                        const struct AGX_VDM_STATE_VERTEX_SHADER_WORD_0 * restrict values)
{
   agx_genxml_validate_bounds("VDM State Vertex Shader Word 0::unknown_0", values->unknown_0, 0x2ull);
   agx_genxml_validate_bounds("VDM State Vertex Shader Word 0::uniform_register_count", __gen_to_groups(values->uniform_register_count, 64, 3), 0x8ull);
   agx_genxml_validate_bounds("VDM State Vertex Shader Word 0::texture_state_register_count", __gen_to_groups(values->texture_state_register_count, 8, 5), 0x20ull);
   agx_genxml_validate_bounds("VDM State Vertex Shader Word 0::sampler_state_register_count", values->sampler_state_register_count, 0x8ull);
   agx_genxml_validate_bounds("VDM State Vertex Shader Word 0::preshader_register_count", __gen_to_groups(values->preshader_register_count, 16, 4), 0x10ull);
   cl[ 0] = util_bitpack_uint(values->unknown_0, 0, 0) |
            util_bitpack_uint(__gen_to_groups(values->uniform_register_count, 64, 3), 1, 3) |
            util_bitpack_uint(__gen_to_groups(values->texture_state_register_count, 8, 5), 4, 8) |
            util_bitpack_uint(values->sampler_state_register_count, 9, 11) |
            util_bitpack_uint(__gen_to_groups(values->preshader_register_count, 16, 4), 12, 15);
}

#define AGX_VDM_STATE_VERTEX_SHADER_WORD_0_LENGTH 4
struct agx_vdm_state_vertex_shader_word_0_packed { uint32_t opaque[1];};
static inline bool
AGX_VDM_STATE_VERTEX_SHADER_WORD_0_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                          struct AGX_VDM_STATE_VERTEX_SHADER_WORD_0 * restrict values)
{
   values->unknown_0 = __gen_unpack_uint((CONST uint32_t *) cl, 0, 0);
   values->uniform_register_count = __gen_from_groups(__gen_unpack_uint((CONST uint32_t *) cl, 1, 3), 64, 3);
   values->texture_state_register_count = __gen_from_groups(__gen_unpack_uint((CONST uint32_t *) cl, 4, 8), 8, 5);
   values->sampler_state_register_count = (enum agx_sampler_states) __gen_unpack_uint((CONST uint32_t *) cl, 9, 11);
   values->preshader_register_count = __gen_from_groups(__gen_unpack_uint((CONST uint32_t *) cl, 12, 15), 16, 4);
   return agx_genxml_validate_mask(fp, "VDM State Vertex Shader Word 0", cl, 0, 0xffff0000);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VDM_STATE_VERTEX_SHADER_WORD_0_print(FILE *fp, const struct AGX_VDM_STATE_VERTEX_SHADER_WORD_0 * values, unsigned indent)
{
   fprintf(fp, "%*sUnknown 0: 0x%" PRIx32 "\n", indent, "", values->unknown_0);
   fprintf(fp, "%*sUniform register count: %u\n", indent, "", values->uniform_register_count);
   fprintf(fp, "%*sTexture state register count: %u\n", indent, "", values->texture_state_register_count);
   if (agx_sampler_states_as_str(values->sampler_state_register_count))
     fprintf(fp, "%*sSampler state register count: %s\n", indent, "", agx_sampler_states_as_str(values->sampler_state_register_count));
   else
     fprintf(fp, "%*sSampler state register count: unknown %X (XXX)\n", indent, "", values->sampler_state_register_count);
   fprintf(fp, "%*sPreshader register count: %u\n", indent, "", values->preshader_register_count);
}
#endif

struct AGX_VDM_STATE_VERTEX_SHADER_WORD_1 {
   uint32_t                             unknown_10;
   uint64_t                             pipeline;
};

#define AGX_VDM_STATE_VERTEX_SHADER_WORD_1_header 0

static inline void
AGX_VDM_STATE_VERTEX_SHADER_WORD_1_pack(GLOBAL uint32_t * restrict cl,
                                        const struct AGX_VDM_STATE_VERTEX_SHADER_WORD_1 * restrict values)
{
   assert((values->pipeline & 0x3f) == 0);
   agx_genxml_validate_bounds("VDM State Vertex Shader Word 1::unknown_10", values->unknown_10, 0x4ull);
   agx_genxml_validate_bounds("VDM State Vertex Shader Word 1::pipeline", values->pipeline, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->unknown_10, 0, 1) |
            util_bitpack_uint(values->pipeline, 0, 31);
}

#define AGX_VDM_STATE_VERTEX_SHADER_WORD_1_LENGTH 4
struct agx_vdm_state_vertex_shader_word_1_packed { uint32_t opaque[1];};
static inline bool
AGX_VDM_STATE_VERTEX_SHADER_WORD_1_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                          struct AGX_VDM_STATE_VERTEX_SHADER_WORD_1 * restrict values)
{
   values->unknown_10 = __gen_unpack_uint((CONST uint32_t *) cl, 0, 1);
   values->pipeline = __gen_unpack_uint((CONST uint32_t *) cl, 6, 31) << 6;
   return agx_genxml_validate_mask(fp, "VDM State Vertex Shader Word 1", cl, 0, 0x3c);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VDM_STATE_VERTEX_SHADER_WORD_1_print(FILE *fp, const struct AGX_VDM_STATE_VERTEX_SHADER_WORD_1 * values, unsigned indent)
{
   fprintf(fp, "%*sUnknown 1:0: 0x%" PRIx32 "\n", indent, "", values->unknown_10);
   fprintf(fp, "%*sPipeline: 0x%" PRIx64 "\n", indent, "", values->pipeline);
}
#endif

struct AGX_VDM_STATE_VERTEX_OUTPUTS {
   uint32_t                             output_count_1;
   uint32_t                             output_count_2;
};

#define AGX_VDM_STATE_VERTEX_OUTPUTS_header 0

static inline void
AGX_VDM_STATE_VERTEX_OUTPUTS_pack(GLOBAL uint32_t * restrict cl,
                                  const struct AGX_VDM_STATE_VERTEX_OUTPUTS * restrict values)
{
   agx_genxml_validate_bounds("VDM State Vertex Outputs::output_count_1", values->output_count_1, 0x100ull);
   agx_genxml_validate_bounds("VDM State Vertex Outputs::output_count_2", values->output_count_2, 0x100ull);
   cl[ 0] = util_bitpack_uint(values->output_count_1, 0, 7) |
            util_bitpack_uint(values->output_count_2, 8, 15);
}

#define AGX_VDM_STATE_VERTEX_OUTPUTS_LENGTH 4
struct agx_vdm_state_vertex_outputs_packed { uint32_t opaque[1];};
static inline bool
AGX_VDM_STATE_VERTEX_OUTPUTS_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                    struct AGX_VDM_STATE_VERTEX_OUTPUTS * restrict values)
{
   values->output_count_1 = __gen_unpack_uint((CONST uint32_t *) cl, 0, 7);
   values->output_count_2 = __gen_unpack_uint((CONST uint32_t *) cl, 8, 15);
   return agx_genxml_validate_mask(fp, "VDM State Vertex Outputs", cl, 0, 0xffff0000);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VDM_STATE_VERTEX_OUTPUTS_print(FILE *fp, const struct AGX_VDM_STATE_VERTEX_OUTPUTS * values, unsigned indent)
{
   fprintf(fp, "%*sOutput count 1: %u\n", indent, "", values->output_count_1);
   fprintf(fp, "%*sOutput count 2: %u\n", indent, "", values->output_count_2);
}
#endif

enum agx_vdm_vertex {
   AGX_VDM_VERTEX_0 = 0,
   AGX_VDM_VERTEX_1 = 1,
   AGX_VDM_VERTEX_2 = 2,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_vdm_vertex_as_str(enum agx_vdm_vertex imm)
{
    switch (imm) {
    case AGX_VDM_VERTEX_0: return "0";
    case AGX_VDM_VERTEX_1: return "1";
    case AGX_VDM_VERTEX_2: return "2";
    default: return NULL;
    }
}
#endif

enum agx_partition_mode {
   AGX_PARTITION_MODE_POW_2 = 0,
   AGX_PARTITION_MODE_INTEGER = 1,
   AGX_PARTITION_MODE_FRACTIONAL_ODD = 2,
   AGX_PARTITION_MODE_FRACTIONAL_EVEN = 3,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_partition_mode_as_str(enum agx_partition_mode imm)
{
    switch (imm) {
    case AGX_PARTITION_MODE_POW_2: return "Pow 2";
    case AGX_PARTITION_MODE_INTEGER: return "Integer";
    case AGX_PARTITION_MODE_FRACTIONAL_ODD: return "Fractional odd";
    case AGX_PARTITION_MODE_FRACTIONAL_EVEN: return "Fractional even";
    default: return NULL;
    }
}
#endif

enum agx_step_function {
   AGX_STEP_FUNCTION_CONSTANT = 0,
   AGX_STEP_FUNCTION_PER_PATCH = 1,
   AGX_STEP_FUNCTION_PER_INSTANCE = 2,
   AGX_STEP_FUNCTION_PER_PATCH_AND_PER_INSTANCE = 3,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_step_function_as_str(enum agx_step_function imm)
{
    switch (imm) {
    case AGX_STEP_FUNCTION_CONSTANT: return "Constant";
    case AGX_STEP_FUNCTION_PER_PATCH: return "Per patch";
    case AGX_STEP_FUNCTION_PER_INSTANCE: return "Per instance";
    case AGX_STEP_FUNCTION_PER_PATCH_AND_PER_INSTANCE: return "Per patch and per instance";
    default: return NULL;
    }
}
#endif

enum agx_patch_type {
   AGX_PATCH_TYPE_TRIANGLES = 0,
   AGX_PATCH_TYPE_QUADS = 1,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_patch_type_as_str(enum agx_patch_type imm)
{
    switch (imm) {
    case AGX_PATCH_TYPE_TRIANGLES: return "Triangles";
    case AGX_PATCH_TYPE_QUADS: return "Quads";
    default: return NULL;
    }
}
#endif

enum agx_factor_type {
   AGX_FACTOR_TYPE_FP16 = 0,
   AGX_FACTOR_TYPE_FP32 = 1,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_factor_type_as_str(enum agx_factor_type imm)
{
    switch (imm) {
    case AGX_FACTOR_TYPE_FP16: return "FP16";
    case AGX_FACTOR_TYPE_FP32: return "FP32";
    default: return NULL;
    }
}
#endif

struct AGX_VDM_STATE_TESSELLATION {
   uint32_t                             unknown;
   enum agx_step_function               step_function;
   enum agx_factor_type                 factor_type;
   enum agx_patch_type                  patch_type;
   uint32_t                             unknown_3;
   uint32_t                             max_tess_factor;
   bool                                 counterclockwise_winding;
   enum agx_partition_mode              partition_mode;
};

#define AGX_VDM_STATE_TESSELLATION_header       \
   .max_tess_factor = 1

static inline void
AGX_VDM_STATE_TESSELLATION_pack(GLOBAL uint32_t * restrict cl,
                                const struct AGX_VDM_STATE_TESSELLATION * restrict values)
{
   assert(values->max_tess_factor >= 1);
   agx_genxml_validate_bounds("VDM State Tessellation::unknown", values->unknown, 0x20000ull);
   agx_genxml_validate_bounds("VDM State Tessellation::step_function", values->step_function, 0x4ull);
   agx_genxml_validate_bounds("VDM State Tessellation::factor_type", values->factor_type, 0x2ull);
   agx_genxml_validate_bounds("VDM State Tessellation::patch_type", values->patch_type, 0x2ull);
   agx_genxml_validate_bounds("VDM State Tessellation::unknown_3", values->unknown_3, 0x4ull);
   agx_genxml_validate_bounds("VDM State Tessellation::max_tess_factor", values->max_tess_factor - 1, 0x40ull);
   agx_genxml_validate_bounds("VDM State Tessellation::counterclockwise_winding", values->counterclockwise_winding, 0x2ull);
   agx_genxml_validate_bounds("VDM State Tessellation::partition_mode", values->partition_mode, 0x4ull);
   cl[ 0] = util_bitpack_uint(values->unknown, 0, 16) |
            util_bitpack_uint(values->step_function, 17, 18) |
            util_bitpack_uint(values->factor_type, 19, 19) |
            util_bitpack_uint(values->patch_type, 20, 20) |
            util_bitpack_uint(values->unknown_3, 21, 22) |
            util_bitpack_uint(values->max_tess_factor - 1, 23, 28) |
            util_bitpack_uint(values->counterclockwise_winding, 29, 29) |
            util_bitpack_uint(values->partition_mode, 30, 31);
}

#define AGX_VDM_STATE_TESSELLATION_LENGTH 4
struct agx_vdm_state_tessellation_packed { uint32_t opaque[1];};
static inline bool
AGX_VDM_STATE_TESSELLATION_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                  struct AGX_VDM_STATE_TESSELLATION * restrict values)
{
   values->unknown = __gen_unpack_uint((CONST uint32_t *) cl, 0, 16);
   values->step_function = (enum agx_step_function) __gen_unpack_uint((CONST uint32_t *) cl, 17, 18);
   values->factor_type = (enum agx_factor_type) __gen_unpack_uint((CONST uint32_t *) cl, 19, 19);
   values->patch_type = (enum agx_patch_type) __gen_unpack_uint((CONST uint32_t *) cl, 20, 20);
   values->unknown_3 = __gen_unpack_uint((CONST uint32_t *) cl, 21, 22);
   values->max_tess_factor = __gen_unpack_uint((CONST uint32_t *) cl, 23, 28) + 1;
   values->counterclockwise_winding = __gen_unpack_uint((CONST uint32_t *) cl, 29, 29);
   values->partition_mode = (enum agx_partition_mode) __gen_unpack_uint((CONST uint32_t *) cl, 30, 31);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VDM_STATE_TESSELLATION_print(FILE *fp, const struct AGX_VDM_STATE_TESSELLATION * values, unsigned indent)
{
   fprintf(fp, "%*sUnknown: 0x%" PRIx32 "\n", indent, "", values->unknown);
   if (agx_step_function_as_str(values->step_function))
     fprintf(fp, "%*sStep function: %s\n", indent, "", agx_step_function_as_str(values->step_function));
   else
     fprintf(fp, "%*sStep function: unknown %X (XXX)\n", indent, "", values->step_function);
   if (agx_factor_type_as_str(values->factor_type))
     fprintf(fp, "%*sFactor type: %s\n", indent, "", agx_factor_type_as_str(values->factor_type));
   else
     fprintf(fp, "%*sFactor type: unknown %X (XXX)\n", indent, "", values->factor_type);
   if (agx_patch_type_as_str(values->patch_type))
     fprintf(fp, "%*sPatch type: %s\n", indent, "", agx_patch_type_as_str(values->patch_type));
   else
     fprintf(fp, "%*sPatch type: unknown %X (XXX)\n", indent, "", values->patch_type);
   fprintf(fp, "%*sUnknown 3: 0x%" PRIx32 "\n", indent, "", values->unknown_3);
   fprintf(fp, "%*sMax tess factor: %u\n", indent, "", values->max_tess_factor);
   fprintf(fp, "%*sCounterclockwise winding: %s\n", indent, "", values->counterclockwise_winding ? "true" : "false");
   if (agx_partition_mode_as_str(values->partition_mode))
     fprintf(fp, "%*sPartition mode: %s\n", indent, "", agx_partition_mode_as_str(values->partition_mode));
   else
     fprintf(fp, "%*sPartition mode: unknown %X (XXX)\n", indent, "", values->partition_mode);
}
#endif

struct AGX_VDM_STATE_TESSELLATION_SCALE {
   float                                scale;
};

#define AGX_VDM_STATE_TESSELLATION_SCALE_header 0

static inline void
AGX_VDM_STATE_TESSELLATION_SCALE_pack(GLOBAL uint32_t * restrict cl,
                                      const struct AGX_VDM_STATE_TESSELLATION_SCALE * restrict values)
{
   cl[ 0] = _mesa_float_to_half(values->scale);
}

#define AGX_VDM_STATE_TESSELLATION_SCALE_LENGTH 4
struct agx_vdm_state_tessellation_scale_packed { uint32_t opaque[1];};
static inline bool
AGX_VDM_STATE_TESSELLATION_SCALE_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                        struct AGX_VDM_STATE_TESSELLATION_SCALE * restrict values)
{
   values->scale = __gen_unpack_half((CONST uint32_t *) cl, 0, 15);
   return agx_genxml_validate_mask(fp, "VDM State Tessellation Scale", cl, 0, 0xffff0000);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VDM_STATE_TESSELLATION_SCALE_print(FILE *fp, const struct AGX_VDM_STATE_TESSELLATION_SCALE * values, unsigned indent)
{
   fprintf(fp, "%*sScale: %f\n", indent, "", values->scale);
}
#endif

struct AGX_VDM_STATE_VERTEX_UNKNOWN {
   enum agx_vdm_vertex                  flat_shading_control;
   bool                                 unknown_4;
   bool                                 unknown_5;
   bool                                 generate_primitive_id;
};

#define AGX_VDM_STATE_VERTEX_UNKNOWN_header 0

static inline void
AGX_VDM_STATE_VERTEX_UNKNOWN_pack(GLOBAL uint32_t * restrict cl,
                                  const struct AGX_VDM_STATE_VERTEX_UNKNOWN * restrict values)
{
   agx_genxml_validate_bounds("VDM State Vertex Unknown::flat_shading_control", values->flat_shading_control, 0x4ull);
   agx_genxml_validate_bounds("VDM State Vertex Unknown::unknown_4", values->unknown_4, 0x2ull);
   agx_genxml_validate_bounds("VDM State Vertex Unknown::unknown_5", values->unknown_5, 0x2ull);
   agx_genxml_validate_bounds("VDM State Vertex Unknown::generate_primitive_id", values->generate_primitive_id, 0x2ull);
   cl[ 0] = util_bitpack_uint(values->flat_shading_control, 0, 1) |
            util_bitpack_uint(values->unknown_4, 4, 4) |
            util_bitpack_uint(values->unknown_5, 5, 5) |
            util_bitpack_uint(values->generate_primitive_id, 6, 6);
}

#define AGX_VDM_STATE_VERTEX_UNKNOWN_LENGTH 4
struct agx_vdm_state_vertex_unknown_packed { uint32_t opaque[1];};
static inline bool
AGX_VDM_STATE_VERTEX_UNKNOWN_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                    struct AGX_VDM_STATE_VERTEX_UNKNOWN * restrict values)
{
   values->flat_shading_control = (enum agx_vdm_vertex) __gen_unpack_uint((CONST uint32_t *) cl, 0, 1);
   values->unknown_4 = __gen_unpack_uint((CONST uint32_t *) cl, 4, 4);
   values->unknown_5 = __gen_unpack_uint((CONST uint32_t *) cl, 5, 5);
   values->generate_primitive_id = __gen_unpack_uint((CONST uint32_t *) cl, 6, 6);
   return agx_genxml_validate_mask(fp, "VDM State Vertex Unknown", cl, 0, 0xffffff8c);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VDM_STATE_VERTEX_UNKNOWN_print(FILE *fp, const struct AGX_VDM_STATE_VERTEX_UNKNOWN * values, unsigned indent)
{
   if (agx_vdm_vertex_as_str(values->flat_shading_control))
     fprintf(fp, "%*sFlat shading control: %s\n", indent, "", agx_vdm_vertex_as_str(values->flat_shading_control));
   else
     fprintf(fp, "%*sFlat shading control: unknown %X (XXX)\n", indent, "", values->flat_shading_control);
   fprintf(fp, "%*sUnknown 4: %s\n", indent, "", values->unknown_4 ? "true" : "false");
   fprintf(fp, "%*sUnknown 5: %s\n", indent, "", values->unknown_5 ? "true" : "false");
   fprintf(fp, "%*sGenerate primitive ID: %s\n", indent, "", values->generate_primitive_id ? "true" : "false");
}
#endif

struct AGX_INDEX_LIST {
   uint32_t                             index_buffer_hi;
   enum agx_primitive                   primitive;
   bool                                 restart_enable;
   enum agx_index_size                  index_size;
   bool                                 index_buffer_size_present;
   bool                                 index_buffer_present;
   bool                                 index_count_present;
   bool                                 instance_count_present;
   bool                                 start_present;
   bool                                 unk_1_present;
   bool                                 indirect_buffer_present;
   bool                                 unk_2_present;
};

#define AGX_INDEX_LIST_header 0

static inline void
AGX_INDEX_LIST_pack(GLOBAL uint32_t * restrict cl,
                    const struct AGX_INDEX_LIST * restrict values)
{
   agx_genxml_validate_bounds("Index List::index_buffer_hi", values->index_buffer_hi, 0x100ull);
   agx_genxml_validate_bounds("Index List::primitive", values->primitive, 0x100ull);
   agx_genxml_validate_bounds("Index List::restart_enable", values->restart_enable, 0x2ull);
   agx_genxml_validate_bounds("Index List::index_size", values->index_size, 0x8ull);
   agx_genxml_validate_bounds("Index List::index_buffer_size_present", values->index_buffer_size_present, 0x2ull);
   agx_genxml_validate_bounds("Index List::index_buffer_present", values->index_buffer_present, 0x2ull);
   agx_genxml_validate_bounds("Index List::index_count_present", values->index_count_present, 0x2ull);
   agx_genxml_validate_bounds("Index List::instance_count_present", values->instance_count_present, 0x2ull);
   agx_genxml_validate_bounds("Index List::start_present", values->start_present, 0x2ull);
   agx_genxml_validate_bounds("Index List::unk_1_present", values->unk_1_present, 0x2ull);
   agx_genxml_validate_bounds("Index List::indirect_buffer_present", values->indirect_buffer_present, 0x2ull);
   agx_genxml_validate_bounds("Index List::unk_2_present", values->unk_2_present, 0x2ull);
   cl[ 0] = util_bitpack_uint(values->index_buffer_hi, 0, 7) |
            util_bitpack_uint(values->primitive, 8, 15) |
            util_bitpack_uint(values->restart_enable, 16, 16) |
            util_bitpack_uint(values->index_size, 17, 19) |
            util_bitpack_uint(values->index_buffer_size_present, 20, 20) |
            util_bitpack_uint(values->index_buffer_present, 21, 21) |
            util_bitpack_uint(values->index_count_present, 22, 22) |
            util_bitpack_uint(values->instance_count_present, 23, 23) |
            util_bitpack_uint(values->start_present, 24, 24) |
            util_bitpack_uint(values->unk_1_present, 25, 25) |
            util_bitpack_uint(values->indirect_buffer_present, 26, 26) |
            util_bitpack_uint(values->unk_2_present, 27, 27) |
            util_bitpack_uint(AGX_VDM_BLOCK_TYPE_INDEX_LIST, 29, 31);
}

#define AGX_INDEX_LIST_LENGTH 4
struct agx_index_list_packed { uint32_t opaque[1];};
static inline bool
AGX_INDEX_LIST_unpack(FILE *fp, CONST uint8_t * restrict cl,
                      struct AGX_INDEX_LIST * restrict values)
{
   values->index_buffer_hi = __gen_unpack_uint((CONST uint32_t *) cl, 0, 7);
   values->primitive = (enum agx_primitive) __gen_unpack_uint((CONST uint32_t *) cl, 8, 15);
   values->restart_enable = __gen_unpack_uint((CONST uint32_t *) cl, 16, 16);
   values->index_size = (enum agx_index_size) __gen_unpack_uint((CONST uint32_t *) cl, 17, 19);
   values->index_buffer_size_present = __gen_unpack_uint((CONST uint32_t *) cl, 20, 20);
   values->index_buffer_present = __gen_unpack_uint((CONST uint32_t *) cl, 21, 21);
   values->index_count_present = __gen_unpack_uint((CONST uint32_t *) cl, 22, 22);
   values->instance_count_present = __gen_unpack_uint((CONST uint32_t *) cl, 23, 23);
   values->start_present = __gen_unpack_uint((CONST uint32_t *) cl, 24, 24);
   values->unk_1_present = __gen_unpack_uint((CONST uint32_t *) cl, 25, 25);
   values->indirect_buffer_present = __gen_unpack_uint((CONST uint32_t *) cl, 26, 26);
   values->unk_2_present = __gen_unpack_uint((CONST uint32_t *) cl, 27, 27);
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "Index List", cl, 0, 0x10000000);
   valid &= agx_genxml_validate_exact(fp, "Index List", __gen_unpack_uint((CONST uint32_t *) cl, 29, 31), AGX_VDM_BLOCK_TYPE_INDEX_LIST);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_INDEX_LIST_print(FILE *fp, const struct AGX_INDEX_LIST * values, unsigned indent)
{
   fprintf(fp, "%*sIndex buffer hi: 0x%" PRIx32 "\n", indent, "", values->index_buffer_hi);
   if (agx_primitive_as_str(values->primitive))
     fprintf(fp, "%*sPrimitive: %s\n", indent, "", agx_primitive_as_str(values->primitive));
   else
     fprintf(fp, "%*sPrimitive: unknown %X (XXX)\n", indent, "", values->primitive);
   fprintf(fp, "%*sRestart enable: %s\n", indent, "", values->restart_enable ? "true" : "false");
   if (agx_index_size_as_str(values->index_size))
     fprintf(fp, "%*sIndex size: %s\n", indent, "", agx_index_size_as_str(values->index_size));
   else
     fprintf(fp, "%*sIndex size: unknown %X (XXX)\n", indent, "", values->index_size);
   fprintf(fp, "%*sIndex buffer size present: %s\n", indent, "", values->index_buffer_size_present ? "true" : "false");
   fprintf(fp, "%*sIndex buffer present: %s\n", indent, "", values->index_buffer_present ? "true" : "false");
   fprintf(fp, "%*sIndex count present: %s\n", indent, "", values->index_count_present ? "true" : "false");
   fprintf(fp, "%*sInstance count present: %s\n", indent, "", values->instance_count_present ? "true" : "false");
   fprintf(fp, "%*sStart present: %s\n", indent, "", values->start_present ? "true" : "false");
   fprintf(fp, "%*sUnk 1 present: %s\n", indent, "", values->unk_1_present ? "true" : "false");
   fprintf(fp, "%*sIndirect buffer present: %s\n", indent, "", values->indirect_buffer_present ? "true" : "false");
   fprintf(fp, "%*sUnk 2 present: %s\n", indent, "", values->unk_2_present ? "true" : "false");
}
#endif

struct AGX_INDEX_LIST_BUFFER_LO {
   uint32_t                             buffer_lo;
};

#define AGX_INDEX_LIST_BUFFER_LO_header 0

static inline void
AGX_INDEX_LIST_BUFFER_LO_pack(GLOBAL uint32_t * restrict cl,
                              const struct AGX_INDEX_LIST_BUFFER_LO * restrict values)
{
   agx_genxml_validate_bounds("Index List: Buffer lo::buffer_lo", values->buffer_lo, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->buffer_lo, 0, 31);
}

#define AGX_INDEX_LIST_BUFFER_LO_LENGTH 4
struct agx_index_list_buffer_lo_packed { uint32_t opaque[1];};
static inline bool
AGX_INDEX_LIST_BUFFER_LO_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                struct AGX_INDEX_LIST_BUFFER_LO * restrict values)
{
   values->buffer_lo = __gen_unpack_uint((CONST uint32_t *) cl, 0, 31);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_INDEX_LIST_BUFFER_LO_print(FILE *fp, const struct AGX_INDEX_LIST_BUFFER_LO * values, unsigned indent)
{
   fprintf(fp, "%*sBuffer lo: 0x%" PRIx32 "\n", indent, "", values->buffer_lo);
}
#endif

struct AGX_INDEX_LIST_COUNT {
   uint32_t                             count;
};

#define AGX_INDEX_LIST_COUNT_header 0

static inline void
AGX_INDEX_LIST_COUNT_pack(GLOBAL uint32_t * restrict cl,
                          const struct AGX_INDEX_LIST_COUNT * restrict values)
{
   agx_genxml_validate_bounds("Index List: Count::count", values->count, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->count, 0, 31);
}

#define AGX_INDEX_LIST_COUNT_LENGTH 4
struct agx_index_list_count_packed { uint32_t opaque[1];};
static inline bool
AGX_INDEX_LIST_COUNT_unpack(FILE *fp, CONST uint8_t * restrict cl,
                            struct AGX_INDEX_LIST_COUNT * restrict values)
{
   values->count = __gen_unpack_uint((CONST uint32_t *) cl, 0, 31);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_INDEX_LIST_COUNT_print(FILE *fp, const struct AGX_INDEX_LIST_COUNT * values, unsigned indent)
{
   fprintf(fp, "%*sCount: %u\n", indent, "", values->count);
}
#endif

struct AGX_INDEX_LIST_INSTANCES {
   uint32_t                             count;
};

#define AGX_INDEX_LIST_INSTANCES_header 0

static inline void
AGX_INDEX_LIST_INSTANCES_pack(GLOBAL uint32_t * restrict cl,
                              const struct AGX_INDEX_LIST_INSTANCES * restrict values)
{
   agx_genxml_validate_bounds("Index List: Instances::count", values->count, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->count, 0, 31);
}

#define AGX_INDEX_LIST_INSTANCES_LENGTH 4
struct agx_index_list_instances_packed { uint32_t opaque[1];};
static inline bool
AGX_INDEX_LIST_INSTANCES_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                struct AGX_INDEX_LIST_INSTANCES * restrict values)
{
   values->count = __gen_unpack_uint((CONST uint32_t *) cl, 0, 31);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_INDEX_LIST_INSTANCES_print(FILE *fp, const struct AGX_INDEX_LIST_INSTANCES * values, unsigned indent)
{
   fprintf(fp, "%*sCount: %u\n", indent, "", values->count);
}
#endif

struct AGX_INDEX_LIST_START {
   uint32_t                             start;
};

#define AGX_INDEX_LIST_START_header 0

static inline void
AGX_INDEX_LIST_START_pack(GLOBAL uint32_t * restrict cl,
                          const struct AGX_INDEX_LIST_START * restrict values)
{
   agx_genxml_validate_bounds("Index List: Start::start", values->start, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->start, 0, 31);
}

#define AGX_INDEX_LIST_START_LENGTH 4
struct agx_index_list_start_packed { uint32_t opaque[1];};
static inline bool
AGX_INDEX_LIST_START_unpack(FILE *fp, CONST uint8_t * restrict cl,
                            struct AGX_INDEX_LIST_START * restrict values)
{
   values->start = __gen_unpack_uint((CONST uint32_t *) cl, 0, 31);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_INDEX_LIST_START_print(FILE *fp, const struct AGX_INDEX_LIST_START * values, unsigned indent)
{
   fprintf(fp, "%*sStart: %u\n", indent, "", values->start);
}
#endif

struct AGX_INDEX_LIST_BUFFER_SIZE {
   uint32_t                             size;
};

#define AGX_INDEX_LIST_BUFFER_SIZE_header 0

static inline void
AGX_INDEX_LIST_BUFFER_SIZE_pack(GLOBAL uint32_t * restrict cl,
                                const struct AGX_INDEX_LIST_BUFFER_SIZE * restrict values)
{
   assert((values->size & 0x3) == 0);
   agx_genxml_validate_bounds("Index List: Buffer size::size", values->size >> 2, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->size >> 2, 0, 31);
}

#define AGX_INDEX_LIST_BUFFER_SIZE_LENGTH 4
struct agx_index_list_buffer_size_packed { uint32_t opaque[1];};
static inline bool
AGX_INDEX_LIST_BUFFER_SIZE_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                  struct AGX_INDEX_LIST_BUFFER_SIZE * restrict values)
{
   values->size = __gen_unpack_uint((CONST uint32_t *) cl, 0, 31) << 2;
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_INDEX_LIST_BUFFER_SIZE_print(FILE *fp, const struct AGX_INDEX_LIST_BUFFER_SIZE * values, unsigned indent)
{
   fprintf(fp, "%*sSize: %u\n", indent, "", values->size);
}
#endif

struct AGX_INDEX_LIST_INDIRECT_BUFFER {
   uint32_t                             address_hi;
   uint32_t                             address_lo;
};

#define AGX_INDEX_LIST_INDIRECT_BUFFER_header 0

static inline void
AGX_INDEX_LIST_INDIRECT_BUFFER_pack(GLOBAL uint32_t * restrict cl,
                                    const struct AGX_INDEX_LIST_INDIRECT_BUFFER * restrict values)
{
   agx_genxml_validate_bounds("Index List: Indirect buffer::address_hi", values->address_hi, 0x100ull);
   cl[ 0] = util_bitpack_uint(values->address_hi, 0, 7);
   agx_genxml_validate_bounds("Index List: Indirect buffer::address_lo", values->address_lo, 0x100000000ull);
   cl[ 1] = util_bitpack_uint(values->address_lo, 0, 31);
}

#define AGX_INDEX_LIST_INDIRECT_BUFFER_LENGTH 8
struct agx_index_list_indirect_buffer_packed { uint32_t opaque[2];};
static inline bool
AGX_INDEX_LIST_INDIRECT_BUFFER_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                      struct AGX_INDEX_LIST_INDIRECT_BUFFER * restrict values)
{
   values->address_hi = __gen_unpack_uint((CONST uint32_t *) cl, 0, 7);
   values->address_lo = __gen_unpack_uint((CONST uint32_t *) cl, 32, 63);
   return agx_genxml_validate_mask(fp, "Index List: Indirect buffer", cl, 0, 0xffffff00);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_INDEX_LIST_INDIRECT_BUFFER_print(FILE *fp, const struct AGX_INDEX_LIST_INDIRECT_BUFFER * values, unsigned indent)
{
   fprintf(fp, "%*sAddress hi: 0x%" PRIx32 "\n", indent, "", values->address_hi);
   fprintf(fp, "%*sAddress lo: 0x%" PRIx32 "\n", indent, "", values->address_lo);
}
#endif

struct AGX_VDM_STREAM_LINK {
   uint32_t                             target_hi;
   bool                                 with_return;
   uint32_t                             target_lo;
};

#define AGX_VDM_STREAM_LINK_header 0

static inline void
AGX_VDM_STREAM_LINK_pack(GLOBAL uint32_t * restrict cl,
                         const struct AGX_VDM_STREAM_LINK * restrict values)
{
   agx_genxml_validate_bounds("VDM Stream Link::target_hi", values->target_hi, 0x100ull);
   agx_genxml_validate_bounds("VDM Stream Link::with_return", values->with_return, 0x2ull);
   cl[ 0] = util_bitpack_uint(values->target_hi, 0, 7) |
            util_bitpack_uint(values->with_return, 28, 28) |
            util_bitpack_uint(AGX_VDM_BLOCK_TYPE_STREAM_LINK, 29, 31);
   agx_genxml_validate_bounds("VDM Stream Link::target_lo", values->target_lo, 0x100000000ull);
   cl[ 1] = util_bitpack_uint(values->target_lo, 0, 31);
}

#define AGX_VDM_STREAM_LINK_LENGTH 8
struct agx_vdm_stream_link_packed { uint32_t opaque[2];};
static inline bool
AGX_VDM_STREAM_LINK_unpack(FILE *fp, CONST uint8_t * restrict cl,
                           struct AGX_VDM_STREAM_LINK * restrict values)
{
   values->target_hi = __gen_unpack_uint((CONST uint32_t *) cl, 0, 7);
   values->with_return = __gen_unpack_uint((CONST uint32_t *) cl, 28, 28);
   values->target_lo = __gen_unpack_uint((CONST uint32_t *) cl, 32, 63);
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "VDM Stream Link", cl, 0, 0xfffff00);
   valid &= agx_genxml_validate_exact(fp, "VDM Stream Link", __gen_unpack_uint((CONST uint32_t *) cl, 29, 31), AGX_VDM_BLOCK_TYPE_STREAM_LINK);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VDM_STREAM_LINK_print(FILE *fp, const struct AGX_VDM_STREAM_LINK * values, unsigned indent)
{
   fprintf(fp, "%*sTarget hi: 0x%" PRIx32 "\n", indent, "", values->target_hi);
   fprintf(fp, "%*sWith return: %s\n", indent, "", values->with_return ? "true" : "false");
   fprintf(fp, "%*sTarget lo: 0x%" PRIx32 "\n", indent, "", values->target_lo);
}
#endif

struct AGX_VDM_TESSELLATE {
   uint32_t                             factor_buffer_hi;
   uint32_t                             unknown_0;
   bool                                 unknown_2;
   bool                                 factor_buffer_present;
   bool                                 patch_count_present;
   bool                                 instance_count_present;
   bool                                 base_patch_present;
   bool                                 base_instance_present;
   bool                                 instance_stride_present;
   bool                                 indirect_present;
   bool                                 factor_buffer_size_present;
   uint32_t                             unknown_1;
};

#define AGX_VDM_TESSELLATE_header 0

static inline void
AGX_VDM_TESSELLATE_pack(GLOBAL uint32_t * restrict cl,
                        const struct AGX_VDM_TESSELLATE * restrict values)
{
   agx_genxml_validate_bounds("VDM Tessellate::factor_buffer_hi", values->factor_buffer_hi, 0x100ull);
   agx_genxml_validate_bounds("VDM Tessellate::unknown_0", values->unknown_0, 0x1000ull);
   agx_genxml_validate_bounds("VDM Tessellate::unknown_2", values->unknown_2, 0x2ull);
   agx_genxml_validate_bounds("VDM Tessellate::factor_buffer_present", values->factor_buffer_present, 0x2ull);
   agx_genxml_validate_bounds("VDM Tessellate::patch_count_present", values->patch_count_present, 0x2ull);
   agx_genxml_validate_bounds("VDM Tessellate::instance_count_present", values->instance_count_present, 0x2ull);
   agx_genxml_validate_bounds("VDM Tessellate::base_patch_present", values->base_patch_present, 0x2ull);
   agx_genxml_validate_bounds("VDM Tessellate::base_instance_present", values->base_instance_present, 0x2ull);
   agx_genxml_validate_bounds("VDM Tessellate::instance_stride_present", values->instance_stride_present, 0x2ull);
   agx_genxml_validate_bounds("VDM Tessellate::indirect_present", values->indirect_present, 0x2ull);
   agx_genxml_validate_bounds("VDM Tessellate::factor_buffer_size_present", values->factor_buffer_size_present, 0x2ull);
   agx_genxml_validate_bounds("VDM Tessellate::unknown_1", values->unknown_1, 0x2ull);
   cl[ 0] = util_bitpack_uint(values->factor_buffer_hi, 0, 7) |
            util_bitpack_uint(values->unknown_0, 8, 19) |
            util_bitpack_uint(values->unknown_2, 20, 20) |
            util_bitpack_uint(values->factor_buffer_present, 21, 21) |
            util_bitpack_uint(values->patch_count_present, 22, 22) |
            util_bitpack_uint(values->instance_count_present, 22, 22) |
            util_bitpack_uint(values->base_patch_present, 23, 23) |
            util_bitpack_uint(values->base_instance_present, 25, 25) |
            util_bitpack_uint(values->instance_stride_present, 25, 25) |
            util_bitpack_uint(values->indirect_present, 26, 26) |
            util_bitpack_uint(values->factor_buffer_size_present, 28, 28) |
            util_bitpack_uint(values->unknown_1, 28, 28) |
            util_bitpack_uint(AGX_VDM_BLOCK_TYPE_TESSELLATE, 29, 31);
}

#define AGX_VDM_TESSELLATE_LENGTH 4
struct agx_vdm_tessellate_packed { uint32_t opaque[1];};
static inline bool
AGX_VDM_TESSELLATE_unpack(FILE *fp, CONST uint8_t * restrict cl,
                          struct AGX_VDM_TESSELLATE * restrict values)
{
   values->factor_buffer_hi = __gen_unpack_uint((CONST uint32_t *) cl, 0, 7);
   values->unknown_0 = __gen_unpack_uint((CONST uint32_t *) cl, 8, 19);
   values->unknown_2 = __gen_unpack_uint((CONST uint32_t *) cl, 20, 20);
   values->factor_buffer_present = __gen_unpack_uint((CONST uint32_t *) cl, 21, 21);
   values->patch_count_present = __gen_unpack_uint((CONST uint32_t *) cl, 22, 22);
   values->instance_count_present = __gen_unpack_uint((CONST uint32_t *) cl, 22, 22);
   values->base_patch_present = __gen_unpack_uint((CONST uint32_t *) cl, 23, 23);
   values->base_instance_present = __gen_unpack_uint((CONST uint32_t *) cl, 25, 25);
   values->instance_stride_present = __gen_unpack_uint((CONST uint32_t *) cl, 25, 25);
   values->indirect_present = __gen_unpack_uint((CONST uint32_t *) cl, 26, 26);
   values->factor_buffer_size_present = __gen_unpack_uint((CONST uint32_t *) cl, 28, 28);
   values->unknown_1 = __gen_unpack_uint((CONST uint32_t *) cl, 28, 28);
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "VDM Tessellate", cl, 0, 0x9000000);
   valid &= agx_genxml_validate_exact(fp, "VDM Tessellate", __gen_unpack_uint((CONST uint32_t *) cl, 29, 31), AGX_VDM_BLOCK_TYPE_TESSELLATE);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VDM_TESSELLATE_print(FILE *fp, const struct AGX_VDM_TESSELLATE * values, unsigned indent)
{
   fprintf(fp, "%*sFactor buffer hi: 0x%" PRIx32 "\n", indent, "", values->factor_buffer_hi);
   fprintf(fp, "%*sUnknown 0: 0x%" PRIx32 "\n", indent, "", values->unknown_0);
   fprintf(fp, "%*sUnknown 2: %s\n", indent, "", values->unknown_2 ? "true" : "false");
   fprintf(fp, "%*sFactor buffer present: %s\n", indent, "", values->factor_buffer_present ? "true" : "false");
   fprintf(fp, "%*sPatch count present: %s\n", indent, "", values->patch_count_present ? "true" : "false");
   fprintf(fp, "%*sInstance count present: %s\n", indent, "", values->instance_count_present ? "true" : "false");
   fprintf(fp, "%*sBase patch present: %s\n", indent, "", values->base_patch_present ? "true" : "false");
   fprintf(fp, "%*sBase instance present: %s\n", indent, "", values->base_instance_present ? "true" : "false");
   fprintf(fp, "%*sInstance stride present: %s\n", indent, "", values->instance_stride_present ? "true" : "false");
   fprintf(fp, "%*sIndirect present: %s\n", indent, "", values->indirect_present ? "true" : "false");
   fprintf(fp, "%*sFactor buffer size present: %s\n", indent, "", values->factor_buffer_size_present ? "true" : "false");
   fprintf(fp, "%*sUnknown 1: 0x%" PRIx32 "\n", indent, "", values->unknown_1);
}
#endif

struct AGX_VDM_TESSELLATE_FACTOR_BUFFER {
   uint32_t                             factor_buffer_lo;
};

#define AGX_VDM_TESSELLATE_FACTOR_BUFFER_header 0

static inline void
AGX_VDM_TESSELLATE_FACTOR_BUFFER_pack(GLOBAL uint32_t * restrict cl,
                                      const struct AGX_VDM_TESSELLATE_FACTOR_BUFFER * restrict values)
{
   agx_genxml_validate_bounds("VDM Tessellate: Factor buffer::factor_buffer_lo", values->factor_buffer_lo, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->factor_buffer_lo, 0, 31);
}

#define AGX_VDM_TESSELLATE_FACTOR_BUFFER_LENGTH 4
struct agx_vdm_tessellate_factor_buffer_packed { uint32_t opaque[1];};
static inline bool
AGX_VDM_TESSELLATE_FACTOR_BUFFER_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                        struct AGX_VDM_TESSELLATE_FACTOR_BUFFER * restrict values)
{
   values->factor_buffer_lo = __gen_unpack_uint((CONST uint32_t *) cl, 0, 31);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VDM_TESSELLATE_FACTOR_BUFFER_print(FILE *fp, const struct AGX_VDM_TESSELLATE_FACTOR_BUFFER * values, unsigned indent)
{
   fprintf(fp, "%*sFactor buffer lo: 0x%" PRIx32 "\n", indent, "", values->factor_buffer_lo);
}
#endif

struct AGX_VDM_TESSELLATE_PATCH_COUNT {
   uint32_t                             patch_count;
};

#define AGX_VDM_TESSELLATE_PATCH_COUNT_header 0

static inline void
AGX_VDM_TESSELLATE_PATCH_COUNT_pack(GLOBAL uint32_t * restrict cl,
                                    const struct AGX_VDM_TESSELLATE_PATCH_COUNT * restrict values)
{
   agx_genxml_validate_bounds("VDM Tessellate: Patch count::patch_count", values->patch_count, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->patch_count, 0, 31);
}

#define AGX_VDM_TESSELLATE_PATCH_COUNT_LENGTH 4
struct agx_vdm_tessellate_patch_count_packed { uint32_t opaque[1];};
static inline bool
AGX_VDM_TESSELLATE_PATCH_COUNT_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                      struct AGX_VDM_TESSELLATE_PATCH_COUNT * restrict values)
{
   values->patch_count = __gen_unpack_uint((CONST uint32_t *) cl, 0, 31);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VDM_TESSELLATE_PATCH_COUNT_print(FILE *fp, const struct AGX_VDM_TESSELLATE_PATCH_COUNT * values, unsigned indent)
{
   fprintf(fp, "%*sPatch count: %u\n", indent, "", values->patch_count);
}
#endif

struct AGX_VDM_TESSELLATE_INSTANCE_COUNT {
   uint32_t                             instance_count;
};

#define AGX_VDM_TESSELLATE_INSTANCE_COUNT_header 0

static inline void
AGX_VDM_TESSELLATE_INSTANCE_COUNT_pack(GLOBAL uint32_t * restrict cl,
                                       const struct AGX_VDM_TESSELLATE_INSTANCE_COUNT * restrict values)
{
   agx_genxml_validate_bounds("VDM Tessellate: Instance count::instance_count", values->instance_count, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->instance_count, 0, 31);
}

#define AGX_VDM_TESSELLATE_INSTANCE_COUNT_LENGTH 4
struct agx_vdm_tessellate_instance_count_packed { uint32_t opaque[1];};
static inline bool
AGX_VDM_TESSELLATE_INSTANCE_COUNT_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                         struct AGX_VDM_TESSELLATE_INSTANCE_COUNT * restrict values)
{
   values->instance_count = __gen_unpack_uint((CONST uint32_t *) cl, 0, 31);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VDM_TESSELLATE_INSTANCE_COUNT_print(FILE *fp, const struct AGX_VDM_TESSELLATE_INSTANCE_COUNT * values, unsigned indent)
{
   fprintf(fp, "%*sInstance count: %u\n", indent, "", values->instance_count);
}
#endif

struct AGX_VDM_TESSELLATE_BASE_PATCH {
   uint32_t                             base_patch;
};

#define AGX_VDM_TESSELLATE_BASE_PATCH_header 0

static inline void
AGX_VDM_TESSELLATE_BASE_PATCH_pack(GLOBAL uint32_t * restrict cl,
                                   const struct AGX_VDM_TESSELLATE_BASE_PATCH * restrict values)
{
   agx_genxml_validate_bounds("VDM Tessellate: Base patch::base_patch", values->base_patch, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->base_patch, 0, 31);
}

#define AGX_VDM_TESSELLATE_BASE_PATCH_LENGTH 4
struct agx_vdm_tessellate_base_patch_packed { uint32_t opaque[1];};
static inline bool
AGX_VDM_TESSELLATE_BASE_PATCH_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                     struct AGX_VDM_TESSELLATE_BASE_PATCH * restrict values)
{
   values->base_patch = __gen_unpack_uint((CONST uint32_t *) cl, 0, 31);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VDM_TESSELLATE_BASE_PATCH_print(FILE *fp, const struct AGX_VDM_TESSELLATE_BASE_PATCH * values, unsigned indent)
{
   fprintf(fp, "%*sBase patch: %u\n", indent, "", values->base_patch);
}
#endif

struct AGX_VDM_TESSELLATE_BASE_INSTANCE {
   uint32_t                             base_instance;
};

#define AGX_VDM_TESSELLATE_BASE_INSTANCE_header 0

static inline void
AGX_VDM_TESSELLATE_BASE_INSTANCE_pack(GLOBAL uint32_t * restrict cl,
                                      const struct AGX_VDM_TESSELLATE_BASE_INSTANCE * restrict values)
{
   agx_genxml_validate_bounds("VDM Tessellate: Base instance::base_instance", values->base_instance, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->base_instance, 0, 31);
}

#define AGX_VDM_TESSELLATE_BASE_INSTANCE_LENGTH 4
struct agx_vdm_tessellate_base_instance_packed { uint32_t opaque[1];};
static inline bool
AGX_VDM_TESSELLATE_BASE_INSTANCE_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                        struct AGX_VDM_TESSELLATE_BASE_INSTANCE * restrict values)
{
   values->base_instance = __gen_unpack_uint((CONST uint32_t *) cl, 0, 31);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VDM_TESSELLATE_BASE_INSTANCE_print(FILE *fp, const struct AGX_VDM_TESSELLATE_BASE_INSTANCE * values, unsigned indent)
{
   fprintf(fp, "%*sBase instance: %u\n", indent, "", values->base_instance);
}
#endif

struct AGX_VDM_TESSELLATE_INSTANCE_STRIDE {
   uint32_t                             instance_stride;
};

#define AGX_VDM_TESSELLATE_INSTANCE_STRIDE_header 0

static inline void
AGX_VDM_TESSELLATE_INSTANCE_STRIDE_pack(GLOBAL uint32_t * restrict cl,
                                        const struct AGX_VDM_TESSELLATE_INSTANCE_STRIDE * restrict values)
{
   agx_genxml_validate_bounds("VDM Tessellate: Instance stride::instance_stride", values->instance_stride, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->instance_stride, 0, 31);
}

#define AGX_VDM_TESSELLATE_INSTANCE_STRIDE_LENGTH 4
struct agx_vdm_tessellate_instance_stride_packed { uint32_t opaque[1];};
static inline bool
AGX_VDM_TESSELLATE_INSTANCE_STRIDE_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                          struct AGX_VDM_TESSELLATE_INSTANCE_STRIDE * restrict values)
{
   values->instance_stride = __gen_unpack_uint((CONST uint32_t *) cl, 0, 31);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VDM_TESSELLATE_INSTANCE_STRIDE_print(FILE *fp, const struct AGX_VDM_TESSELLATE_INSTANCE_STRIDE * values, unsigned indent)
{
   fprintf(fp, "%*sInstance stride: %u\n", indent, "", values->instance_stride);
}
#endif

struct AGX_VDM_TESSELLATE_INDIRECT {
   uint32_t                             address_hi;
   uint32_t                             address_lo;
   uint32_t                             size;
};

#define AGX_VDM_TESSELLATE_INDIRECT_header 0

static inline void
AGX_VDM_TESSELLATE_INDIRECT_pack(GLOBAL uint32_t * restrict cl,
                                 const struct AGX_VDM_TESSELLATE_INDIRECT * restrict values)
{
   agx_genxml_validate_bounds("VDM Tessellate: Indirect::address_hi", values->address_hi, 0x100ull);
   cl[ 0] = util_bitpack_uint(values->address_hi, 0, 7);
   agx_genxml_validate_bounds("VDM Tessellate: Indirect::address_lo", values->address_lo, 0x100000000ull);
   cl[ 1] = util_bitpack_uint(values->address_lo, 0, 31);
   agx_genxml_validate_bounds("VDM Tessellate: Indirect::size", values->size, 0x100000000ull);
   cl[ 2] = util_bitpack_uint(values->size, 0, 31);
}

#define AGX_VDM_TESSELLATE_INDIRECT_LENGTH 12
struct agx_vdm_tessellate_indirect_packed { uint32_t opaque[3];};
static inline bool
AGX_VDM_TESSELLATE_INDIRECT_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                   struct AGX_VDM_TESSELLATE_INDIRECT * restrict values)
{
   values->address_hi = __gen_unpack_uint((CONST uint32_t *) cl, 0, 7);
   values->address_lo = __gen_unpack_uint((CONST uint32_t *) cl, 32, 63);
   values->size = __gen_unpack_uint((CONST uint32_t *) cl, 64, 95);
   return agx_genxml_validate_mask(fp, "VDM Tessellate: Indirect", cl, 0, 0xffffff00);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VDM_TESSELLATE_INDIRECT_print(FILE *fp, const struct AGX_VDM_TESSELLATE_INDIRECT * values, unsigned indent)
{
   fprintf(fp, "%*sAddress hi: 0x%" PRIx32 "\n", indent, "", values->address_hi);
   fprintf(fp, "%*sAddress lo: 0x%" PRIx32 "\n", indent, "", values->address_lo);
   fprintf(fp, "%*sSize: %u\n", indent, "", values->size);
}
#endif

struct AGX_VDM_TESSELLATE_FACTOR_BUFFER_SIZE {
   uint32_t                             size_words;
};

#define AGX_VDM_TESSELLATE_FACTOR_BUFFER_SIZE_header\
   .size_words = 1

static inline void
AGX_VDM_TESSELLATE_FACTOR_BUFFER_SIZE_pack(GLOBAL uint32_t * restrict cl,
                                           const struct AGX_VDM_TESSELLATE_FACTOR_BUFFER_SIZE * restrict values)
{
   assert(values->size_words >= 1);
   agx_genxml_validate_bounds("VDM Tessellate: Factor buffer size::size_words", values->size_words - 1, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->size_words - 1, 0, 31);
}

#define AGX_VDM_TESSELLATE_FACTOR_BUFFER_SIZE_LENGTH 4
struct agx_vdm_tessellate_factor_buffer_size_packed { uint32_t opaque[1];};
static inline bool
AGX_VDM_TESSELLATE_FACTOR_BUFFER_SIZE_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                             struct AGX_VDM_TESSELLATE_FACTOR_BUFFER_SIZE * restrict values)
{
   values->size_words = __gen_unpack_uint((CONST uint32_t *) cl, 0, 31) + 1;
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VDM_TESSELLATE_FACTOR_BUFFER_SIZE_print(FILE *fp, const struct AGX_VDM_TESSELLATE_FACTOR_BUFFER_SIZE * values, unsigned indent)
{
   fprintf(fp, "%*sSize (words): %u\n", indent, "", values->size_words);
}
#endif

struct AGX_VDM_STREAM_TERMINATE {
   int dummy;
};

#define AGX_VDM_STREAM_TERMINATE_header 0

static inline void
AGX_VDM_STREAM_TERMINATE_pack(GLOBAL uint32_t * restrict cl,
                              const struct AGX_VDM_STREAM_TERMINATE * restrict values)
{
   cl[ 0] = util_bitpack_uint(AGX_VDM_BLOCK_TYPE_STREAM_TERMINATE, 29, 31);
   cl[ 1] = 0;
   cl[ 2] = 0;
   cl[ 3] = 0;
   cl[ 4] = 0;
   cl[ 5] = 0;
   cl[ 6] = 0;
   cl[ 7] = 0;
}

#define AGX_VDM_STREAM_TERMINATE_LENGTH 32
struct agx_vdm_stream_terminate_packed { uint32_t opaque[8];};
static inline bool
AGX_VDM_STREAM_TERMINATE_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                struct AGX_VDM_STREAM_TERMINATE * restrict values)
{
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "VDM Stream Terminate", cl, 0, 0x1fffffff);
   valid &= agx_genxml_validate_mask(fp, "VDM Stream Terminate", cl, 1, 0xffffffff);
   valid &= agx_genxml_validate_mask(fp, "VDM Stream Terminate", cl, 2, 0xffffffff);
   valid &= agx_genxml_validate_mask(fp, "VDM Stream Terminate", cl, 3, 0xffffffff);
   valid &= agx_genxml_validate_mask(fp, "VDM Stream Terminate", cl, 4, 0xffffffff);
   valid &= agx_genxml_validate_mask(fp, "VDM Stream Terminate", cl, 5, 0xffffffff);
   valid &= agx_genxml_validate_mask(fp, "VDM Stream Terminate", cl, 6, 0xffffffff);
   valid &= agx_genxml_validate_mask(fp, "VDM Stream Terminate", cl, 7, 0xffffffff);
   valid &= agx_genxml_validate_exact(fp, "VDM Stream Terminate", __gen_unpack_uint((CONST uint32_t *) cl, 29, 31), AGX_VDM_BLOCK_TYPE_STREAM_TERMINATE);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_VDM_STREAM_TERMINATE_print(FILE *fp, const struct AGX_VDM_STREAM_TERMINATE * values, unsigned indent)
{
}
#endif

enum agx_cdm_block_type {
   AGX_CDM_BLOCK_TYPE_LAUNCH = 0,
   AGX_CDM_BLOCK_TYPE_STREAM_LINK = 1,
   AGX_CDM_BLOCK_TYPE_STREAM_TERMINATE = 2,
   AGX_CDM_BLOCK_TYPE_BARRIER = 3,
   AGX_CDM_BLOCK_TYPE_STREAM_RETURN = 4,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_cdm_block_type_as_str(enum agx_cdm_block_type imm)
{
    switch (imm) {
    case AGX_CDM_BLOCK_TYPE_LAUNCH: return "Launch";
    case AGX_CDM_BLOCK_TYPE_STREAM_LINK: return "Stream Link";
    case AGX_CDM_BLOCK_TYPE_STREAM_TERMINATE: return "Stream Terminate";
    case AGX_CDM_BLOCK_TYPE_BARRIER: return "Barrier";
    case AGX_CDM_BLOCK_TYPE_STREAM_RETURN: return "Stream Return";
    default: return NULL;
    }
}
#endif

enum agx_cdm_mode {
   AGX_CDM_MODE_DIRECT = 0,
   AGX_CDM_MODE_INDIRECT_GLOBAL = 1,
   AGX_CDM_MODE_INDIRECT_LOCAL = 2,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_cdm_mode_as_str(enum agx_cdm_mode imm)
{
    switch (imm) {
    case AGX_CDM_MODE_DIRECT: return "Direct";
    case AGX_CDM_MODE_INDIRECT_GLOBAL: return "Indirect global";
    case AGX_CDM_MODE_INDIRECT_LOCAL: return "Indirect local";
    default: return NULL;
    }
}
#endif

struct AGX_CDM_LAUNCH_WORD_0 {
   uint32_t                             uniform_register_count;
   uint32_t                             texture_state_register_count;
   enum agx_sampler_states              sampler_state_register_count;
   uint32_t                             preshader_register_count;
   enum agx_cdm_mode                    mode;
};

#define AGX_CDM_LAUNCH_WORD_0_header            \
   .uniform_register_count = 512,  \
   .texture_state_register_count = 256,  \
   .preshader_register_count = 256

static inline void
AGX_CDM_LAUNCH_WORD_0_pack(GLOBAL uint32_t * restrict cl,
                           const struct AGX_CDM_LAUNCH_WORD_0 * restrict values)
{
   agx_genxml_validate_bounds("CDM Launch Word 0::uniform_register_count", __gen_to_groups(values->uniform_register_count, 64, 3), 0x8ull);
   agx_genxml_validate_bounds("CDM Launch Word 0::texture_state_register_count", __gen_to_groups(values->texture_state_register_count, 8, 5), 0x20ull);
   agx_genxml_validate_bounds("CDM Launch Word 0::sampler_state_register_count", values->sampler_state_register_count, 0x8ull);
   agx_genxml_validate_bounds("CDM Launch Word 0::preshader_register_count", __gen_to_groups(values->preshader_register_count, 16, 4), 0x10ull);
   agx_genxml_validate_bounds("CDM Launch Word 0::mode", values->mode, 0x4ull);
   cl[ 0] = util_bitpack_uint(__gen_to_groups(values->uniform_register_count, 64, 3), 1, 3) |
            util_bitpack_uint(__gen_to_groups(values->texture_state_register_count, 8, 5), 4, 8) |
            util_bitpack_uint(values->sampler_state_register_count, 9, 11) |
            util_bitpack_uint(__gen_to_groups(values->preshader_register_count, 16, 4), 12, 15) |
            util_bitpack_uint(values->mode, 27, 28) |
            util_bitpack_uint(AGX_CDM_BLOCK_TYPE_LAUNCH, 29, 31);
}

#define AGX_CDM_LAUNCH_WORD_0_LENGTH 4
struct agx_cdm_launch_word_0_packed { uint32_t opaque[1];};
static inline bool
AGX_CDM_LAUNCH_WORD_0_unpack(FILE *fp, CONST uint8_t * restrict cl,
                             struct AGX_CDM_LAUNCH_WORD_0 * restrict values)
{
   values->uniform_register_count = __gen_from_groups(__gen_unpack_uint((CONST uint32_t *) cl, 1, 3), 64, 3);
   values->texture_state_register_count = __gen_from_groups(__gen_unpack_uint((CONST uint32_t *) cl, 4, 8), 8, 5);
   values->sampler_state_register_count = (enum agx_sampler_states) __gen_unpack_uint((CONST uint32_t *) cl, 9, 11);
   values->preshader_register_count = __gen_from_groups(__gen_unpack_uint((CONST uint32_t *) cl, 12, 15), 16, 4);
   values->mode = (enum agx_cdm_mode) __gen_unpack_uint((CONST uint32_t *) cl, 27, 28);
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "CDM Launch Word 0", cl, 0, 0x7ff0001);
   valid &= agx_genxml_validate_exact(fp, "CDM Launch Word 0", __gen_unpack_uint((CONST uint32_t *) cl, 29, 31), AGX_CDM_BLOCK_TYPE_LAUNCH);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_CDM_LAUNCH_WORD_0_print(FILE *fp, const struct AGX_CDM_LAUNCH_WORD_0 * values, unsigned indent)
{
   fprintf(fp, "%*sUniform register count: %u\n", indent, "", values->uniform_register_count);
   fprintf(fp, "%*sTexture state register count: %u\n", indent, "", values->texture_state_register_count);
   if (agx_sampler_states_as_str(values->sampler_state_register_count))
     fprintf(fp, "%*sSampler state register count: %s\n", indent, "", agx_sampler_states_as_str(values->sampler_state_register_count));
   else
     fprintf(fp, "%*sSampler state register count: unknown %X (XXX)\n", indent, "", values->sampler_state_register_count);
   fprintf(fp, "%*sPreshader register count: %u\n", indent, "", values->preshader_register_count);
   if (agx_cdm_mode_as_str(values->mode))
     fprintf(fp, "%*sMode: %s\n", indent, "", agx_cdm_mode_as_str(values->mode));
   else
     fprintf(fp, "%*sMode: unknown %X (XXX)\n", indent, "", values->mode);
}
#endif

struct AGX_CDM_LAUNCH_WORD_1 {
   uint64_t                             pipeline;
};

#define AGX_CDM_LAUNCH_WORD_1_header 0

static inline void
AGX_CDM_LAUNCH_WORD_1_pack(GLOBAL uint32_t * restrict cl,
                           const struct AGX_CDM_LAUNCH_WORD_1 * restrict values)
{
   assert((values->pipeline & 0x3f) == 0);
   agx_genxml_validate_bounds("CDM Launch Word 1::pipeline", values->pipeline, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->pipeline, 0, 31);
}

#define AGX_CDM_LAUNCH_WORD_1_LENGTH 4
struct agx_cdm_launch_word_1_packed { uint32_t opaque[1];};
static inline bool
AGX_CDM_LAUNCH_WORD_1_unpack(FILE *fp, CONST uint8_t * restrict cl,
                             struct AGX_CDM_LAUNCH_WORD_1 * restrict values)
{
   values->pipeline = __gen_unpack_uint((CONST uint32_t *) cl, 6, 31) << 6;
   return agx_genxml_validate_mask(fp, "CDM Launch Word 1", cl, 0, 0x3f);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_CDM_LAUNCH_WORD_1_print(FILE *fp, const struct AGX_CDM_LAUNCH_WORD_1 * values, unsigned indent)
{
   fprintf(fp, "%*sPipeline: 0x%" PRIx64 "\n", indent, "", values->pipeline);
}
#endif

struct AGX_CDM_UNK_G14X {
   int dummy;
};

#define AGX_CDM_UNK_G14X_header 0

static inline void
AGX_CDM_UNK_G14X_pack(GLOBAL uint32_t * restrict cl,
                      const struct AGX_CDM_UNK_G14X * restrict values)
{
   cl[ 0] = 0;
   cl[ 1] = util_bitpack_uint(true, 30, 30);
}

#define AGX_CDM_UNK_G14X_LENGTH 8
struct agx_cdm_unk_g14x_packed { uint32_t opaque[2];};
static inline bool
AGX_CDM_UNK_G14X_unpack(FILE *fp, CONST uint8_t * restrict cl,
                        struct AGX_CDM_UNK_G14X * restrict values)
{
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "CDM Unk G14X", cl, 0, 0xffffffff);
   valid &= agx_genxml_validate_mask(fp, "CDM Unk G14X", cl, 1, 0xbfffffff);
   valid &= agx_genxml_validate_exact(fp, "CDM Unk G14X", __gen_unpack_uint((CONST uint32_t *) cl, 62, 62), true);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_CDM_UNK_G14X_print(FILE *fp, const struct AGX_CDM_UNK_G14X * values, unsigned indent)
{
}
#endif

struct AGX_CDM_INDIRECT {
   uint32_t                             address_hi;
   uint32_t                             address_lo;
};

#define AGX_CDM_INDIRECT_header 0

static inline void
AGX_CDM_INDIRECT_pack(GLOBAL uint32_t * restrict cl,
                      const struct AGX_CDM_INDIRECT * restrict values)
{
   assert((values->address_lo & 0x3) == 0);
   agx_genxml_validate_bounds("CDM Indirect::address_hi", values->address_hi, 0x100ull);
   cl[ 0] = util_bitpack_uint(values->address_hi, 0, 7);
   agx_genxml_validate_bounds("CDM Indirect::address_lo", values->address_lo, 0x100000000ull);
   cl[ 1] = util_bitpack_uint(values->address_lo, 0, 31);
}

#define AGX_CDM_INDIRECT_LENGTH 8
struct agx_cdm_indirect_packed { uint32_t opaque[2];};
static inline bool
AGX_CDM_INDIRECT_unpack(FILE *fp, CONST uint8_t * restrict cl,
                        struct AGX_CDM_INDIRECT * restrict values)
{
   values->address_hi = __gen_unpack_uint((CONST uint32_t *) cl, 0, 7);
   values->address_lo = __gen_unpack_uint((CONST uint32_t *) cl, 34, 63) << 2;
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "CDM Indirect", cl, 0, 0xffffff00);
   valid &= agx_genxml_validate_mask(fp, "CDM Indirect", cl, 1, 0x3);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_CDM_INDIRECT_print(FILE *fp, const struct AGX_CDM_INDIRECT * values, unsigned indent)
{
   fprintf(fp, "%*sAddress hi: 0x%" PRIx32 "\n", indent, "", values->address_hi);
   fprintf(fp, "%*sAddress lo: 0x%" PRIx32 "\n", indent, "", values->address_lo);
}
#endif

struct AGX_CDM_GLOBAL_SIZE {
   uint32_t                             x;
   uint32_t                             y;
   uint32_t                             z;
};

#define AGX_CDM_GLOBAL_SIZE_header 0

static inline void
AGX_CDM_GLOBAL_SIZE_pack(GLOBAL uint32_t * restrict cl,
                         const struct AGX_CDM_GLOBAL_SIZE * restrict values)
{
   agx_genxml_validate_bounds("CDM Global size::x", values->x, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->x, 0, 31);
   agx_genxml_validate_bounds("CDM Global size::y", values->y, 0x100000000ull);
   cl[ 1] = util_bitpack_uint(values->y, 0, 31);
   agx_genxml_validate_bounds("CDM Global size::z", values->z, 0x100000000ull);
   cl[ 2] = util_bitpack_uint(values->z, 0, 31);
}

#define AGX_CDM_GLOBAL_SIZE_LENGTH 12
struct agx_cdm_global_size_packed { uint32_t opaque[3];};
static inline bool
AGX_CDM_GLOBAL_SIZE_unpack(FILE *fp, CONST uint8_t * restrict cl,
                           struct AGX_CDM_GLOBAL_SIZE * restrict values)
{
   values->x = __gen_unpack_uint((CONST uint32_t *) cl, 0, 31);
   values->y = __gen_unpack_uint((CONST uint32_t *) cl, 32, 63);
   values->z = __gen_unpack_uint((CONST uint32_t *) cl, 64, 95);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_CDM_GLOBAL_SIZE_print(FILE *fp, const struct AGX_CDM_GLOBAL_SIZE * values, unsigned indent)
{
   fprintf(fp, "%*sX: %u\n", indent, "", values->x);
   fprintf(fp, "%*sY: %u\n", indent, "", values->y);
   fprintf(fp, "%*sZ: %u\n", indent, "", values->z);
}
#endif

struct AGX_CDM_LOCAL_SIZE {
   uint32_t                             x;
   uint32_t                             y;
   uint32_t                             z;
};

#define AGX_CDM_LOCAL_SIZE_header 0

static inline void
AGX_CDM_LOCAL_SIZE_pack(GLOBAL uint32_t * restrict cl,
                        const struct AGX_CDM_LOCAL_SIZE * restrict values)
{
   agx_genxml_validate_bounds("CDM Local size::x", values->x, 0x100000000ull);
   cl[ 0] = util_bitpack_uint(values->x, 0, 31);
   agx_genxml_validate_bounds("CDM Local size::y", values->y, 0x100000000ull);
   cl[ 1] = util_bitpack_uint(values->y, 0, 31);
   agx_genxml_validate_bounds("CDM Local size::z", values->z, 0x100000000ull);
   cl[ 2] = util_bitpack_uint(values->z, 0, 31);
}

#define AGX_CDM_LOCAL_SIZE_LENGTH 12
struct agx_cdm_local_size_packed { uint32_t opaque[3];};
static inline bool
AGX_CDM_LOCAL_SIZE_unpack(FILE *fp, CONST uint8_t * restrict cl,
                          struct AGX_CDM_LOCAL_SIZE * restrict values)
{
   values->x = __gen_unpack_uint((CONST uint32_t *) cl, 0, 31);
   values->y = __gen_unpack_uint((CONST uint32_t *) cl, 32, 63);
   values->z = __gen_unpack_uint((CONST uint32_t *) cl, 64, 95);
   return true;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_CDM_LOCAL_SIZE_print(FILE *fp, const struct AGX_CDM_LOCAL_SIZE * values, unsigned indent)
{
   fprintf(fp, "%*sX: %u\n", indent, "", values->x);
   fprintf(fp, "%*sY: %u\n", indent, "", values->y);
   fprintf(fp, "%*sZ: %u\n", indent, "", values->z);
}
#endif

struct AGX_CDM_BARRIER {
   bool                                 unk_0;
   bool                                 unk_1;
   bool                                 unk_2;
   bool                                 usc_cache_inval;
   bool                                 unk_4;
   bool                                 unk_5;
   bool                                 unk_6;
   bool                                 unk_7;
   bool                                 unk_8;
   bool                                 unk_9;
   bool                                 unk_10;
   bool                                 unk_11;
   bool                                 unk_12;
   bool                                 unk_13;
   bool                                 unk_14;
   bool                                 unk_15;
   bool                                 unk_16;
   bool                                 unk_17;
   bool                                 unk_18;
   bool                                 unk_19;
   bool                                 unk_20;
   bool                                 unk_24;
   bool                                 unk_26;
   bool                                 returns;
};

#define AGX_CDM_BARRIER_header 0

static inline void
AGX_CDM_BARRIER_pack(GLOBAL uint32_t * restrict cl,
                     const struct AGX_CDM_BARRIER * restrict values)
{
   agx_genxml_validate_bounds("CDM Barrier::unk_0", values->unk_0, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_1", values->unk_1, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_2", values->unk_2, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::usc_cache_inval", values->usc_cache_inval, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_4", values->unk_4, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_5", values->unk_5, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_6", values->unk_6, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_7", values->unk_7, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_8", values->unk_8, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_9", values->unk_9, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_10", values->unk_10, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_11", values->unk_11, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_12", values->unk_12, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_13", values->unk_13, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_14", values->unk_14, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_15", values->unk_15, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_16", values->unk_16, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_17", values->unk_17, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_18", values->unk_18, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_19", values->unk_19, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_20", values->unk_20, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_24", values->unk_24, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::unk_26", values->unk_26, 0x2ull);
   agx_genxml_validate_bounds("CDM Barrier::returns", values->returns, 0x2ull);
   cl[ 0] = util_bitpack_uint(values->unk_0, 0, 0) |
            util_bitpack_uint(values->unk_1, 1, 1) |
            util_bitpack_uint(values->unk_2, 2, 2) |
            util_bitpack_uint(values->usc_cache_inval, 3, 3) |
            util_bitpack_uint(values->unk_4, 4, 4) |
            util_bitpack_uint(values->unk_5, 5, 5) |
            util_bitpack_uint(values->unk_6, 6, 6) |
            util_bitpack_uint(values->unk_7, 7, 7) |
            util_bitpack_uint(values->unk_8, 8, 8) |
            util_bitpack_uint(values->unk_9, 9, 9) |
            util_bitpack_uint(values->unk_10, 10, 10) |
            util_bitpack_uint(values->unk_11, 11, 11) |
            util_bitpack_uint(values->unk_12, 12, 12) |
            util_bitpack_uint(values->unk_13, 13, 13) |
            util_bitpack_uint(values->unk_14, 14, 14) |
            util_bitpack_uint(values->unk_15, 15, 15) |
            util_bitpack_uint(values->unk_16, 16, 16) |
            util_bitpack_uint(values->unk_17, 17, 17) |
            util_bitpack_uint(values->unk_18, 18, 18) |
            util_bitpack_uint(values->unk_19, 19, 19) |
            util_bitpack_uint(values->unk_20, 20, 20) |
            util_bitpack_uint(values->unk_24, 24, 24) |
            util_bitpack_uint(values->unk_26, 26, 26) |
            util_bitpack_uint(values->returns, 27, 27) |
            util_bitpack_uint(AGX_CDM_BLOCK_TYPE_BARRIER, 29, 31);
}

#define AGX_CDM_BARRIER_LENGTH 4
struct agx_cdm_barrier_packed { uint32_t opaque[1];};
static inline bool
AGX_CDM_BARRIER_unpack(FILE *fp, CONST uint8_t * restrict cl,
                       struct AGX_CDM_BARRIER * restrict values)
{
   values->unk_0 = __gen_unpack_uint((CONST uint32_t *) cl, 0, 0);
   values->unk_1 = __gen_unpack_uint((CONST uint32_t *) cl, 1, 1);
   values->unk_2 = __gen_unpack_uint((CONST uint32_t *) cl, 2, 2);
   values->usc_cache_inval = __gen_unpack_uint((CONST uint32_t *) cl, 3, 3);
   values->unk_4 = __gen_unpack_uint((CONST uint32_t *) cl, 4, 4);
   values->unk_5 = __gen_unpack_uint((CONST uint32_t *) cl, 5, 5);
   values->unk_6 = __gen_unpack_uint((CONST uint32_t *) cl, 6, 6);
   values->unk_7 = __gen_unpack_uint((CONST uint32_t *) cl, 7, 7);
   values->unk_8 = __gen_unpack_uint((CONST uint32_t *) cl, 8, 8);
   values->unk_9 = __gen_unpack_uint((CONST uint32_t *) cl, 9, 9);
   values->unk_10 = __gen_unpack_uint((CONST uint32_t *) cl, 10, 10);
   values->unk_11 = __gen_unpack_uint((CONST uint32_t *) cl, 11, 11);
   values->unk_12 = __gen_unpack_uint((CONST uint32_t *) cl, 12, 12);
   values->unk_13 = __gen_unpack_uint((CONST uint32_t *) cl, 13, 13);
   values->unk_14 = __gen_unpack_uint((CONST uint32_t *) cl, 14, 14);
   values->unk_15 = __gen_unpack_uint((CONST uint32_t *) cl, 15, 15);
   values->unk_16 = __gen_unpack_uint((CONST uint32_t *) cl, 16, 16);
   values->unk_17 = __gen_unpack_uint((CONST uint32_t *) cl, 17, 17);
   values->unk_18 = __gen_unpack_uint((CONST uint32_t *) cl, 18, 18);
   values->unk_19 = __gen_unpack_uint((CONST uint32_t *) cl, 19, 19);
   values->unk_20 = __gen_unpack_uint((CONST uint32_t *) cl, 20, 20);
   values->unk_24 = __gen_unpack_uint((CONST uint32_t *) cl, 24, 24);
   values->unk_26 = __gen_unpack_uint((CONST uint32_t *) cl, 26, 26);
   values->returns = __gen_unpack_uint((CONST uint32_t *) cl, 27, 27);
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "CDM Barrier", cl, 0, 0x12e00000);
   valid &= agx_genxml_validate_exact(fp, "CDM Barrier", __gen_unpack_uint((CONST uint32_t *) cl, 29, 31), AGX_CDM_BLOCK_TYPE_BARRIER);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_CDM_BARRIER_print(FILE *fp, const struct AGX_CDM_BARRIER * values, unsigned indent)
{
   fprintf(fp, "%*sUnk 0: %s\n", indent, "", values->unk_0 ? "true" : "false");
   fprintf(fp, "%*sUnk 1: %s\n", indent, "", values->unk_1 ? "true" : "false");
   fprintf(fp, "%*sUnk 2: %s\n", indent, "", values->unk_2 ? "true" : "false");
   fprintf(fp, "%*sUSC cache inval: %s\n", indent, "", values->usc_cache_inval ? "true" : "false");
   fprintf(fp, "%*sUnk 4: %s\n", indent, "", values->unk_4 ? "true" : "false");
   fprintf(fp, "%*sUnk 5: %s\n", indent, "", values->unk_5 ? "true" : "false");
   fprintf(fp, "%*sUnk 6: %s\n", indent, "", values->unk_6 ? "true" : "false");
   fprintf(fp, "%*sUnk 7: %s\n", indent, "", values->unk_7 ? "true" : "false");
   fprintf(fp, "%*sUnk 8: %s\n", indent, "", values->unk_8 ? "true" : "false");
   fprintf(fp, "%*sUnk 9: %s\n", indent, "", values->unk_9 ? "true" : "false");
   fprintf(fp, "%*sUnk 10: %s\n", indent, "", values->unk_10 ? "true" : "false");
   fprintf(fp, "%*sUnk 11: %s\n", indent, "", values->unk_11 ? "true" : "false");
   fprintf(fp, "%*sUnk 12: %s\n", indent, "", values->unk_12 ? "true" : "false");
   fprintf(fp, "%*sUnk 13: %s\n", indent, "", values->unk_13 ? "true" : "false");
   fprintf(fp, "%*sUnk 14: %s\n", indent, "", values->unk_14 ? "true" : "false");
   fprintf(fp, "%*sUnk 15: %s\n", indent, "", values->unk_15 ? "true" : "false");
   fprintf(fp, "%*sUnk 16: %s\n", indent, "", values->unk_16 ? "true" : "false");
   fprintf(fp, "%*sUnk 17: %s\n", indent, "", values->unk_17 ? "true" : "false");
   fprintf(fp, "%*sUnk 18: %s\n", indent, "", values->unk_18 ? "true" : "false");
   fprintf(fp, "%*sUnk 19: %s\n", indent, "", values->unk_19 ? "true" : "false");
   fprintf(fp, "%*sUnk 20: %s\n", indent, "", values->unk_20 ? "true" : "false");
   fprintf(fp, "%*sUnk 24: %s\n", indent, "", values->unk_24 ? "true" : "false");
   fprintf(fp, "%*sUnk 26: %s\n", indent, "", values->unk_26 ? "true" : "false");
   fprintf(fp, "%*sReturns: %s\n", indent, "", values->returns ? "true" : "false");
}
#endif

struct AGX_CDM_STREAM_LINK {
   uint32_t                             target_hi;
   bool                                 with_return;
   uint32_t                             target_lo;
};

#define AGX_CDM_STREAM_LINK_header 0

static inline void
AGX_CDM_STREAM_LINK_pack(GLOBAL uint32_t * restrict cl,
                         const struct AGX_CDM_STREAM_LINK * restrict values)
{
   agx_genxml_validate_bounds("CDM Stream Link::target_hi", values->target_hi, 0x100ull);
   agx_genxml_validate_bounds("CDM Stream Link::with_return", values->with_return, 0x2ull);
   cl[ 0] = util_bitpack_uint(values->target_hi, 0, 7) |
            util_bitpack_uint(values->with_return, 28, 28) |
            util_bitpack_uint(AGX_CDM_BLOCK_TYPE_STREAM_LINK, 29, 31);
   agx_genxml_validate_bounds("CDM Stream Link::target_lo", values->target_lo, 0x100000000ull);
   cl[ 1] = util_bitpack_uint(values->target_lo, 0, 31);
}

#define AGX_CDM_STREAM_LINK_LENGTH 8
struct agx_cdm_stream_link_packed { uint32_t opaque[2];};
static inline bool
AGX_CDM_STREAM_LINK_unpack(FILE *fp, CONST uint8_t * restrict cl,
                           struct AGX_CDM_STREAM_LINK * restrict values)
{
   values->target_hi = __gen_unpack_uint((CONST uint32_t *) cl, 0, 7);
   values->with_return = __gen_unpack_uint((CONST uint32_t *) cl, 28, 28);
   values->target_lo = __gen_unpack_uint((CONST uint32_t *) cl, 32, 63);
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "CDM Stream Link", cl, 0, 0xfffff00);
   valid &= agx_genxml_validate_exact(fp, "CDM Stream Link", __gen_unpack_uint((CONST uint32_t *) cl, 29, 31), AGX_CDM_BLOCK_TYPE_STREAM_LINK);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_CDM_STREAM_LINK_print(FILE *fp, const struct AGX_CDM_STREAM_LINK * values, unsigned indent)
{
   fprintf(fp, "%*sTarget hi: 0x%" PRIx32 "\n", indent, "", values->target_hi);
   fprintf(fp, "%*sWith return: %s\n", indent, "", values->with_return ? "true" : "false");
   fprintf(fp, "%*sTarget lo: 0x%" PRIx32 "\n", indent, "", values->target_lo);
}
#endif

struct AGX_CDM_STREAM_TERMINATE {
   int dummy;
};

#define AGX_CDM_STREAM_TERMINATE_header 0

static inline void
AGX_CDM_STREAM_TERMINATE_pack(GLOBAL uint32_t * restrict cl,
                              const struct AGX_CDM_STREAM_TERMINATE * restrict values)
{
   cl[ 0] = util_bitpack_uint(AGX_CDM_BLOCK_TYPE_STREAM_TERMINATE, 29, 31);
   cl[ 1] = 0;
}

#define AGX_CDM_STREAM_TERMINATE_LENGTH 8
struct agx_cdm_stream_terminate_packed { uint32_t opaque[2];};
static inline bool
AGX_CDM_STREAM_TERMINATE_unpack(FILE *fp, CONST uint8_t * restrict cl,
                                struct AGX_CDM_STREAM_TERMINATE * restrict values)
{
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "CDM Stream Terminate", cl, 0, 0x1fffffff);
   valid &= agx_genxml_validate_mask(fp, "CDM Stream Terminate", cl, 1, 0xffffffff);
   valid &= agx_genxml_validate_exact(fp, "CDM Stream Terminate", __gen_unpack_uint((CONST uint32_t *) cl, 29, 31), AGX_CDM_BLOCK_TYPE_STREAM_TERMINATE);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_CDM_STREAM_TERMINATE_print(FILE *fp, const struct AGX_CDM_STREAM_TERMINATE * values, unsigned indent)
{
}
#endif

struct AGX_CDM_STREAM_RETURN {
   int dummy;
};

#define AGX_CDM_STREAM_RETURN_header 0

static inline void
AGX_CDM_STREAM_RETURN_pack(GLOBAL uint32_t * restrict cl,
                           const struct AGX_CDM_STREAM_RETURN * restrict values)
{
   cl[ 0] = util_bitpack_uint(AGX_CDM_BLOCK_TYPE_STREAM_RETURN, 29, 31);
   cl[ 1] = 0;
}

#define AGX_CDM_STREAM_RETURN_LENGTH 8
struct agx_cdm_stream_return_packed { uint32_t opaque[2];};
static inline bool
AGX_CDM_STREAM_RETURN_unpack(FILE *fp, CONST uint8_t * restrict cl,
                             struct AGX_CDM_STREAM_RETURN * restrict values)
{
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "CDM Stream Return", cl, 0, 0x1fffffff);
   valid &= agx_genxml_validate_mask(fp, "CDM Stream Return", cl, 1, 0xffffffff);
   valid &= agx_genxml_validate_exact(fp, "CDM Stream Return", __gen_unpack_uint((CONST uint32_t *) cl, 29, 31), AGX_CDM_BLOCK_TYPE_STREAM_RETURN);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_CDM_STREAM_RETURN_print(FILE *fp, const struct AGX_CDM_STREAM_RETURN * values, unsigned indent)
{
}
#endif

enum agx_zls_format {
   AGX_ZLS_FORMAT_32F = 0,
   AGX_ZLS_FORMAT_16 = 2,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_zls_format_as_str(enum agx_zls_format imm)
{
    switch (imm) {
    case AGX_ZLS_FORMAT_32F: return "32F";
    case AGX_ZLS_FORMAT_16: return "16";
    default: return NULL;
    }
}
#endif

enum agx_zls_tiling {
   AGX_ZLS_TILING_GPU = 0,
   AGX_ZLS_TILING_TWIDDLED = 1,
};

#ifndef __OPENCL_VERSION__
static inline const char *
agx_zls_tiling_as_str(enum agx_zls_tiling imm)
{
    switch (imm) {
    case AGX_ZLS_TILING_GPU: return "GPU";
    case AGX_ZLS_TILING_TWIDDLED: return "Twiddled";
    default: return NULL;
    }
}
#endif

struct AGX_ZLS_CONTROL {
   bool                                 unknown_0;
   enum agx_zls_tiling                  z_load_tiling;
   bool                                 z_load_compress;
   enum agx_zls_tiling                  s_load_tiling;
   bool                                 s_load_compress;
   enum agx_zls_tiling                  z_store_tiling;
   bool                                 z_store_compress;
   enum agx_zls_tiling                  s_store_tiling;
   bool                                 s_store_compress;
   bool                                 s_load;
   bool                                 z_load;
   bool                                 s_store;
   bool                                 z_store;
   enum agx_zls_format                  z_format;
   bool                                 z_resolve;
   bool                                 s_resolve;
};

#define AGX_ZLS_CONTROL_header 0

static inline void
AGX_ZLS_CONTROL_pack(GLOBAL uint32_t * restrict cl,
                     const struct AGX_ZLS_CONTROL * restrict values)
{
   agx_genxml_validate_bounds("ZLS Control::unknown_0", values->unknown_0, 0x2ull);
   agx_genxml_validate_bounds("ZLS Control::z_load_tiling", values->z_load_tiling, 0x2ull);
   agx_genxml_validate_bounds("ZLS Control::z_load_compress", values->z_load_compress, 0x2ull);
   agx_genxml_validate_bounds("ZLS Control::s_load_tiling", values->s_load_tiling, 0x2ull);
   agx_genxml_validate_bounds("ZLS Control::s_load_compress", values->s_load_compress, 0x2ull);
   agx_genxml_validate_bounds("ZLS Control::z_store_tiling", values->z_store_tiling, 0x2ull);
   agx_genxml_validate_bounds("ZLS Control::z_store_compress", values->z_store_compress, 0x2ull);
   agx_genxml_validate_bounds("ZLS Control::s_store_tiling", values->s_store_tiling, 0x2ull);
   agx_genxml_validate_bounds("ZLS Control::s_store_compress", values->s_store_compress, 0x2ull);
   agx_genxml_validate_bounds("ZLS Control::s_load", values->s_load, 0x2ull);
   agx_genxml_validate_bounds("ZLS Control::z_load", values->z_load, 0x2ull);
   agx_genxml_validate_bounds("ZLS Control::s_store", values->s_store, 0x2ull);
   agx_genxml_validate_bounds("ZLS Control::z_store", values->z_store, 0x2ull);
   agx_genxml_validate_bounds("ZLS Control::z_format", values->z_format, 0x4ull);
   cl[ 0] = util_bitpack_uint(values->unknown_0, 0, 0) |
            util_bitpack_uint(values->z_load_tiling, 1, 1) |
            util_bitpack_uint(values->z_load_compress, 2, 2) |
            util_bitpack_uint(values->s_load_tiling, 3, 3) |
            util_bitpack_uint(values->s_load_compress, 4, 4) |
            util_bitpack_uint(values->z_store_tiling, 5, 5) |
            util_bitpack_uint(values->z_store_compress, 6, 6) |
            util_bitpack_uint(values->s_store_tiling, 7, 7) |
            util_bitpack_uint(values->s_store_compress, 8, 8) |
            util_bitpack_uint(values->s_load, 14, 14) |
            util_bitpack_uint(values->z_load, 15, 15) |
            util_bitpack_uint(values->s_store, 18, 18) |
            util_bitpack_uint(values->z_store, 19, 19) |
            util_bitpack_uint(values->z_format, 25, 26);
   agx_genxml_validate_bounds("ZLS Control::z_resolve", values->z_resolve, 0x2ull);
   agx_genxml_validate_bounds("ZLS Control::s_resolve", values->s_resolve, 0x2ull);
   cl[ 1] = util_bitpack_uint(values->z_resolve, 24, 24) |
            util_bitpack_uint(values->s_resolve, 26, 26);
}

#define AGX_ZLS_CONTROL_LENGTH 8
struct agx_zls_control_packed { uint32_t opaque[2];};
static inline bool
AGX_ZLS_CONTROL_unpack(FILE *fp, CONST uint8_t * restrict cl,
                       struct AGX_ZLS_CONTROL * restrict values)
{
   values->unknown_0 = __gen_unpack_uint((CONST uint32_t *) cl, 0, 0);
   values->z_load_tiling = (enum agx_zls_tiling) __gen_unpack_uint((CONST uint32_t *) cl, 1, 1);
   values->z_load_compress = __gen_unpack_uint((CONST uint32_t *) cl, 2, 2);
   values->s_load_tiling = (enum agx_zls_tiling) __gen_unpack_uint((CONST uint32_t *) cl, 3, 3);
   values->s_load_compress = __gen_unpack_uint((CONST uint32_t *) cl, 4, 4);
   values->z_store_tiling = (enum agx_zls_tiling) __gen_unpack_uint((CONST uint32_t *) cl, 5, 5);
   values->z_store_compress = __gen_unpack_uint((CONST uint32_t *) cl, 6, 6);
   values->s_store_tiling = (enum agx_zls_tiling) __gen_unpack_uint((CONST uint32_t *) cl, 7, 7);
   values->s_store_compress = __gen_unpack_uint((CONST uint32_t *) cl, 8, 8);
   values->s_load = __gen_unpack_uint((CONST uint32_t *) cl, 14, 14);
   values->z_load = __gen_unpack_uint((CONST uint32_t *) cl, 15, 15);
   values->s_store = __gen_unpack_uint((CONST uint32_t *) cl, 18, 18);
   values->z_store = __gen_unpack_uint((CONST uint32_t *) cl, 19, 19);
   values->z_format = (enum agx_zls_format) __gen_unpack_uint((CONST uint32_t *) cl, 25, 26);
   values->z_resolve = __gen_unpack_uint((CONST uint32_t *) cl, 56, 56);
   values->s_resolve = __gen_unpack_uint((CONST uint32_t *) cl, 58, 58);
   bool valid = true;
   valid &= agx_genxml_validate_mask(fp, "ZLS Control", cl, 0, 0xf9f33e00);
   valid &= agx_genxml_validate_mask(fp, "ZLS Control", cl, 1, 0xfaffffff);
   return valid;
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_ZLS_CONTROL_print(FILE *fp, const struct AGX_ZLS_CONTROL * values, unsigned indent)
{
   fprintf(fp, "%*sUnknown 0: %s\n", indent, "", values->unknown_0 ? "true" : "false");
   if (agx_zls_tiling_as_str(values->z_load_tiling))
     fprintf(fp, "%*sZ Load Tiling: %s\n", indent, "", agx_zls_tiling_as_str(values->z_load_tiling));
   else
     fprintf(fp, "%*sZ Load Tiling: unknown %X (XXX)\n", indent, "", values->z_load_tiling);
   fprintf(fp, "%*sZ Load Compress: %s\n", indent, "", values->z_load_compress ? "true" : "false");
   if (agx_zls_tiling_as_str(values->s_load_tiling))
     fprintf(fp, "%*sS Load Tiling: %s\n", indent, "", agx_zls_tiling_as_str(values->s_load_tiling));
   else
     fprintf(fp, "%*sS Load Tiling: unknown %X (XXX)\n", indent, "", values->s_load_tiling);
   fprintf(fp, "%*sS Load Compress: %s\n", indent, "", values->s_load_compress ? "true" : "false");
   if (agx_zls_tiling_as_str(values->z_store_tiling))
     fprintf(fp, "%*sZ Store Tiling: %s\n", indent, "", agx_zls_tiling_as_str(values->z_store_tiling));
   else
     fprintf(fp, "%*sZ Store Tiling: unknown %X (XXX)\n", indent, "", values->z_store_tiling);
   fprintf(fp, "%*sZ Store Compress: %s\n", indent, "", values->z_store_compress ? "true" : "false");
   if (agx_zls_tiling_as_str(values->s_store_tiling))
     fprintf(fp, "%*sS Store Tiling: %s\n", indent, "", agx_zls_tiling_as_str(values->s_store_tiling));
   else
     fprintf(fp, "%*sS Store Tiling: unknown %X (XXX)\n", indent, "", values->s_store_tiling);
   fprintf(fp, "%*sS Store Compress: %s\n", indent, "", values->s_store_compress ? "true" : "false");
   fprintf(fp, "%*sS Load: %s\n", indent, "", values->s_load ? "true" : "false");
   fprintf(fp, "%*sZ Load: %s\n", indent, "", values->z_load ? "true" : "false");
   fprintf(fp, "%*sS Store: %s\n", indent, "", values->s_store ? "true" : "false");
   fprintf(fp, "%*sZ Store: %s\n", indent, "", values->z_store ? "true" : "false");
   if (agx_zls_format_as_str(values->z_format))
     fprintf(fp, "%*sZ Format: %s\n", indent, "", agx_zls_format_as_str(values->z_format));
   else
     fprintf(fp, "%*sZ Format: unknown %X (XXX)\n", indent, "", values->z_format);
   fprintf(fp, "%*sZ Resolve: %s\n", indent, "", values->z_resolve ? "true" : "false");
   fprintf(fp, "%*sS Resolve: %s\n", indent, "", values->s_resolve ? "true" : "false");
}
#endif

struct AGX_CR_ISP_ZLS_PIXELS {
   uint32_t                             x;
   uint32_t                             y;
};

#define AGX_CR_ISP_ZLS_PIXELS_header            \
   .x = 1,  \
   .y = 1

static inline void
AGX_CR_ISP_ZLS_PIXELS_pack(GLOBAL uint32_t * restrict cl,
                           const struct AGX_CR_ISP_ZLS_PIXELS * restrict values)
{
   assert(values->x >= 1);
   assert(values->y >= 1);
   agx_genxml_validate_bounds("CR ISP ZLS Pixels::x", values->x - 1, 0x8000ull);
   agx_genxml_validate_bounds("CR ISP ZLS Pixels::y", values->y - 1, 0x8000ull);
   cl[ 0] = util_bitpack_uint(values->x - 1, 0, 14) |
            util_bitpack_uint(values->y - 1, 15, 29);
}

#define AGX_CR_ISP_ZLS_PIXELS_LENGTH 4
struct agx_cr_isp_zls_pixels_packed { uint32_t opaque[1];};
static inline bool
AGX_CR_ISP_ZLS_PIXELS_unpack(FILE *fp, CONST uint8_t * restrict cl,
                             struct AGX_CR_ISP_ZLS_PIXELS * restrict values)
{
   values->x = __gen_unpack_uint((CONST uint32_t *) cl, 0, 14) + 1;
   values->y = __gen_unpack_uint((CONST uint32_t *) cl, 15, 29) + 1;
   return agx_genxml_validate_mask(fp, "CR ISP ZLS Pixels", cl, 0, 0xc0000000);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_CR_ISP_ZLS_PIXELS_print(FILE *fp, const struct AGX_CR_ISP_ZLS_PIXELS * values, unsigned indent)
{
   fprintf(fp, "%*sX: %u\n", indent, "", values->x);
   fprintf(fp, "%*sY: %u\n", indent, "", values->y);
}
#endif

struct AGX_CR_PPP_CONTROL {
   bool                                 opengl;
   bool                                 enable_w_clamp;
   bool                                 default_point_size;
   uint32_t                             fixed_point_format;
};

#define AGX_CR_PPP_CONTROL_header 0

static inline void
AGX_CR_PPP_CONTROL_pack(GLOBAL uint32_t * restrict cl,
                        const struct AGX_CR_PPP_CONTROL * restrict values)
{
   agx_genxml_validate_bounds("CR PPP Control::opengl", values->opengl, 0x2ull);
   agx_genxml_validate_bounds("CR PPP Control::enable_w_clamp", values->enable_w_clamp, 0x2ull);
   agx_genxml_validate_bounds("CR PPP Control::default_point_size", values->default_point_size, 0x2ull);
   agx_genxml_validate_bounds("CR PPP Control::fixed_point_format", values->fixed_point_format, 0x2ull);
   cl[ 0] = util_bitpack_uint(values->opengl, 0, 0) |
            util_bitpack_uint(values->enable_w_clamp, 1, 1) |
            util_bitpack_uint(values->default_point_size, 8, 8) |
            util_bitpack_uint(values->fixed_point_format, 9, 9);
}

#define AGX_CR_PPP_CONTROL_LENGTH 4
struct agx_cr_ppp_control_packed { uint32_t opaque[1];};
static inline bool
AGX_CR_PPP_CONTROL_unpack(FILE *fp, CONST uint8_t * restrict cl,
                          struct AGX_CR_PPP_CONTROL * restrict values)
{
   values->opengl = __gen_unpack_uint((CONST uint32_t *) cl, 0, 0);
   values->enable_w_clamp = __gen_unpack_uint((CONST uint32_t *) cl, 1, 1);
   values->default_point_size = __gen_unpack_uint((CONST uint32_t *) cl, 8, 8);
   values->fixed_point_format = __gen_unpack_uint((CONST uint32_t *) cl, 9, 9);
   return agx_genxml_validate_mask(fp, "CR PPP Control", cl, 0, 0xfffffcfc);
}

#ifndef __OPENCL_VERSION__
static inline void
AGX_CR_PPP_CONTROL_print(FILE *fp, const struct AGX_CR_PPP_CONTROL * values, unsigned indent)
{
   fprintf(fp, "%*sOpenGL: %s\n", indent, "", values->opengl ? "true" : "false");
   fprintf(fp, "%*sEnable W Clamp: %s\n", indent, "", values->enable_w_clamp ? "true" : "false");
   fprintf(fp, "%*sDefault point size: %s\n", indent, "", values->default_point_size ? "true" : "false");
   fprintf(fp, "%*sFixed point format: %u\n", indent, "", values->fixed_point_format);
}
#endif

