// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A6__F534

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_224_0_T_T_0___gfx120x)xyzw" };
#else
{ R"xyzw(amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_224_0_T_T_0___gfx120x)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp32@16_224_0_T_T_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1201)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xeaec55c0u, 0xd065ddd9u, 0, 46 }, // eaec55c0d065ddd9 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_224_0_T_T_0___gfx120x__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1201
    { 0xc5423d85u, 0x004bcfc9u, 0, 63 }, // c5423d85004bcfc9 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_224_0_T_T_0___gfx120x__P__16_16__CO__wave2_warp2_stg1--Arch_gfx1201
    { 0xb75915c8u, 0x8a8c1204u, 0, 6 }, // b75915c88a8c1204 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_224_0_T_T_0___gfx120x__P__16_16__CO__wave3_warp2_stg1--Arch_gfx1201
    { 0xbfa2f33au, 0xb8c4c448u, 0, 23 }, // bfa2f33ab8c4c448 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_224_0_T_T_0___gfx120x__P__16_16__CO__wave4_warp2_stg1--Arch_gfx1201
    { 0xc741746bu, 0x7039fd1bu, 0, 103 }, // c741746b7039fd1b = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_224_0_T_T_0___gfx120x__P__16_16__CO__wave4_warp4_stg1--Arch_gfx1201
    { 0xb00eacaau, 0x16cac3ecu, 40, 138 }, // b00eacaa16cac3ec = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_224_0_T_T_0___gfx120x__P__32_16__CO__wave1_warp4_stg1--Arch_gfx1201
    { 0xf91235a4u, 0x78af2709u, 40, 80 }, // f91235a478af2709 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_224_0_T_T_0___gfx120x__P__32_16__CO__wave2_warp4_stg1--Arch_gfx1201
    { 0x195ab69au, 0x1d0d4141u, 40, 155 }, // 195ab69a1d0d4141 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_224_0_T_T_0___gfx120x__P__32_16__CO__wave3_warp4_stg1--Arch_gfx1201
    { 0x1c40e1e7u, 0x5ebb1ce6u, 40, 103 }, // 1c40e1e75ebb1ce6 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_224_0_T_T_0___gfx120x__P__32_16__CO__wave4_warp4_stg1--Arch_gfx1201
    { 0xf04e2146u, 0x5fdeb431u, 97, 138 }, // f04e21465fdeb431 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_224_0_T_T_0___gfx120x__P__32_32__CO__wave1_warp4_stg1--Arch_gfx1201
    { 0x16b8b9c4u, 0x7d348fe6u, 97, 80 }, // 16b8b9c47d348fe6 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_224_0_T_T_0___gfx120x__P__32_32__CO__wave2_warp4_stg1--Arch_gfx1201
    { 0x3c723c29u, 0x33b2392bu, 97, 155 }, // 3c723c2933b2392b = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_224_0_T_T_0___gfx120x__P__32_32__CO__wave3_warp4_stg1--Arch_gfx1201
    { 0x6eda9228u, 0xf29da494u, 97, 103 }, // 6eda9228f29da494 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_224_0_T_T_0___gfx120x__P__32_32__CO__wave4_warp4_stg1--Arch_gfx1201
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1201_mod0
{{ 0, 4, 8, 5, 6, 8, 8, 5, 6, 8},
 { 5, 5, 7, 6,12, 9,11,11,12,11},
 {12, 1,12,12,11,11,12,12,12,12},
 { 3, 3, 9, 9,12,12, 9,11,10,10},
 { 3, 0, 3, 9,11,12, 9, 9, 9,10},
 { 0, 3, 3, 9,12, 9,11,12,12,12},
 { 3, 3, 0, 9,12, 9,11, 9,12,11},
 { 3, 0, 3, 9, 9,12, 9, 9, 9,10},
 { 1, 3, 3, 9, 3,11, 9,10,10, 9},
 { 0, 2, 2,10,12,11, 9, 9, 9, 1}}
// End of GPU gfx1201_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A6__F534 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 2;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 2;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp32:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = 0
// BLOCK_DMODEL = 224
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = True
// PADDED_HEAD = True
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dq WHERE gpu IN ('gfx1201_mod0') AND inputs$Q_dtype = 'torch.float32' AND inputs$BLOCK_DMODEL = 224 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

