// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dk_dv.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dk_dv__A5__F210

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_T_0___gfx11xx)xyzw" };
#else
{ R"xyzw(amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_T_0___gfx11xx)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_32_0_F_T_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1150)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x3f59e527u, 0xfbbe9fbeu, 0, 29 }, // 3f59e527fbbe9fbe = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_T_0___gfx11xx__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1150
    { 0x598dfb10u, 0x7d7234bdu, 23, 29 }, // 598dfb107d7234bd = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_T_0___gfx11xx__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1150
    { 0x5fef4986u, 0x9d683e1bu, 23, 6 }, // 5fef49869d683e1b = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_T_0___gfx11xx__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1150
    { 0xe84c64d6u, 0xdcec3cfdu, 23, 46 }, // e84c64d6dcec3cfd = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_T_0___gfx11xx__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1150
    { 0x0d560409u, 0x7e700952u, 23, 63 }, // 0d5604097e700952 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_T_0___gfx11xx__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1150
    { 0xb245658du, 0x1951110au, 103, 109 }, // b245658d1951110a = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_T_0___gfx11xx__P__64_16__CO__wave1_warp4_stg1--Arch_gfx1150
    { 0x58df4561u, 0xb5b9bb71u, 103, 143 }, // 58df4561b5b9bb71 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_T_0___gfx11xx__P__64_16__CO__wave4_warp4_stg1--Arch_gfx1150
    { 0x8ed5f352u, 0xf102e9e4u, 160, 109 }, // 8ed5f352f102e9e4 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_T_0___gfx11xx__P__64_32__CO__wave1_warp4_stg1--Arch_gfx1150
    { 0x2d8efd43u, 0xbd91c9a0u, 160, 126 }, // 2d8efd43bd91c9a0 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_T_0___gfx11xx__P__64_32__CO__wave2_warp4_stg1--Arch_gfx1150
    { 0x243ad9afu, 0xf1e23758u, 160, 86 }, // 243ad9aff1e23758 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_T_0___gfx11xx__P__64_32__CO__wave3_warp4_stg1--Arch_gfx1150
    { 0xb9cb6fd5u, 0xe57e5743u, 160, 143 }, // b9cb6fd5e57e5743 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_32_0_F_T_0___gfx11xx__P__64_32__CO__wave4_warp4_stg1--Arch_gfx1150
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dk_dv_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dk_dv_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1150_mod0
{{ 1, 4, 1, 0, 0, 3, 2, 2, 3, 4},
 { 4, 4, 0, 0, 0, 0, 0, 3, 1, 0},
 { 1, 2, 1, 2, 3, 3, 1, 1, 0, 4},
 { 2, 4, 1, 3, 3, 4, 4, 1, 4, 1},
 { 4, 3, 3, 2, 4, 2, 4, 2, 2, 2},
 { 3, 3, 2, 3, 4, 4, 2, 4, 4, 2},
 { 6, 4, 1, 2, 2, 4, 4, 1, 2, 1},
 { 5, 4, 3, 4, 2,10, 8,10, 3, 3},
 { 2, 2, 2, 4,10, 9, 9, 1, 1, 3},
 { 2, 4, 2, 2, 1,10, 9, 7,10, 9}}
// End of GPU gfx1150_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDkDvContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDkDvContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dk_dv__A5__F210 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 0;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dk_dv__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 0;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = 0
// BLOCK_DMODEL = 32
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = False
// PADDED_HEAD = True
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dk_dv WHERE gpu IN ('gfx1100_mod0') AND inputs$Q_dtype = 'torch.bfloat16' AND inputs$BLOCK_DMODEL = 32 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = False AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

