// Copyright (c) Lawrence Livermore National Security, LLC and other VisIt
// Project developers.  See the top-level LICENSE file for dates and other
// details.  No copyright assignment is required to contribute to VisIt.

// ************************************************************************* //
//                                 avtImage.C                                //
// ************************************************************************* //

#include <avtImage.h>

#include <avtImageSource.h>
#include <avtImageWriter.h>


// ****************************************************************************
//  Method: avtImage constructor
//
//  Arguments:
//      src     An data object source that is the upstream object.
//
//  Programmer: Hank Childs
//  Creation:   November 21, 2000
//
// ****************************************************************************

avtImage::avtImage(avtDataObjectSource *src)
    : avtDataObject(src)
{
    ;
}


// ****************************************************************************
//  Method: avtImage destructor
//
//  Purpose:
//      Defines the destructor.  Note: this should not be inlined in the header
//      because it causes problems for certain compilers.
//
//  Programmer: Hank Childs
//  Creation:   February 5, 2004
//
// ****************************************************************************

avtImage::~avtImage()
{
    ;
}


// ****************************************************************************
//  Method: avtImage::SetImage
//
//  Purpose:
//      Sets this objects image.  This is a protected method that should only
//      be accessed by avtImageSource.
//
//  Arguments:
//      ir      The image representation of the new image.
//
//  Programmer: Hank Childs
//  Creation:   November 21, 2000
//
// ****************************************************************************

void
avtImage::SetImage(const avtImageRepresentation &ir)
{
    image = ir;
}


// ****************************************************************************
//  Method: avtImage::GetImage
//
//  Purpose:
//      Gets the image as an avtImageRepresentation.
//
//  Returns:    the image.
//
//  Programmer: Hank Childs
//  Creation:   November 21, 2000
//
// ****************************************************************************

avtImageRepresentation &
avtImage::GetImage(void)
{
    return image;
}

// ****************************************************************************
//  Method: avtImage::GetSize
//
//  Purpose:
//      Gets size of the image 
//
//  Programmer: Mark C. Miller 
//  Creation:   31Mar04 
//
// ****************************************************************************

void
avtImage::GetSize(int *width, int *height) const
{
   // argument order is inverted due to fact that image rep uses 'rowsize' and
   // 'colsize' instead of width and height
   image.GetSize(height, width);
}

// ****************************************************************************
//  Method: avtImage::GetNumberOfCells
//
//  Purpose:
//      Gets number of cells in an image 
//
//  Programmer: Mark C. Miller 
//  Creation:   19Aug03 
//
//  Modificaitons:
//
//    Mark C. Miller, Wed Nov  5 09:48:13 PST 2003
//    Added option to count polygons only
//
//    Burlen Loring, Sun Sep  6 14:58:03 PDT 2015
//    Changed the return type of GetNumberOfCells to long long
//
// ****************************************************************************

long long
avtImage::GetNumberOfCells(bool polysOnly) const
{
   return image.GetNumberOfCells(polysOnly);
}


// ****************************************************************************
//  Method: avtImage::InstantiateWriter
//
//  Purpose:
//      Instantiates a writer that is appropriate for an avtImage.
//
//  Programmer: Hank Childs
//  Creation:   October 1, 2001
//
// ****************************************************************************

avtDataObjectWriter *
avtImage::InstantiateWriter(void)
{
    return new avtImageWriter;
}


// ****************************************************************************
//  Method: avtImage::ReleaseData
//
//  Purpose:
//      Free the data associated with this image.
//
//  Programmer: Hank Childs
//  Creation:   November 5, 2001
//
// ****************************************************************************

void
avtImage::ReleaseData(void)
{
    image.ReleaseData();
}

// ****************************************************************************
//  Method: avtImage::Instance
//
//  Purpose:
//      Creates an instance of an avtImage.
//
//  Programmer: Mark C. Miller
//  Creation:   February 4, 2004 
//
// ****************************************************************************

avtDataObject *
avtImage::Instance(void)
{
    avtDataObjectSource *src = NULL;
    avtImage *img = new avtImage(src);
    return img;
}

// ****************************************************************************
//  Method: avtImage::DerivedCopy
//
//  Purpose:
//      Copy over the image.
//
//  Programmer: Hank Childs
//  Creation:   November 21, 2001
//
// ****************************************************************************

void
avtImage::DerivedCopy(avtDataObject *dob)
{
    avtImage *img = (avtImage *) dob;
    image = img->image;
}


