#include "js/ForOfIterator.h"
#include "mozilla/dom/BindingCallContext.h"
#include "mozilla/dom/Blob.h"
#include "mozilla/dom/Directory.h"
#include "mozilla/dom/File.h"
#include "mozilla/dom/FormData.h"
#include "mozilla/dom/HTMLCanvasElement.h"
#include "mozilla/dom/HTMLOptGroupElement.h"
#include "mozilla/dom/HTMLOptionElement.h"
#include "mozilla/dom/OffscreenCanvas.h"
#include "mozilla/dom/PrimitiveConversions.h"
#include "mozilla/dom/TrustedHTML.h"
#include "mozilla/dom/TrustedScriptURL.h"
#include "mozilla/dom/UnionTypes.h"
#include "nsGenericHTMLElement.h"
#include "nsIContent.h"

namespace mozilla::dom {
void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsBlob()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsBlob(), "mBlob", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningFileOrDirectory& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsFile()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsFile(), "mFile", aFlags);
  } else if (aUnion.IsDirectory()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsDirectory(), "mDirectory", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningFileOrUSVStringOrFormData& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsFile()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsFile(), "mFile", aFlags);
  } else if (aUnion.IsFormData()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsFormData(), "mFormData", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningHTMLCanvasElementOrOffscreenCanvas& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsHTMLCanvasElement()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsHTMLCanvasElement(), "mHTMLCanvasElement", aFlags);
  } else if (aUnion.IsOffscreenCanvas()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsOffscreenCanvas(), "mOffscreenCanvas", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningHTMLElementOrLong& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsHTMLElement()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsHTMLElement(), "mHTMLElement", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningHTMLOptionElementOrHTMLOptGroupElement& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsHTMLOptionElement()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsHTMLOptionElement(), "mHTMLOptionElement", aFlags);
  } else if (aUnion.IsHTMLOptGroupElement()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsHTMLOptGroupElement(), "mHTMLOptGroupElement", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningNodeOrString& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsNode()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsNode(), "mNode", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningTrustedHTMLOrNullIsEmptyString& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsTrustedHTML()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsTrustedHTML(), "mTrustedHTML", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningTrustedHTMLOrString& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsTrustedHTML()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsTrustedHTML(), "mTrustedHTML", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningTrustedScriptURLOrString& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsTrustedScriptURL()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsTrustedScriptURL(), "mTrustedScriptURL", aFlags);
  }
}

void
ImplCycleCollectionTraverse(nsCycleCollectionTraversalCallback& aCallback, OwningTrustedScriptURLOrUSVString& aUnion, const char* aName, uint32_t aFlags)
{
  if (aUnion.IsTrustedScriptURL()) {
    ImplCycleCollectionTraverse(aCallback, aUnion.GetAsTrustedScriptURL(), "mTrustedScriptURL", aFlags);
  }
}

void
ImplCycleCollectionUnlink(OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningFileOrDirectory& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningFileOrUSVStringOrFormData& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningHTMLCanvasElementOrOffscreenCanvas& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningHTMLElementOrLong& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningHTMLOptionElementOrHTMLOptGroupElement& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningNodeOrString& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningTrustedHTMLOrNullIsEmptyString& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningTrustedHTMLOrString& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningTrustedScriptURLOrString& aUnion)
{
  aUnion.Uninit();
}

void
ImplCycleCollectionUnlink(OwningTrustedScriptURLOrUSVString& aUnion)
{
  aUnion.Uninit();
}

bool
ArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToArrayBufferView(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    RootedSpiderMonkeyInterface<ArrayBufferView>& memberSlot = RawSetAsArrayBufferView(cx);
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBufferView();
      tryNext = true;
      return true;
    }
    if (JS::IsArrayBufferViewShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBufferView branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsLargeArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBufferView branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsResizableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBufferView branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsImmutableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBufferView branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
  }
  return true;
}

bool
ArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToArrayBufferView(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBufferView(cx, value, tryNext, passedToJSImpl);
}







bool
ArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToArrayBuffer(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    RootedSpiderMonkeyInterface<ArrayBuffer>& memberSlot = RawSetAsArrayBuffer(cx);
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBuffer();
      tryNext = true;
      return true;
    }
    if (JS::IsSharedArrayBufferObject(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBuffer branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsLargeArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBuffer branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsResizableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBuffer branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsImmutableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBuffer branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
  }
  return true;
}

bool
ArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToArrayBuffer(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl);
}







bool
ArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToBlob(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::Blob>& memberSlot = RawSetAsBlob();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::Blob, mozilla::dom::Blob>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyBlob();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
ArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToBlob(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToBlob(cx, value, tryNext, passedToJSImpl);
}







bool
ArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToUTF8String(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char>& memberSlot = RawSetAsUTF8String();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
ArrayBufferViewOrArrayBufferOrBlobOrUTF8String::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToArrayBufferView(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToBlob(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUTF8String(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "ArrayBufferView, ArrayBuffer, Blob");
    return false;
  }
  return true;
}

bool
ArrayBufferViewOrArrayBufferOrBlobOrUTF8String::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
ArrayBufferViewOrArrayBufferOrBlobOrUTF8String::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eArrayBufferView: {
      rval.setObject(*mValue.mArrayBufferView.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    case eArrayBuffer: {
      rval.setObject(*mValue.mArrayBuffer.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    case eBlob: {
      if (!GetOrCreateDOMReflector(cx, mValue.mBlob.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eUTF8String: {
      if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8String.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
FileOrDirectory::TrySetToFile(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::File>& memberSlot = RawSetAsFile();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::File, mozilla::dom::File>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyFile();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
FileOrDirectory::TrySetToFile(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToFile(cx, value, tryNext, passedToJSImpl);
}







bool
FileOrDirectory::TrySetToDirectory(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::Directory>& memberSlot = RawSetAsDirectory();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::Directory, mozilla::dom::Directory>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyDirectory();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
FileOrDirectory::TrySetToDirectory(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToDirectory(cx, value, tryNext, passedToJSImpl);
}







bool
FileOrDirectory::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToFile(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToDirectory(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "File, Directory");
    return false;
  }
  return true;
}

bool
FileOrDirectory::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
FileOrDirectory::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eFile: {
      if (!GetOrCreateDOMReflector(cx, mValue.mFile.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eDirectory: {
      if (!GetOrCreateDOMReflector(cx, mValue.mDirectory.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
FileOrUSVStringOrFormData::TrySetToFile(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::File>& memberSlot = RawSetAsFile();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::File, mozilla::dom::File>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyFile();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
FileOrUSVStringOrFormData::TrySetToFile(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToFile(cx, value, tryNext, passedToJSImpl);
}







bool
FileOrUSVStringOrFormData::TrySetToUSVString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsUSVString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
    if (!NormalizeUSVString(memberSlot)) {
      JS_ReportOutOfMemory(cx);
      return false;
    }
  }
  return true;
}








bool
FileOrUSVStringOrFormData::TrySetToFormData(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::FormData>& memberSlot = RawSetAsFormData();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::FormData, mozilla::dom::FormData>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyFormData();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
FileOrUSVStringOrFormData::TrySetToFormData(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToFormData(cx, value, tryNext, passedToJSImpl);
}







bool
FileOrUSVStringOrFormData::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToFile(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToFormData(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUSVString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "File, FormData");
    return false;
  }
  return true;
}

bool
FileOrUSVStringOrFormData::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
FileOrUSVStringOrFormData::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eFile: {
      if (!GetOrCreateDOMReflector(cx, mValue.mFile.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eUSVString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mUSVString.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eFormData: {
      if (!GetOrCreateDOMReflector(cx, mValue.mFormData.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
HTMLCanvasElementOrOffscreenCanvas::TrySetToHTMLCanvasElement(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::HTMLCanvasElement>& memberSlot = RawSetAsHTMLCanvasElement();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::HTMLCanvasElement, mozilla::dom::HTMLCanvasElement>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyHTMLCanvasElement();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
HTMLCanvasElementOrOffscreenCanvas::TrySetToHTMLCanvasElement(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToHTMLCanvasElement(cx, value, tryNext, passedToJSImpl);
}







bool
HTMLCanvasElementOrOffscreenCanvas::TrySetToOffscreenCanvas(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::OffscreenCanvas>& memberSlot = RawSetAsOffscreenCanvas();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::OffscreenCanvas, mozilla::dom::OffscreenCanvas>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyOffscreenCanvas();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
HTMLCanvasElementOrOffscreenCanvas::TrySetToOffscreenCanvas(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToOffscreenCanvas(cx, value, tryNext, passedToJSImpl);
}







bool
HTMLCanvasElementOrOffscreenCanvas::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToHTMLCanvasElement(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToOffscreenCanvas(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "HTMLCanvasElement, OffscreenCanvas");
    return false;
  }
  return true;
}

bool
HTMLCanvasElementOrOffscreenCanvas::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
HTMLCanvasElementOrOffscreenCanvas::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eHTMLCanvasElement: {
      if (!GetOrCreateDOMReflector(cx, mValue.mHTMLCanvasElement.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eOffscreenCanvas: {
      if (!GetOrCreateDOMReflector(cx, mValue.mOffscreenCanvas.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
HTMLElementOrLong::TrySetToHTMLElement(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<nsGenericHTMLElement>& memberSlot = RawSetAsHTMLElement();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::HTMLElement, nsGenericHTMLElement>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyHTMLElement();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
HTMLElementOrLong::TrySetToHTMLElement(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToHTMLElement(cx, value, tryNext, passedToJSImpl);
}







bool
HTMLElementOrLong::TrySetToLong(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    int32_t& memberSlot = RawSetAsLong();
    if (!ValueToPrimitive<int32_t, eDefault>(cx, value, "Long branch of (HTMLElement or long)", &memberSlot)) {
      return false;
    }
  }
  return true;
}







bool
HTMLElementOrLong::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToHTMLElement(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToLong(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "HTMLElement");
    return false;
  }
  return true;
}

bool
HTMLElementOrLong::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
HTMLElementOrLong::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eHTMLElement: {
      if (!GetOrCreateDOMReflector(cx, mValue.mHTMLElement.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eLong: {
      rval.setInt32(int32_t(mValue.mLong.Value()));
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
HTMLOptionElementOrHTMLOptGroupElement::TrySetToHTMLOptionElement(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::HTMLOptionElement>& memberSlot = RawSetAsHTMLOptionElement();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::HTMLOptionElement, mozilla::dom::HTMLOptionElement>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyHTMLOptionElement();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
HTMLOptionElementOrHTMLOptGroupElement::TrySetToHTMLOptionElement(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToHTMLOptionElement(cx, value, tryNext, passedToJSImpl);
}







bool
HTMLOptionElementOrHTMLOptGroupElement::TrySetToHTMLOptGroupElement(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::HTMLOptGroupElement>& memberSlot = RawSetAsHTMLOptGroupElement();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::HTMLOptGroupElement, mozilla::dom::HTMLOptGroupElement>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyHTMLOptGroupElement();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
HTMLOptionElementOrHTMLOptGroupElement::TrySetToHTMLOptGroupElement(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToHTMLOptGroupElement(cx, value, tryNext, passedToJSImpl);
}







bool
HTMLOptionElementOrHTMLOptGroupElement::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToHTMLOptionElement(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToHTMLOptGroupElement(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "HTMLOptionElement, HTMLOptGroupElement");
    return false;
  }
  return true;
}

bool
HTMLOptionElementOrHTMLOptGroupElement::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
HTMLOptionElementOrHTMLOptGroupElement::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eHTMLOptionElement: {
      if (!GetOrCreateDOMReflector(cx, mValue.mHTMLOptionElement.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eHTMLOptGroupElement: {
      if (!GetOrCreateDOMReflector(cx, mValue.mHTMLOptGroupElement.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
NodeOrString::TrySetToNode(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<nsINode>& memberSlot = RawSetAsNode();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::Node, nsINode>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyNode();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
NodeOrString::TrySetToNode(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToNode(cx, value, tryNext, passedToJSImpl);
}







bool
NodeOrString::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
NodeOrString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToNode(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "Node");
    return false;
  }
  return true;
}

bool
NodeOrString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
NodeOrString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eNode: {
      if (!GetOrCreateDOMReflector(cx, mValue.mNode.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
StringOrBooleanOrObject::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
StringOrBooleanOrObject::TrySetToBoolean(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    bool& memberSlot = RawSetAsBoolean();
    if (!ValueToPrimitive<bool, eDefault>(cx, value, "Boolean branch of (DOMString or boolean or object)", &memberSlot)) {
      return false;
    }
  }
  return true;
}












bool
StringOrBooleanOrObject::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    if (!SetToObject(cx, &value.toObject(), passedToJSImpl)) {
      return false;
    }
    done = true;
  } else {
    do {
      if (value.isBoolean()) {
        done = (failed = !TrySetToBoolean(cx, value, tryNext)) || !tryNext;
        break;
      }
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "object");
    return false;
  }
  return true;
}

bool
StringOrBooleanOrObject::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
StringOrBooleanOrObject::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eBoolean: {
      rval.setBoolean(mValue.mBoolean.Value());
      return true;
    }
    case eObject: {
      JS::ExposeObjectToActiveJS(mValue.mObject.Value());
      rval.setObject(*mValue.mObject.Value());
      if (!MaybeWrapObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
StringOrStringSequence::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
StringOrStringSequence::TrySetToStringSequence(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::AutoSequence<nsString>& memberSlot = RawSetAsStringSequence();
    JS::ForOfIterator iter(cx);
    if (!iter.init(value, JS::ForOfIterator::AllowNonIterable)) {
      return false;
    }
    if (!iter.valueIsIterable()) {
      DestroyStringSequence();
      tryNext = true;
      return true;
    }
    binding_detail::AutoSequence<nsString> &arr = memberSlot;
    JS::Rooted<JS::Value> temp(cx);
    while (true) {
      bool done;
      if (!iter.next(&temp, &done)) {
        return false;
      }
      if (done) {
        break;
      }
      nsString* slotPtr = arr.AppendElement(mozilla::fallible);
      if (!slotPtr) {
        JS_ReportOutOfMemory(cx);
        return false;
      }
      nsString& slot = *slotPtr;
      if (!ConvertJSValueToString(cx, temp, eStringify, eStringify, slot)) {
        return false;
      }
    }
  }
  return true;
}

bool
StringOrStringSequence::TrySetToStringSequence(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToStringSequence(cx, value, tryNext, passedToJSImpl);
}







bool
StringOrStringSequence::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToStringSequence(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "sequence<DOMString>");
    return false;
  }
  return true;
}

bool
StringOrStringSequence::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
StringOrStringSequence::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eStringSequence: {

      uint32_t length = mValue.mStringSequence.Value().Length();
      JS::Rooted<JSObject*> returnArray(cx, JS::NewArrayObject(cx, length));
      if (!returnArray) {
        return false;
      }
      // Scope for 'tmp'
      {
        JS::Rooted<JS::Value> tmp(cx);
        for (uint32_t sequenceIdx0 = 0; sequenceIdx0 < length; ++sequenceIdx0) {
          // Control block to let us common up the JS_DefineElement calls when there
          // are different ways to succeed at wrapping the object.
          do {
            if (!xpc::NonVoidStringToJsval(cx, mValue.mStringSequence.Value()[sequenceIdx0], &tmp)) {
              return false;
            }
            break;
          } while (false);
          if (!JS_DefineElement(cx, returnArray, sequenceIdx0, tmp,
                                JSPROP_ENUMERATE)) {
            return false;
          }
        }
      }
      rval.setObject(*returnArray);
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
TrustedHTMLOrNullIsEmptyString::TrySetToTrustedHTML(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::TrustedHTML>& memberSlot = RawSetAsTrustedHTML();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::TrustedHTML, mozilla::dom::TrustedHTML>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyTrustedHTML();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
TrustedHTMLOrNullIsEmptyString::TrySetToTrustedHTML(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToTrustedHTML(cx, value, tryNext, passedToJSImpl);
}







bool
TrustedHTMLOrNullIsEmptyString::TrySetToNullIsEmptyString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsNullIsEmptyString();
    if (!ConvertJSValueToString(cx, value, eEmpty, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
TrustedHTMLOrNullIsEmptyString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToTrustedHTML(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToNullIsEmptyString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "TrustedHTML");
    return false;
  }
  return true;
}

bool
TrustedHTMLOrNullIsEmptyString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
TrustedHTMLOrNullIsEmptyString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eTrustedHTML: {
      if (!WrapNewBindingNonWrapperCachedObject(cx, scopeObj, mValue.mTrustedHTML.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eNullIsEmptyString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mNullIsEmptyString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
TrustedHTMLOrString::TrySetToTrustedHTML(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::TrustedHTML>& memberSlot = RawSetAsTrustedHTML();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::TrustedHTML, mozilla::dom::TrustedHTML>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyTrustedHTML();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
TrustedHTMLOrString::TrySetToTrustedHTML(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToTrustedHTML(cx, value, tryNext, passedToJSImpl);
}







bool
TrustedHTMLOrString::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
TrustedHTMLOrString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToTrustedHTML(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "TrustedHTML");
    return false;
  }
  return true;
}

bool
TrustedHTMLOrString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
TrustedHTMLOrString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eTrustedHTML: {
      if (!WrapNewBindingNonWrapperCachedObject(cx, scopeObj, mValue.mTrustedHTML.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
TrustedScriptURLOrString::TrySetToTrustedScriptURL(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::TrustedScriptURL>& memberSlot = RawSetAsTrustedScriptURL();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::TrustedScriptURL, mozilla::dom::TrustedScriptURL>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyTrustedScriptURL();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
TrustedScriptURLOrString::TrySetToTrustedScriptURL(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToTrustedScriptURL(cx, value, tryNext, passedToJSImpl);
}







bool
TrustedScriptURLOrString::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
TrustedScriptURLOrString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToTrustedScriptURL(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "TrustedScriptURL");
    return false;
  }
  return true;
}

bool
TrustedScriptURLOrString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
TrustedScriptURLOrString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eTrustedScriptURL: {
      if (!WrapNewBindingNonWrapperCachedObject(cx, scopeObj, mValue.mTrustedScriptURL.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
TrustedScriptURLOrUSVString::TrySetToTrustedScriptURL(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    NonNull<mozilla::dom::TrustedScriptURL>& memberSlot = RawSetAsTrustedScriptURL();
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::TrustedScriptURL, mozilla::dom::TrustedScriptURL>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyTrustedScriptURL();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
TrustedScriptURLOrUSVString::TrySetToTrustedScriptURL(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToTrustedScriptURL(cx, value, tryNext, passedToJSImpl);
}







bool
TrustedScriptURLOrUSVString::TrySetToUSVString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char16_t>& memberSlot = RawSetAsUSVString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
    if (!NormalizeUSVString(memberSlot)) {
      JS_ReportOutOfMemory(cx);
      return false;
    }
  }
  return true;
}








bool
TrustedScriptURLOrUSVString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToTrustedScriptURL(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUSVString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "TrustedScriptURL");
    return false;
  }
  return true;
}

bool
TrustedScriptURLOrUSVString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
TrustedScriptURLOrUSVString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eTrustedScriptURL: {
      if (!WrapNewBindingNonWrapperCachedObject(cx, scopeObj, mValue.mTrustedScriptURL.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eUSVString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mUSVString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
UTF8StringOrUTF8StringSequence::TrySetToUTF8String(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char>& memberSlot = RawSetAsUTF8String();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
UTF8StringOrUTF8StringSequence::TrySetToUTF8StringSequence(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::AutoSequence<nsCString>& memberSlot = RawSetAsUTF8StringSequence();
    JS::ForOfIterator iter(cx);
    if (!iter.init(value, JS::ForOfIterator::AllowNonIterable)) {
      return false;
    }
    if (!iter.valueIsIterable()) {
      DestroyUTF8StringSequence();
      tryNext = true;
      return true;
    }
    binding_detail::AutoSequence<nsCString> &arr = memberSlot;
    JS::Rooted<JS::Value> temp(cx);
    while (true) {
      bool done;
      if (!iter.next(&temp, &done)) {
        return false;
      }
      if (done) {
        break;
      }
      nsCString* slotPtr = arr.AppendElement(mozilla::fallible);
      if (!slotPtr) {
        JS_ReportOutOfMemory(cx);
        return false;
      }
      nsCString& slot = *slotPtr;
      if (!ConvertJSValueToString(cx, temp, eStringify, eStringify, slot)) {
        return false;
      }
    }
  }
  return true;
}

bool
UTF8StringOrUTF8StringSequence::TrySetToUTF8StringSequence(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToUTF8StringSequence(cx, value, tryNext, passedToJSImpl);
}







bool
UTF8StringOrUTF8StringSequence::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToUTF8StringSequence(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUTF8String(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "sequence<USVString>");
    return false;
  }
  return true;
}

bool
UTF8StringOrUTF8StringSequence::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
UTF8StringOrUTF8StringSequence::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eUTF8String: {
      if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8String.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eUTF8StringSequence: {

      uint32_t length = mValue.mUTF8StringSequence.Value().Length();
      JS::Rooted<JSObject*> returnArray(cx, JS::NewArrayObject(cx, length));
      if (!returnArray) {
        return false;
      }
      // Scope for 'tmp'
      {
        JS::Rooted<JS::Value> tmp(cx);
        for (uint32_t sequenceIdx0 = 0; sequenceIdx0 < length; ++sequenceIdx0) {
          // Control block to let us common up the JS_DefineElement calls when there
          // are different ways to succeed at wrapping the object.
          do {
            if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8StringSequence.Value()[sequenceIdx0], &tmp)) {
              return false;
            }
            break;
          } while (false);
          if (!JS_DefineElement(cx, returnArray, sequenceIdx0, tmp,
                                JSPROP_ENUMERATE)) {
            return false;
          }
        }
      }
      rval.setObject(*returnArray);
      return true;
    }
    default: {
      return false;
    }
  }
}

bool
UTF8StringOrUint8Array::TrySetToUTF8String(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    binding_detail::FakeString<char>& memberSlot = RawSetAsUTF8String();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}








bool
UTF8StringOrUint8Array::TrySetToUint8Array(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    RootedSpiderMonkeyInterface<Uint8Array>& memberSlot = RawSetAsUint8Array(cx);
    if (!memberSlot.Init(&value.toObject())) {
      DestroyUint8Array();
      tryNext = true;
      return true;
    }
    if (JS::IsArrayBufferViewShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("Uint8Array branch of (USVString or Uint8Array)");
      return false;
    }
    if (JS::IsLargeArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("Uint8Array branch of (USVString or Uint8Array)");
      return false;
    }
    if (JS::IsResizableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("Uint8Array branch of (USVString or Uint8Array)");
      return false;
    }
    if (JS::IsImmutableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("Uint8Array branch of (USVString or Uint8Array)");
      return false;
    }
  }
  return true;
}

bool
UTF8StringOrUint8Array::TrySetToUint8Array(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToUint8Array(cx, value, tryNext, passedToJSImpl);
}







bool
UTF8StringOrUint8Array::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToUint8Array(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUTF8String(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "Uint8Array");
    return false;
  }
  return true;
}

bool
UTF8StringOrUint8Array::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}


bool
UTF8StringOrUint8Array::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eUTF8String: {
      if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8String.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eUint8Array: {
      rval.setObject(*mValue.mUint8Array.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}


OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String(OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eArrayBufferView: {
      mType = eArrayBufferView;
      mValue.mArrayBufferView.SetValue(std::move(aOther.mValue.mArrayBufferView.Value()));
      break;
    }
    case eArrayBuffer: {
      mType = eArrayBuffer;
      mValue.mArrayBuffer.SetValue(std::move(aOther.mValue.mArrayBuffer.Value()));
      break;
    }
    case eBlob: {
      mType = eBlob;
      mValue.mBlob.SetValue(std::move(aOther.mValue.mBlob.Value()));
      break;
    }
    case eUTF8String: {
      mType = eUTF8String;
      mValue.mUTF8String.SetValue(std::move(aOther.mValue.mUTF8String.Value()));
      break;
    }
  }
}


bool
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToArrayBufferView(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    ArrayBufferView& memberSlot = RawSetAsArrayBufferView();
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBufferView();
      tryNext = true;
      return true;
    }
    if (JS::IsArrayBufferViewShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBufferView branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsLargeArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBufferView branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsResizableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBufferView branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsImmutableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBufferView branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
  }
  return true;
}

bool
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToArrayBufferView(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBufferView(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] ArrayBufferView&
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::RawSetAsArrayBufferView()
{
  if (mType == eArrayBufferView) {
    return mValue.mArrayBufferView.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eArrayBufferView;
  return mValue.mArrayBufferView.SetValue();
}

[[nodiscard]] ArrayBufferView&
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::SetAsArrayBufferView()
{
  if (mType == eArrayBufferView) {
    return mValue.mArrayBufferView.Value();
  }
  Uninit();
  mType = eArrayBufferView;
  return mValue.mArrayBufferView.SetValue();
}


void
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::DestroyArrayBufferView()
{
  MOZ_RELEASE_ASSERT(IsArrayBufferView(), "Wrong type!");
  mValue.mArrayBufferView.Destroy();
  mType = eUninitialized;
}



bool
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToArrayBuffer(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    ArrayBuffer& memberSlot = RawSetAsArrayBuffer();
    if (!memberSlot.Init(&value.toObject())) {
      DestroyArrayBuffer();
      tryNext = true;
      return true;
    }
    if (JS::IsSharedArrayBufferObject(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("ArrayBuffer branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsLargeArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("ArrayBuffer branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsResizableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("ArrayBuffer branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
    if (JS::IsImmutableArrayBufferMaybeShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("ArrayBuffer branch of ((ArrayBufferView or ArrayBuffer) or Blob or USVString)");
      return false;
    }
  }
  return true;
}

bool
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToArrayBuffer(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] ArrayBuffer&
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::RawSetAsArrayBuffer()
{
  if (mType == eArrayBuffer) {
    return mValue.mArrayBuffer.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eArrayBuffer;
  return mValue.mArrayBuffer.SetValue();
}

[[nodiscard]] ArrayBuffer&
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::SetAsArrayBuffer()
{
  if (mType == eArrayBuffer) {
    return mValue.mArrayBuffer.Value();
  }
  Uninit();
  mType = eArrayBuffer;
  return mValue.mArrayBuffer.SetValue();
}


void
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::DestroyArrayBuffer()
{
  MOZ_RELEASE_ASSERT(IsArrayBuffer(), "Wrong type!");
  mValue.mArrayBuffer.Destroy();
  mType = eUninitialized;
}



bool
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToBlob(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::Blob>& memberSlot = RawSetAsBlob();
    static_assert(IsRefcounted<mozilla::dom::Blob>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::Blob, mozilla::dom::Blob>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyBlob();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToBlob(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToBlob(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::Blob>&
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::RawSetAsBlob()
{
  if (mType == eBlob) {
    return mValue.mBlob.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eBlob;
  return mValue.mBlob.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::Blob>&
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::SetAsBlob()
{
  if (mType == eBlob) {
    return mValue.mBlob.Value();
  }
  Uninit();
  mType = eBlob;
  return mValue.mBlob.SetValue();
}


void
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::DestroyBlob()
{
  MOZ_RELEASE_ASSERT(IsBlob(), "Wrong type!");
  mValue.mBlob.Destroy();
  mType = eUninitialized;
}



bool
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TrySetToUTF8String(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsCString& memberSlot = RawSetAsUTF8String();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsCString&
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::RawSetAsUTF8String()
{
  if (mType == eUTF8String) {
    return mValue.mUTF8String.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUTF8String;
  return mValue.mUTF8String.SetValue();
}

[[nodiscard]] nsCString&
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::SetAsUTF8String()
{
  if (mType == eUTF8String) {
    return mValue.mUTF8String.Value();
  }
  Uninit();
  mType = eUTF8String;
  return mValue.mUTF8String.SetValue();
}



void
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::DestroyUTF8String()
{
  MOZ_RELEASE_ASSERT(IsUTF8String(), "Wrong type!");
  mValue.mUTF8String.Destroy();
  mType = eUninitialized;
}



bool
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToArrayBufferView(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToArrayBuffer(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToBlob(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUTF8String(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "ArrayBufferView, ArrayBuffer, Blob");
    return false;
  }
  return true;
}

bool
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eArrayBufferView: {
      DestroyArrayBufferView();
      break;
    }
    case eArrayBuffer: {
      DestroyArrayBuffer();
      break;
    }
    case eBlob: {
      DestroyBlob();
      break;
    }
    case eUTF8String: {
      DestroyUTF8String();
      break;
    }
  }
}

bool
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eArrayBufferView: {
      rval.setObject(*mValue.mArrayBufferView.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    case eArrayBuffer: {
      rval.setObject(*mValue.mArrayBuffer.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    case eBlob: {
      if (!GetOrCreateDOMReflector(cx, mValue.mBlob.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eUTF8String: {
      if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8String.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

void
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::TraceUnion(JSTracer* trc)
{
  switch (mType) {
    case eArrayBufferView: {
      mValue.mArrayBufferView.Value().TraceSelf(trc);
      break;
    }
    case eArrayBuffer: {
      mValue.mArrayBuffer.Value().TraceSelf(trc);
      break;
    }
    default: {
    }
  }
}

OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String&
OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String::operator=(OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String&& aOther)
{
  this->~OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String();
  new (this) OwningArrayBufferViewOrArrayBufferOrBlobOrUTF8String (std::move(aOther));
  return *this;
}



OwningFileOrDirectory::OwningFileOrDirectory(OwningFileOrDirectory&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eFile: {
      mType = eFile;
      mValue.mFile.SetValue(std::move(aOther.mValue.mFile.Value()));
      break;
    }
    case eDirectory: {
      mType = eDirectory;
      mValue.mDirectory.SetValue(std::move(aOther.mValue.mDirectory.Value()));
      break;
    }
  }
}



bool
OwningFileOrDirectory::TrySetToFile(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::File>& memberSlot = RawSetAsFile();
    static_assert(IsRefcounted<mozilla::dom::File>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::File, mozilla::dom::File>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyFile();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningFileOrDirectory::TrySetToFile(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToFile(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::File>&
OwningFileOrDirectory::RawSetAsFile()
{
  if (mType == eFile) {
    return mValue.mFile.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eFile;
  return mValue.mFile.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::File>&
OwningFileOrDirectory::SetAsFile()
{
  if (mType == eFile) {
    return mValue.mFile.Value();
  }
  Uninit();
  mType = eFile;
  return mValue.mFile.SetValue();
}


void
OwningFileOrDirectory::DestroyFile()
{
  MOZ_RELEASE_ASSERT(IsFile(), "Wrong type!");
  mValue.mFile.Destroy();
  mType = eUninitialized;
}



bool
OwningFileOrDirectory::TrySetToDirectory(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::Directory>& memberSlot = RawSetAsDirectory();
    static_assert(IsRefcounted<mozilla::dom::Directory>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::Directory, mozilla::dom::Directory>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyDirectory();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningFileOrDirectory::TrySetToDirectory(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToDirectory(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::Directory>&
OwningFileOrDirectory::RawSetAsDirectory()
{
  if (mType == eDirectory) {
    return mValue.mDirectory.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eDirectory;
  return mValue.mDirectory.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::Directory>&
OwningFileOrDirectory::SetAsDirectory()
{
  if (mType == eDirectory) {
    return mValue.mDirectory.Value();
  }
  Uninit();
  mType = eDirectory;
  return mValue.mDirectory.SetValue();
}


void
OwningFileOrDirectory::DestroyDirectory()
{
  MOZ_RELEASE_ASSERT(IsDirectory(), "Wrong type!");
  mValue.mDirectory.Destroy();
  mType = eUninitialized;
}



bool
OwningFileOrDirectory::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToFile(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToDirectory(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "File, Directory");
    return false;
  }
  return true;
}

bool
OwningFileOrDirectory::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningFileOrDirectory::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eFile: {
      DestroyFile();
      break;
    }
    case eDirectory: {
      DestroyDirectory();
      break;
    }
  }
}

bool
OwningFileOrDirectory::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eFile: {
      if (!GetOrCreateDOMReflector(cx, mValue.mFile.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eDirectory: {
      if (!GetOrCreateDOMReflector(cx, mValue.mDirectory.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningFileOrDirectory&
OwningFileOrDirectory::operator=(OwningFileOrDirectory&& aOther)
{
  this->~OwningFileOrDirectory();
  new (this) OwningFileOrDirectory (std::move(aOther));
  return *this;
}


OwningFileOrDirectory&
OwningFileOrDirectory::operator=(const OwningFileOrDirectory& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eFile: {
      SetAsFile() = aOther.GetAsFile();
      break;
    }
    case eDirectory: {
      SetAsDirectory() = aOther.GetAsDirectory();
      break;
    }
  }
  return *this;
}


OwningFileOrUSVStringOrFormData::OwningFileOrUSVStringOrFormData(OwningFileOrUSVStringOrFormData&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eFile: {
      mType = eFile;
      mValue.mFile.SetValue(std::move(aOther.mValue.mFile.Value()));
      break;
    }
    case eUSVString: {
      mType = eUSVString;
      mValue.mUSVString.SetValue(std::move(aOther.mValue.mUSVString.Value()));
      break;
    }
    case eFormData: {
      mType = eFormData;
      mValue.mFormData.SetValue(std::move(aOther.mValue.mFormData.Value()));
      break;
    }
  }
}



bool
OwningFileOrUSVStringOrFormData::TrySetToFile(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::File>& memberSlot = RawSetAsFile();
    static_assert(IsRefcounted<mozilla::dom::File>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::File, mozilla::dom::File>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyFile();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningFileOrUSVStringOrFormData::TrySetToFile(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToFile(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::File>&
OwningFileOrUSVStringOrFormData::RawSetAsFile()
{
  if (mType == eFile) {
    return mValue.mFile.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eFile;
  return mValue.mFile.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::File>&
OwningFileOrUSVStringOrFormData::SetAsFile()
{
  if (mType == eFile) {
    return mValue.mFile.Value();
  }
  Uninit();
  mType = eFile;
  return mValue.mFile.SetValue();
}


void
OwningFileOrUSVStringOrFormData::DestroyFile()
{
  MOZ_RELEASE_ASSERT(IsFile(), "Wrong type!");
  mValue.mFile.Destroy();
  mType = eUninitialized;
}



bool
OwningFileOrUSVStringOrFormData::TrySetToUSVString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsUSVString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
    if (!NormalizeUSVString(memberSlot)) {
      JS_ReportOutOfMemory(cx);
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningFileOrUSVStringOrFormData::RawSetAsUSVString()
{
  if (mType == eUSVString) {
    return mValue.mUSVString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUSVString;
  return mValue.mUSVString.SetValue();
}

[[nodiscard]] nsString&
OwningFileOrUSVStringOrFormData::SetAsUSVString()
{
  if (mType == eUSVString) {
    return mValue.mUSVString.Value();
  }
  Uninit();
  mType = eUSVString;
  return mValue.mUSVString.SetValue();
}



void
OwningFileOrUSVStringOrFormData::DestroyUSVString()
{
  MOZ_RELEASE_ASSERT(IsUSVString(), "Wrong type!");
  mValue.mUSVString.Destroy();
  mType = eUninitialized;
}



bool
OwningFileOrUSVStringOrFormData::TrySetToFormData(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::FormData>& memberSlot = RawSetAsFormData();
    static_assert(IsRefcounted<mozilla::dom::FormData>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::FormData, mozilla::dom::FormData>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyFormData();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningFileOrUSVStringOrFormData::TrySetToFormData(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToFormData(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::FormData>&
OwningFileOrUSVStringOrFormData::RawSetAsFormData()
{
  if (mType == eFormData) {
    return mValue.mFormData.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eFormData;
  return mValue.mFormData.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::FormData>&
OwningFileOrUSVStringOrFormData::SetAsFormData()
{
  if (mType == eFormData) {
    return mValue.mFormData.Value();
  }
  Uninit();
  mType = eFormData;
  return mValue.mFormData.SetValue();
}


void
OwningFileOrUSVStringOrFormData::DestroyFormData()
{
  MOZ_RELEASE_ASSERT(IsFormData(), "Wrong type!");
  mValue.mFormData.Destroy();
  mType = eUninitialized;
}



bool
OwningFileOrUSVStringOrFormData::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToFile(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToFormData(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUSVString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "File, FormData");
    return false;
  }
  return true;
}

bool
OwningFileOrUSVStringOrFormData::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningFileOrUSVStringOrFormData::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eFile: {
      DestroyFile();
      break;
    }
    case eUSVString: {
      DestroyUSVString();
      break;
    }
    case eFormData: {
      DestroyFormData();
      break;
    }
  }
}

bool
OwningFileOrUSVStringOrFormData::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eFile: {
      if (!GetOrCreateDOMReflector(cx, mValue.mFile.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eUSVString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mUSVString.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eFormData: {
      if (!GetOrCreateDOMReflector(cx, mValue.mFormData.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningFileOrUSVStringOrFormData&
OwningFileOrUSVStringOrFormData::operator=(OwningFileOrUSVStringOrFormData&& aOther)
{
  this->~OwningFileOrUSVStringOrFormData();
  new (this) OwningFileOrUSVStringOrFormData (std::move(aOther));
  return *this;
}


OwningFileOrUSVStringOrFormData&
OwningFileOrUSVStringOrFormData::operator=(const OwningFileOrUSVStringOrFormData& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eFile: {
      SetAsFile() = aOther.GetAsFile();
      break;
    }
    case eUSVString: {
      SetAsUSVString() = aOther.GetAsUSVString();
      break;
    }
    case eFormData: {
      SetAsFormData() = aOther.GetAsFormData();
      break;
    }
  }
  return *this;
}


OwningHTMLCanvasElementOrOffscreenCanvas::OwningHTMLCanvasElementOrOffscreenCanvas(OwningHTMLCanvasElementOrOffscreenCanvas&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eHTMLCanvasElement: {
      mType = eHTMLCanvasElement;
      mValue.mHTMLCanvasElement.SetValue(std::move(aOther.mValue.mHTMLCanvasElement.Value()));
      break;
    }
    case eOffscreenCanvas: {
      mType = eOffscreenCanvas;
      mValue.mOffscreenCanvas.SetValue(std::move(aOther.mValue.mOffscreenCanvas.Value()));
      break;
    }
  }
}



bool
OwningHTMLCanvasElementOrOffscreenCanvas::TrySetToHTMLCanvasElement(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::HTMLCanvasElement>& memberSlot = RawSetAsHTMLCanvasElement();
    static_assert(IsRefcounted<mozilla::dom::HTMLCanvasElement>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::HTMLCanvasElement, mozilla::dom::HTMLCanvasElement>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyHTMLCanvasElement();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningHTMLCanvasElementOrOffscreenCanvas::TrySetToHTMLCanvasElement(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToHTMLCanvasElement(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::HTMLCanvasElement>&
OwningHTMLCanvasElementOrOffscreenCanvas::RawSetAsHTMLCanvasElement()
{
  if (mType == eHTMLCanvasElement) {
    return mValue.mHTMLCanvasElement.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eHTMLCanvasElement;
  return mValue.mHTMLCanvasElement.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::HTMLCanvasElement>&
OwningHTMLCanvasElementOrOffscreenCanvas::SetAsHTMLCanvasElement()
{
  if (mType == eHTMLCanvasElement) {
    return mValue.mHTMLCanvasElement.Value();
  }
  Uninit();
  mType = eHTMLCanvasElement;
  return mValue.mHTMLCanvasElement.SetValue();
}


void
OwningHTMLCanvasElementOrOffscreenCanvas::DestroyHTMLCanvasElement()
{
  MOZ_RELEASE_ASSERT(IsHTMLCanvasElement(), "Wrong type!");
  mValue.mHTMLCanvasElement.Destroy();
  mType = eUninitialized;
}



bool
OwningHTMLCanvasElementOrOffscreenCanvas::TrySetToOffscreenCanvas(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::OffscreenCanvas>& memberSlot = RawSetAsOffscreenCanvas();
    static_assert(IsRefcounted<mozilla::dom::OffscreenCanvas>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::OffscreenCanvas, mozilla::dom::OffscreenCanvas>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyOffscreenCanvas();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningHTMLCanvasElementOrOffscreenCanvas::TrySetToOffscreenCanvas(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToOffscreenCanvas(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::OffscreenCanvas>&
OwningHTMLCanvasElementOrOffscreenCanvas::RawSetAsOffscreenCanvas()
{
  if (mType == eOffscreenCanvas) {
    return mValue.mOffscreenCanvas.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eOffscreenCanvas;
  return mValue.mOffscreenCanvas.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::OffscreenCanvas>&
OwningHTMLCanvasElementOrOffscreenCanvas::SetAsOffscreenCanvas()
{
  if (mType == eOffscreenCanvas) {
    return mValue.mOffscreenCanvas.Value();
  }
  Uninit();
  mType = eOffscreenCanvas;
  return mValue.mOffscreenCanvas.SetValue();
}


void
OwningHTMLCanvasElementOrOffscreenCanvas::DestroyOffscreenCanvas()
{
  MOZ_RELEASE_ASSERT(IsOffscreenCanvas(), "Wrong type!");
  mValue.mOffscreenCanvas.Destroy();
  mType = eUninitialized;
}



bool
OwningHTMLCanvasElementOrOffscreenCanvas::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToHTMLCanvasElement(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToOffscreenCanvas(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "HTMLCanvasElement, OffscreenCanvas");
    return false;
  }
  return true;
}

bool
OwningHTMLCanvasElementOrOffscreenCanvas::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningHTMLCanvasElementOrOffscreenCanvas::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eHTMLCanvasElement: {
      DestroyHTMLCanvasElement();
      break;
    }
    case eOffscreenCanvas: {
      DestroyOffscreenCanvas();
      break;
    }
  }
}

bool
OwningHTMLCanvasElementOrOffscreenCanvas::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eHTMLCanvasElement: {
      if (!GetOrCreateDOMReflector(cx, mValue.mHTMLCanvasElement.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eOffscreenCanvas: {
      if (!GetOrCreateDOMReflector(cx, mValue.mOffscreenCanvas.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningHTMLCanvasElementOrOffscreenCanvas&
OwningHTMLCanvasElementOrOffscreenCanvas::operator=(OwningHTMLCanvasElementOrOffscreenCanvas&& aOther)
{
  this->~OwningHTMLCanvasElementOrOffscreenCanvas();
  new (this) OwningHTMLCanvasElementOrOffscreenCanvas (std::move(aOther));
  return *this;
}


OwningHTMLCanvasElementOrOffscreenCanvas&
OwningHTMLCanvasElementOrOffscreenCanvas::operator=(const OwningHTMLCanvasElementOrOffscreenCanvas& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eHTMLCanvasElement: {
      SetAsHTMLCanvasElement() = aOther.GetAsHTMLCanvasElement();
      break;
    }
    case eOffscreenCanvas: {
      SetAsOffscreenCanvas() = aOther.GetAsOffscreenCanvas();
      break;
    }
  }
  return *this;
}


OwningHTMLElementOrLong::OwningHTMLElementOrLong(OwningHTMLElementOrLong&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eHTMLElement: {
      mType = eHTMLElement;
      mValue.mHTMLElement.SetValue(std::move(aOther.mValue.mHTMLElement.Value()));
      break;
    }
    case eLong: {
      mType = eLong;
      mValue.mLong.SetValue(std::move(aOther.mValue.mLong.Value()));
      break;
    }
  }
}



bool
OwningHTMLElementOrLong::TrySetToHTMLElement(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<nsGenericHTMLElement>& memberSlot = RawSetAsHTMLElement();
    static_assert(IsRefcounted<nsGenericHTMLElement>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::HTMLElement, nsGenericHTMLElement>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyHTMLElement();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningHTMLElementOrLong::TrySetToHTMLElement(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToHTMLElement(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<nsGenericHTMLElement>&
OwningHTMLElementOrLong::RawSetAsHTMLElement()
{
  if (mType == eHTMLElement) {
    return mValue.mHTMLElement.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eHTMLElement;
  return mValue.mHTMLElement.SetValue();
}

[[nodiscard]] OwningNonNull<nsGenericHTMLElement>&
OwningHTMLElementOrLong::SetAsHTMLElement()
{
  if (mType == eHTMLElement) {
    return mValue.mHTMLElement.Value();
  }
  Uninit();
  mType = eHTMLElement;
  return mValue.mHTMLElement.SetValue();
}


void
OwningHTMLElementOrLong::DestroyHTMLElement()
{
  MOZ_RELEASE_ASSERT(IsHTMLElement(), "Wrong type!");
  mValue.mHTMLElement.Destroy();
  mType = eUninitialized;
}



bool
OwningHTMLElementOrLong::TrySetToLong(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    int32_t& memberSlot = RawSetAsLong();
    if (!ValueToPrimitive<int32_t, eDefault>(cx, value, "Long branch of (HTMLElement or long)", &memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] int32_t&
OwningHTMLElementOrLong::RawSetAsLong()
{
  if (mType == eLong) {
    return mValue.mLong.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eLong;
  return mValue.mLong.SetValue();
}

[[nodiscard]] int32_t&
OwningHTMLElementOrLong::SetAsLong()
{
  if (mType == eLong) {
    return mValue.mLong.Value();
  }
  Uninit();
  mType = eLong;
  return mValue.mLong.SetValue();
}


void
OwningHTMLElementOrLong::DestroyLong()
{
  MOZ_RELEASE_ASSERT(IsLong(), "Wrong type!");
  mValue.mLong.Destroy();
  mType = eUninitialized;
}



bool
OwningHTMLElementOrLong::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToHTMLElement(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToLong(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "HTMLElement");
    return false;
  }
  return true;
}

bool
OwningHTMLElementOrLong::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningHTMLElementOrLong::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eHTMLElement: {
      DestroyHTMLElement();
      break;
    }
    case eLong: {
      DestroyLong();
      break;
    }
  }
}

bool
OwningHTMLElementOrLong::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eHTMLElement: {
      if (!GetOrCreateDOMReflector(cx, mValue.mHTMLElement.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eLong: {
      rval.setInt32(int32_t(mValue.mLong.Value()));
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningHTMLElementOrLong&
OwningHTMLElementOrLong::operator=(OwningHTMLElementOrLong&& aOther)
{
  this->~OwningHTMLElementOrLong();
  new (this) OwningHTMLElementOrLong (std::move(aOther));
  return *this;
}


OwningHTMLElementOrLong&
OwningHTMLElementOrLong::operator=(const OwningHTMLElementOrLong& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eHTMLElement: {
      SetAsHTMLElement() = aOther.GetAsHTMLElement();
      break;
    }
    case eLong: {
      SetAsLong() = aOther.GetAsLong();
      break;
    }
  }
  return *this;
}


OwningHTMLOptionElementOrHTMLOptGroupElement::OwningHTMLOptionElementOrHTMLOptGroupElement(OwningHTMLOptionElementOrHTMLOptGroupElement&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eHTMLOptionElement: {
      mType = eHTMLOptionElement;
      mValue.mHTMLOptionElement.SetValue(std::move(aOther.mValue.mHTMLOptionElement.Value()));
      break;
    }
    case eHTMLOptGroupElement: {
      mType = eHTMLOptGroupElement;
      mValue.mHTMLOptGroupElement.SetValue(std::move(aOther.mValue.mHTMLOptGroupElement.Value()));
      break;
    }
  }
}



bool
OwningHTMLOptionElementOrHTMLOptGroupElement::TrySetToHTMLOptionElement(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::HTMLOptionElement>& memberSlot = RawSetAsHTMLOptionElement();
    static_assert(IsRefcounted<mozilla::dom::HTMLOptionElement>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::HTMLOptionElement, mozilla::dom::HTMLOptionElement>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyHTMLOptionElement();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningHTMLOptionElementOrHTMLOptGroupElement::TrySetToHTMLOptionElement(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToHTMLOptionElement(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::HTMLOptionElement>&
OwningHTMLOptionElementOrHTMLOptGroupElement::RawSetAsHTMLOptionElement()
{
  if (mType == eHTMLOptionElement) {
    return mValue.mHTMLOptionElement.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eHTMLOptionElement;
  return mValue.mHTMLOptionElement.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::HTMLOptionElement>&
OwningHTMLOptionElementOrHTMLOptGroupElement::SetAsHTMLOptionElement()
{
  if (mType == eHTMLOptionElement) {
    return mValue.mHTMLOptionElement.Value();
  }
  Uninit();
  mType = eHTMLOptionElement;
  return mValue.mHTMLOptionElement.SetValue();
}


void
OwningHTMLOptionElementOrHTMLOptGroupElement::DestroyHTMLOptionElement()
{
  MOZ_RELEASE_ASSERT(IsHTMLOptionElement(), "Wrong type!");
  mValue.mHTMLOptionElement.Destroy();
  mType = eUninitialized;
}



bool
OwningHTMLOptionElementOrHTMLOptGroupElement::TrySetToHTMLOptGroupElement(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::HTMLOptGroupElement>& memberSlot = RawSetAsHTMLOptGroupElement();
    static_assert(IsRefcounted<mozilla::dom::HTMLOptGroupElement>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::HTMLOptGroupElement, mozilla::dom::HTMLOptGroupElement>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyHTMLOptGroupElement();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningHTMLOptionElementOrHTMLOptGroupElement::TrySetToHTMLOptGroupElement(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToHTMLOptGroupElement(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::HTMLOptGroupElement>&
OwningHTMLOptionElementOrHTMLOptGroupElement::RawSetAsHTMLOptGroupElement()
{
  if (mType == eHTMLOptGroupElement) {
    return mValue.mHTMLOptGroupElement.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eHTMLOptGroupElement;
  return mValue.mHTMLOptGroupElement.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::HTMLOptGroupElement>&
OwningHTMLOptionElementOrHTMLOptGroupElement::SetAsHTMLOptGroupElement()
{
  if (mType == eHTMLOptGroupElement) {
    return mValue.mHTMLOptGroupElement.Value();
  }
  Uninit();
  mType = eHTMLOptGroupElement;
  return mValue.mHTMLOptGroupElement.SetValue();
}


void
OwningHTMLOptionElementOrHTMLOptGroupElement::DestroyHTMLOptGroupElement()
{
  MOZ_RELEASE_ASSERT(IsHTMLOptGroupElement(), "Wrong type!");
  mValue.mHTMLOptGroupElement.Destroy();
  mType = eUninitialized;
}



bool
OwningHTMLOptionElementOrHTMLOptGroupElement::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToHTMLOptionElement(cx, value, tryNext, passedToJSImpl)) || !tryNext ||
           (failed = !TrySetToHTMLOptGroupElement(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "HTMLOptionElement, HTMLOptGroupElement");
    return false;
  }
  return true;
}

bool
OwningHTMLOptionElementOrHTMLOptGroupElement::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningHTMLOptionElementOrHTMLOptGroupElement::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eHTMLOptionElement: {
      DestroyHTMLOptionElement();
      break;
    }
    case eHTMLOptGroupElement: {
      DestroyHTMLOptGroupElement();
      break;
    }
  }
}

bool
OwningHTMLOptionElementOrHTMLOptGroupElement::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eHTMLOptionElement: {
      if (!GetOrCreateDOMReflector(cx, mValue.mHTMLOptionElement.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eHTMLOptGroupElement: {
      if (!GetOrCreateDOMReflector(cx, mValue.mHTMLOptGroupElement.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningHTMLOptionElementOrHTMLOptGroupElement&
OwningHTMLOptionElementOrHTMLOptGroupElement::operator=(OwningHTMLOptionElementOrHTMLOptGroupElement&& aOther)
{
  this->~OwningHTMLOptionElementOrHTMLOptGroupElement();
  new (this) OwningHTMLOptionElementOrHTMLOptGroupElement (std::move(aOther));
  return *this;
}


OwningHTMLOptionElementOrHTMLOptGroupElement&
OwningHTMLOptionElementOrHTMLOptGroupElement::operator=(const OwningHTMLOptionElementOrHTMLOptGroupElement& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eHTMLOptionElement: {
      SetAsHTMLOptionElement() = aOther.GetAsHTMLOptionElement();
      break;
    }
    case eHTMLOptGroupElement: {
      SetAsHTMLOptGroupElement() = aOther.GetAsHTMLOptGroupElement();
      break;
    }
  }
  return *this;
}


OwningNodeOrString::OwningNodeOrString(OwningNodeOrString&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eNode: {
      mType = eNode;
      mValue.mNode.SetValue(std::move(aOther.mValue.mNode.Value()));
      break;
    }
    case eString: {
      mType = eString;
      mValue.mString.SetValue(std::move(aOther.mValue.mString.Value()));
      break;
    }
  }
}



bool
OwningNodeOrString::TrySetToNode(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<nsINode>& memberSlot = RawSetAsNode();
    static_assert(IsRefcounted<nsINode>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::Node, nsINode>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyNode();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningNodeOrString::TrySetToNode(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToNode(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<nsINode>&
OwningNodeOrString::RawSetAsNode()
{
  if (mType == eNode) {
    return mValue.mNode.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eNode;
  return mValue.mNode.SetValue();
}

[[nodiscard]] OwningNonNull<nsINode>&
OwningNodeOrString::SetAsNode()
{
  if (mType == eNode) {
    return mValue.mNode.Value();
  }
  Uninit();
  mType = eNode;
  return mValue.mNode.SetValue();
}


void
OwningNodeOrString::DestroyNode()
{
  MOZ_RELEASE_ASSERT(IsNode(), "Wrong type!");
  mValue.mNode.Destroy();
  mType = eUninitialized;
}



bool
OwningNodeOrString::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningNodeOrString::RawSetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eString;
  return mValue.mString.SetValue();
}

[[nodiscard]] nsString&
OwningNodeOrString::SetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  Uninit();
  mType = eString;
  return mValue.mString.SetValue();
}



void
OwningNodeOrString::DestroyString()
{
  MOZ_RELEASE_ASSERT(IsString(), "Wrong type!");
  mValue.mString.Destroy();
  mType = eUninitialized;
}



bool
OwningNodeOrString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToNode(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "Node");
    return false;
  }
  return true;
}

bool
OwningNodeOrString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningNodeOrString::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eNode: {
      DestroyNode();
      break;
    }
    case eString: {
      DestroyString();
      break;
    }
  }
}

bool
OwningNodeOrString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eNode: {
      if (!GetOrCreateDOMReflector(cx, mValue.mNode.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningNodeOrString&
OwningNodeOrString::operator=(OwningNodeOrString&& aOther)
{
  this->~OwningNodeOrString();
  new (this) OwningNodeOrString (std::move(aOther));
  return *this;
}


OwningNodeOrString&
OwningNodeOrString::operator=(const OwningNodeOrString& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eNode: {
      SetAsNode() = aOther.GetAsNode();
      break;
    }
    case eString: {
      SetAsString() = aOther.GetAsString();
      break;
    }
  }
  return *this;
}


OwningStringOrBooleanOrObject::OwningStringOrBooleanOrObject(OwningStringOrBooleanOrObject&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eString: {
      mType = eString;
      mValue.mString.SetValue(std::move(aOther.mValue.mString.Value()));
      break;
    }
    case eBoolean: {
      mType = eBoolean;
      mValue.mBoolean.SetValue(std::move(aOther.mValue.mBoolean.Value()));
      break;
    }
    case eObject: {
      mType = eObject;
      mValue.mObject.SetValue(std::move(aOther.mValue.mObject.Value()));
      break;
    }
  }
}


bool
OwningStringOrBooleanOrObject::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningStringOrBooleanOrObject::RawSetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eString;
  return mValue.mString.SetValue();
}

[[nodiscard]] nsString&
OwningStringOrBooleanOrObject::SetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  Uninit();
  mType = eString;
  return mValue.mString.SetValue();
}



void
OwningStringOrBooleanOrObject::DestroyString()
{
  MOZ_RELEASE_ASSERT(IsString(), "Wrong type!");
  mValue.mString.Destroy();
  mType = eUninitialized;
}



bool
OwningStringOrBooleanOrObject::TrySetToBoolean(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    bool& memberSlot = RawSetAsBoolean();
    if (!ValueToPrimitive<bool, eDefault>(cx, value, "Boolean branch of (DOMString or boolean or object)", &memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] bool&
OwningStringOrBooleanOrObject::RawSetAsBoolean()
{
  if (mType == eBoolean) {
    return mValue.mBoolean.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eBoolean;
  return mValue.mBoolean.SetValue();
}

[[nodiscard]] bool&
OwningStringOrBooleanOrObject::SetAsBoolean()
{
  if (mType == eBoolean) {
    return mValue.mBoolean.Value();
  }
  Uninit();
  mType = eBoolean;
  return mValue.mBoolean.SetValue();
}


void
OwningStringOrBooleanOrObject::DestroyBoolean()
{
  MOZ_RELEASE_ASSERT(IsBoolean(), "Wrong type!");
  mValue.mBoolean.Destroy();
  mType = eUninitialized;
}




[[nodiscard]] JSObject*&
OwningStringOrBooleanOrObject::RawSetAsObject()
{
  if (mType == eObject) {
    return mValue.mObject.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eObject;
  return mValue.mObject.SetValue();
}

[[nodiscard]] JSObject*&
OwningStringOrBooleanOrObject::SetAsObject()
{
  if (mType == eObject) {
    return mValue.mObject.Value();
  }
  Uninit();
  mType = eObject;
  return mValue.mObject.SetValue();
}


void
OwningStringOrBooleanOrObject::DestroyObject()
{
  MOZ_RELEASE_ASSERT(IsObject(), "Wrong type!");
  mValue.mObject.Destroy();
  mType = eUninitialized;
}



bool
OwningStringOrBooleanOrObject::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    if (!SetToObject(cx, &value.toObject(), passedToJSImpl)) {
      return false;
    }
    done = true;
  } else {
    do {
      if (value.isBoolean()) {
        done = (failed = !TrySetToBoolean(cx, value, tryNext)) || !tryNext;
        break;
      }
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "object");
    return false;
  }
  return true;
}

bool
OwningStringOrBooleanOrObject::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningStringOrBooleanOrObject::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eString: {
      DestroyString();
      break;
    }
    case eBoolean: {
      DestroyBoolean();
      break;
    }
    case eObject: {
      DestroyObject();
      break;
    }
  }
}

bool
OwningStringOrBooleanOrObject::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eBoolean: {
      rval.setBoolean(mValue.mBoolean.Value());
      return true;
    }
    case eObject: {
      JS::ExposeObjectToActiveJS(mValue.mObject.Value());
      rval.setObject(*mValue.mObject.Value());
      if (!MaybeWrapObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

void
OwningStringOrBooleanOrObject::TraceUnion(JSTracer* trc)
{
  switch (mType) {
    case eObject: {
      JS::TraceRoot(trc, &mValue.mObject.Value(), "mValue.mObject");
      break;
    }
    default: {
    }
  }
}

OwningStringOrBooleanOrObject&
OwningStringOrBooleanOrObject::operator=(OwningStringOrBooleanOrObject&& aOther)
{
  this->~OwningStringOrBooleanOrObject();
  new (this) OwningStringOrBooleanOrObject (std::move(aOther));
  return *this;
}



OwningStringOrStringSequence::OwningStringOrStringSequence(OwningStringOrStringSequence&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eString: {
      mType = eString;
      mValue.mString.SetValue(std::move(aOther.mValue.mString.Value()));
      break;
    }
    case eStringSequence: {
      mType = eStringSequence;
      mValue.mStringSequence.SetValue(std::move(aOther.mValue.mStringSequence.Value()));
      break;
    }
  }
}



bool
OwningStringOrStringSequence::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningStringOrStringSequence::RawSetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eString;
  return mValue.mString.SetValue();
}

[[nodiscard]] nsString&
OwningStringOrStringSequence::SetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  Uninit();
  mType = eString;
  return mValue.mString.SetValue();
}



void
OwningStringOrStringSequence::DestroyString()
{
  MOZ_RELEASE_ASSERT(IsString(), "Wrong type!");
  mValue.mString.Destroy();
  mType = eUninitialized;
}



bool
OwningStringOrStringSequence::TrySetToStringSequence(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    Sequence<nsString>& memberSlot = RawSetAsStringSequence();
    JS::ForOfIterator iter(cx);
    if (!iter.init(value, JS::ForOfIterator::AllowNonIterable)) {
      return false;
    }
    if (!iter.valueIsIterable()) {
      DestroyStringSequence();
      tryNext = true;
      return true;
    }
    Sequence<nsString> &arr = memberSlot;
    JS::Rooted<JS::Value> temp(cx);
    while (true) {
      bool done;
      if (!iter.next(&temp, &done)) {
        return false;
      }
      if (done) {
        break;
      }
      nsString* slotPtr = arr.AppendElement(mozilla::fallible);
      if (!slotPtr) {
        JS_ReportOutOfMemory(cx);
        return false;
      }
      nsString& slot = *slotPtr;
      if (!ConvertJSValueToString(cx, temp, eStringify, eStringify, slot)) {
        return false;
      }
    }
  }
  return true;
}

bool
OwningStringOrStringSequence::TrySetToStringSequence(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToStringSequence(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] Sequence<nsString>&
OwningStringOrStringSequence::RawSetAsStringSequence()
{
  if (mType == eStringSequence) {
    return mValue.mStringSequence.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eStringSequence;
  return mValue.mStringSequence.SetValue();
}

[[nodiscard]] Sequence<nsString>&
OwningStringOrStringSequence::SetAsStringSequence()
{
  if (mType == eStringSequence) {
    return mValue.mStringSequence.Value();
  }
  Uninit();
  mType = eStringSequence;
  return mValue.mStringSequence.SetValue();
}


void
OwningStringOrStringSequence::DestroyStringSequence()
{
  MOZ_RELEASE_ASSERT(IsStringSequence(), "Wrong type!");
  mValue.mStringSequence.Destroy();
  mType = eUninitialized;
}



bool
OwningStringOrStringSequence::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToStringSequence(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "sequence<DOMString>");
    return false;
  }
  return true;
}

bool
OwningStringOrStringSequence::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningStringOrStringSequence::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eString: {
      DestroyString();
      break;
    }
    case eStringSequence: {
      DestroyStringSequence();
      break;
    }
  }
}

bool
OwningStringOrStringSequence::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eStringSequence: {

      uint32_t length = mValue.mStringSequence.Value().Length();
      JS::Rooted<JSObject*> returnArray(cx, JS::NewArrayObject(cx, length));
      if (!returnArray) {
        return false;
      }
      // Scope for 'tmp'
      {
        JS::Rooted<JS::Value> tmp(cx);
        for (uint32_t sequenceIdx0 = 0; sequenceIdx0 < length; ++sequenceIdx0) {
          // Control block to let us common up the JS_DefineElement calls when there
          // are different ways to succeed at wrapping the object.
          do {
            if (!xpc::NonVoidStringToJsval(cx, mValue.mStringSequence.Value()[sequenceIdx0], &tmp)) {
              return false;
            }
            break;
          } while (false);
          if (!JS_DefineElement(cx, returnArray, sequenceIdx0, tmp,
                                JSPROP_ENUMERATE)) {
            return false;
          }
        }
      }
      rval.setObject(*returnArray);
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningStringOrStringSequence&
OwningStringOrStringSequence::operator=(OwningStringOrStringSequence&& aOther)
{
  this->~OwningStringOrStringSequence();
  new (this) OwningStringOrStringSequence (std::move(aOther));
  return *this;
}


OwningStringOrStringSequence&
OwningStringOrStringSequence::operator=(const OwningStringOrStringSequence& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eString: {
      SetAsString() = aOther.GetAsString();
      break;
    }
    case eStringSequence: {
      SetAsStringSequence() = aOther.GetAsStringSequence();
      break;
    }
  }
  return *this;
}


OwningTrustedHTMLOrNullIsEmptyString::OwningTrustedHTMLOrNullIsEmptyString(OwningTrustedHTMLOrNullIsEmptyString&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eTrustedHTML: {
      mType = eTrustedHTML;
      mValue.mTrustedHTML.SetValue(std::move(aOther.mValue.mTrustedHTML.Value()));
      break;
    }
    case eNullIsEmptyString: {
      mType = eNullIsEmptyString;
      mValue.mNullIsEmptyString.SetValue(std::move(aOther.mValue.mNullIsEmptyString.Value()));
      break;
    }
  }
}



bool
OwningTrustedHTMLOrNullIsEmptyString::TrySetToTrustedHTML(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::TrustedHTML>& memberSlot = RawSetAsTrustedHTML();
    static_assert(IsRefcounted<mozilla::dom::TrustedHTML>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::TrustedHTML, mozilla::dom::TrustedHTML>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyTrustedHTML();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningTrustedHTMLOrNullIsEmptyString::TrySetToTrustedHTML(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToTrustedHTML(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::TrustedHTML>&
OwningTrustedHTMLOrNullIsEmptyString::RawSetAsTrustedHTML()
{
  if (mType == eTrustedHTML) {
    return mValue.mTrustedHTML.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eTrustedHTML;
  return mValue.mTrustedHTML.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::TrustedHTML>&
OwningTrustedHTMLOrNullIsEmptyString::SetAsTrustedHTML()
{
  if (mType == eTrustedHTML) {
    return mValue.mTrustedHTML.Value();
  }
  Uninit();
  mType = eTrustedHTML;
  return mValue.mTrustedHTML.SetValue();
}


void
OwningTrustedHTMLOrNullIsEmptyString::DestroyTrustedHTML()
{
  MOZ_RELEASE_ASSERT(IsTrustedHTML(), "Wrong type!");
  mValue.mTrustedHTML.Destroy();
  mType = eUninitialized;
}



bool
OwningTrustedHTMLOrNullIsEmptyString::TrySetToNullIsEmptyString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsNullIsEmptyString();
    if (!ConvertJSValueToString(cx, value, eEmpty, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningTrustedHTMLOrNullIsEmptyString::RawSetAsNullIsEmptyString()
{
  if (mType == eNullIsEmptyString) {
    return mValue.mNullIsEmptyString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eNullIsEmptyString;
  return mValue.mNullIsEmptyString.SetValue();
}

[[nodiscard]] nsString&
OwningTrustedHTMLOrNullIsEmptyString::SetAsNullIsEmptyString()
{
  if (mType == eNullIsEmptyString) {
    return mValue.mNullIsEmptyString.Value();
  }
  Uninit();
  mType = eNullIsEmptyString;
  return mValue.mNullIsEmptyString.SetValue();
}



void
OwningTrustedHTMLOrNullIsEmptyString::DestroyNullIsEmptyString()
{
  MOZ_RELEASE_ASSERT(IsNullIsEmptyString(), "Wrong type!");
  mValue.mNullIsEmptyString.Destroy();
  mType = eUninitialized;
}



bool
OwningTrustedHTMLOrNullIsEmptyString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToTrustedHTML(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToNullIsEmptyString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "TrustedHTML");
    return false;
  }
  return true;
}

bool
OwningTrustedHTMLOrNullIsEmptyString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningTrustedHTMLOrNullIsEmptyString::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eTrustedHTML: {
      DestroyTrustedHTML();
      break;
    }
    case eNullIsEmptyString: {
      DestroyNullIsEmptyString();
      break;
    }
  }
}

bool
OwningTrustedHTMLOrNullIsEmptyString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eTrustedHTML: {
      if (!WrapNewBindingNonWrapperCachedObject(cx, scopeObj, mValue.mTrustedHTML.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eNullIsEmptyString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mNullIsEmptyString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningTrustedHTMLOrNullIsEmptyString&
OwningTrustedHTMLOrNullIsEmptyString::operator=(OwningTrustedHTMLOrNullIsEmptyString&& aOther)
{
  this->~OwningTrustedHTMLOrNullIsEmptyString();
  new (this) OwningTrustedHTMLOrNullIsEmptyString (std::move(aOther));
  return *this;
}


OwningTrustedHTMLOrNullIsEmptyString&
OwningTrustedHTMLOrNullIsEmptyString::operator=(const OwningTrustedHTMLOrNullIsEmptyString& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eTrustedHTML: {
      SetAsTrustedHTML() = aOther.GetAsTrustedHTML();
      break;
    }
    case eNullIsEmptyString: {
      SetAsNullIsEmptyString() = aOther.GetAsNullIsEmptyString();
      break;
    }
  }
  return *this;
}


OwningTrustedHTMLOrString::OwningTrustedHTMLOrString(OwningTrustedHTMLOrString&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eTrustedHTML: {
      mType = eTrustedHTML;
      mValue.mTrustedHTML.SetValue(std::move(aOther.mValue.mTrustedHTML.Value()));
      break;
    }
    case eString: {
      mType = eString;
      mValue.mString.SetValue(std::move(aOther.mValue.mString.Value()));
      break;
    }
  }
}



bool
OwningTrustedHTMLOrString::TrySetToTrustedHTML(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::TrustedHTML>& memberSlot = RawSetAsTrustedHTML();
    static_assert(IsRefcounted<mozilla::dom::TrustedHTML>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::TrustedHTML, mozilla::dom::TrustedHTML>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyTrustedHTML();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningTrustedHTMLOrString::TrySetToTrustedHTML(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToTrustedHTML(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::TrustedHTML>&
OwningTrustedHTMLOrString::RawSetAsTrustedHTML()
{
  if (mType == eTrustedHTML) {
    return mValue.mTrustedHTML.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eTrustedHTML;
  return mValue.mTrustedHTML.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::TrustedHTML>&
OwningTrustedHTMLOrString::SetAsTrustedHTML()
{
  if (mType == eTrustedHTML) {
    return mValue.mTrustedHTML.Value();
  }
  Uninit();
  mType = eTrustedHTML;
  return mValue.mTrustedHTML.SetValue();
}


void
OwningTrustedHTMLOrString::DestroyTrustedHTML()
{
  MOZ_RELEASE_ASSERT(IsTrustedHTML(), "Wrong type!");
  mValue.mTrustedHTML.Destroy();
  mType = eUninitialized;
}



bool
OwningTrustedHTMLOrString::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningTrustedHTMLOrString::RawSetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eString;
  return mValue.mString.SetValue();
}

[[nodiscard]] nsString&
OwningTrustedHTMLOrString::SetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  Uninit();
  mType = eString;
  return mValue.mString.SetValue();
}



void
OwningTrustedHTMLOrString::DestroyString()
{
  MOZ_RELEASE_ASSERT(IsString(), "Wrong type!");
  mValue.mString.Destroy();
  mType = eUninitialized;
}



bool
OwningTrustedHTMLOrString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToTrustedHTML(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "TrustedHTML");
    return false;
  }
  return true;
}

bool
OwningTrustedHTMLOrString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningTrustedHTMLOrString::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eTrustedHTML: {
      DestroyTrustedHTML();
      break;
    }
    case eString: {
      DestroyString();
      break;
    }
  }
}

bool
OwningTrustedHTMLOrString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eTrustedHTML: {
      if (!WrapNewBindingNonWrapperCachedObject(cx, scopeObj, mValue.mTrustedHTML.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningTrustedHTMLOrString&
OwningTrustedHTMLOrString::operator=(OwningTrustedHTMLOrString&& aOther)
{
  this->~OwningTrustedHTMLOrString();
  new (this) OwningTrustedHTMLOrString (std::move(aOther));
  return *this;
}


OwningTrustedHTMLOrString&
OwningTrustedHTMLOrString::operator=(const OwningTrustedHTMLOrString& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eTrustedHTML: {
      SetAsTrustedHTML() = aOther.GetAsTrustedHTML();
      break;
    }
    case eString: {
      SetAsString() = aOther.GetAsString();
      break;
    }
  }
  return *this;
}


OwningTrustedScriptURLOrString::OwningTrustedScriptURLOrString(OwningTrustedScriptURLOrString&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eTrustedScriptURL: {
      mType = eTrustedScriptURL;
      mValue.mTrustedScriptURL.SetValue(std::move(aOther.mValue.mTrustedScriptURL.Value()));
      break;
    }
    case eString: {
      mType = eString;
      mValue.mString.SetValue(std::move(aOther.mValue.mString.Value()));
      break;
    }
  }
}



bool
OwningTrustedScriptURLOrString::TrySetToTrustedScriptURL(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::TrustedScriptURL>& memberSlot = RawSetAsTrustedScriptURL();
    static_assert(IsRefcounted<mozilla::dom::TrustedScriptURL>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::TrustedScriptURL, mozilla::dom::TrustedScriptURL>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyTrustedScriptURL();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningTrustedScriptURLOrString::TrySetToTrustedScriptURL(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToTrustedScriptURL(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::TrustedScriptURL>&
OwningTrustedScriptURLOrString::RawSetAsTrustedScriptURL()
{
  if (mType == eTrustedScriptURL) {
    return mValue.mTrustedScriptURL.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eTrustedScriptURL;
  return mValue.mTrustedScriptURL.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::TrustedScriptURL>&
OwningTrustedScriptURLOrString::SetAsTrustedScriptURL()
{
  if (mType == eTrustedScriptURL) {
    return mValue.mTrustedScriptURL.Value();
  }
  Uninit();
  mType = eTrustedScriptURL;
  return mValue.mTrustedScriptURL.SetValue();
}


void
OwningTrustedScriptURLOrString::DestroyTrustedScriptURL()
{
  MOZ_RELEASE_ASSERT(IsTrustedScriptURL(), "Wrong type!");
  mValue.mTrustedScriptURL.Destroy();
  mType = eUninitialized;
}



bool
OwningTrustedScriptURLOrString::TrySetToString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningTrustedScriptURLOrString::RawSetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eString;
  return mValue.mString.SetValue();
}

[[nodiscard]] nsString&
OwningTrustedScriptURLOrString::SetAsString()
{
  if (mType == eString) {
    return mValue.mString.Value();
  }
  Uninit();
  mType = eString;
  return mValue.mString.SetValue();
}



void
OwningTrustedScriptURLOrString::DestroyString()
{
  MOZ_RELEASE_ASSERT(IsString(), "Wrong type!");
  mValue.mString.Destroy();
  mType = eUninitialized;
}



bool
OwningTrustedScriptURLOrString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToTrustedScriptURL(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "TrustedScriptURL");
    return false;
  }
  return true;
}

bool
OwningTrustedScriptURLOrString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningTrustedScriptURLOrString::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eTrustedScriptURL: {
      DestroyTrustedScriptURL();
      break;
    }
    case eString: {
      DestroyString();
      break;
    }
  }
}

bool
OwningTrustedScriptURLOrString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eTrustedScriptURL: {
      if (!WrapNewBindingNonWrapperCachedObject(cx, scopeObj, mValue.mTrustedScriptURL.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningTrustedScriptURLOrString&
OwningTrustedScriptURLOrString::operator=(OwningTrustedScriptURLOrString&& aOther)
{
  this->~OwningTrustedScriptURLOrString();
  new (this) OwningTrustedScriptURLOrString (std::move(aOther));
  return *this;
}


OwningTrustedScriptURLOrString&
OwningTrustedScriptURLOrString::operator=(const OwningTrustedScriptURLOrString& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eTrustedScriptURL: {
      SetAsTrustedScriptURL() = aOther.GetAsTrustedScriptURL();
      break;
    }
    case eString: {
      SetAsString() = aOther.GetAsString();
      break;
    }
  }
  return *this;
}


OwningTrustedScriptURLOrUSVString::OwningTrustedScriptURLOrUSVString(OwningTrustedScriptURLOrUSVString&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eTrustedScriptURL: {
      mType = eTrustedScriptURL;
      mValue.mTrustedScriptURL.SetValue(std::move(aOther.mValue.mTrustedScriptURL.Value()));
      break;
    }
    case eUSVString: {
      mType = eUSVString;
      mValue.mUSVString.SetValue(std::move(aOther.mValue.mUSVString.Value()));
      break;
    }
  }
}



bool
OwningTrustedScriptURLOrUSVString::TrySetToTrustedScriptURL(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    OwningNonNull<mozilla::dom::TrustedScriptURL>& memberSlot = RawSetAsTrustedScriptURL();
    static_assert(IsRefcounted<mozilla::dom::TrustedScriptURL>::value, "We can only store refcounted classes.");
    {
      // Our JSContext should be in the right global to do unwrapping in.
      nsresult rv = UnwrapObject<prototypes::id::TrustedScriptURL, mozilla::dom::TrustedScriptURL>(value, memberSlot, cx);
      if (NS_FAILED(rv)) {
        DestroyTrustedScriptURL();
        tryNext = true;
        return true;
      }
    }
  }
  return true;
}

bool
OwningTrustedScriptURLOrUSVString::TrySetToTrustedScriptURL(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToTrustedScriptURL(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] OwningNonNull<mozilla::dom::TrustedScriptURL>&
OwningTrustedScriptURLOrUSVString::RawSetAsTrustedScriptURL()
{
  if (mType == eTrustedScriptURL) {
    return mValue.mTrustedScriptURL.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eTrustedScriptURL;
  return mValue.mTrustedScriptURL.SetValue();
}

[[nodiscard]] OwningNonNull<mozilla::dom::TrustedScriptURL>&
OwningTrustedScriptURLOrUSVString::SetAsTrustedScriptURL()
{
  if (mType == eTrustedScriptURL) {
    return mValue.mTrustedScriptURL.Value();
  }
  Uninit();
  mType = eTrustedScriptURL;
  return mValue.mTrustedScriptURL.SetValue();
}


void
OwningTrustedScriptURLOrUSVString::DestroyTrustedScriptURL()
{
  MOZ_RELEASE_ASSERT(IsTrustedScriptURL(), "Wrong type!");
  mValue.mTrustedScriptURL.Destroy();
  mType = eUninitialized;
}



bool
OwningTrustedScriptURLOrUSVString::TrySetToUSVString(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsString& memberSlot = RawSetAsUSVString();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
    if (!NormalizeUSVString(memberSlot)) {
      JS_ReportOutOfMemory(cx);
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsString&
OwningTrustedScriptURLOrUSVString::RawSetAsUSVString()
{
  if (mType == eUSVString) {
    return mValue.mUSVString.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUSVString;
  return mValue.mUSVString.SetValue();
}

[[nodiscard]] nsString&
OwningTrustedScriptURLOrUSVString::SetAsUSVString()
{
  if (mType == eUSVString) {
    return mValue.mUSVString.Value();
  }
  Uninit();
  mType = eUSVString;
  return mValue.mUSVString.SetValue();
}



void
OwningTrustedScriptURLOrUSVString::DestroyUSVString()
{
  MOZ_RELEASE_ASSERT(IsUSVString(), "Wrong type!");
  mValue.mUSVString.Destroy();
  mType = eUninitialized;
}



bool
OwningTrustedScriptURLOrUSVString::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToTrustedScriptURL(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUSVString(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "TrustedScriptURL");
    return false;
  }
  return true;
}

bool
OwningTrustedScriptURLOrUSVString::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningTrustedScriptURLOrUSVString::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eTrustedScriptURL: {
      DestroyTrustedScriptURL();
      break;
    }
    case eUSVString: {
      DestroyUSVString();
      break;
    }
  }
}

bool
OwningTrustedScriptURLOrUSVString::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eTrustedScriptURL: {
      if (!WrapNewBindingNonWrapperCachedObject(cx, scopeObj, mValue.mTrustedScriptURL.Value(), rval)) {
        MOZ_ASSERT(JS_IsExceptionPending(cx));
        return false;
      }
      return true;
    }
    case eUSVString: {
      if (!xpc::NonVoidStringToJsval(cx, mValue.mUSVString.Value(), rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningTrustedScriptURLOrUSVString&
OwningTrustedScriptURLOrUSVString::operator=(OwningTrustedScriptURLOrUSVString&& aOther)
{
  this->~OwningTrustedScriptURLOrUSVString();
  new (this) OwningTrustedScriptURLOrUSVString (std::move(aOther));
  return *this;
}


OwningTrustedScriptURLOrUSVString&
OwningTrustedScriptURLOrUSVString::operator=(const OwningTrustedScriptURLOrUSVString& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eTrustedScriptURL: {
      SetAsTrustedScriptURL() = aOther.GetAsTrustedScriptURL();
      break;
    }
    case eUSVString: {
      SetAsUSVString() = aOther.GetAsUSVString();
      break;
    }
  }
  return *this;
}


OwningUTF8StringOrUTF8StringSequence::OwningUTF8StringOrUTF8StringSequence(OwningUTF8StringOrUTF8StringSequence&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eUTF8String: {
      mType = eUTF8String;
      mValue.mUTF8String.SetValue(std::move(aOther.mValue.mUTF8String.Value()));
      break;
    }
    case eUTF8StringSequence: {
      mType = eUTF8StringSequence;
      mValue.mUTF8StringSequence.SetValue(std::move(aOther.mValue.mUTF8StringSequence.Value()));
      break;
    }
  }
}



bool
OwningUTF8StringOrUTF8StringSequence::TrySetToUTF8String(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsCString& memberSlot = RawSetAsUTF8String();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsCString&
OwningUTF8StringOrUTF8StringSequence::RawSetAsUTF8String()
{
  if (mType == eUTF8String) {
    return mValue.mUTF8String.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUTF8String;
  return mValue.mUTF8String.SetValue();
}

[[nodiscard]] nsCString&
OwningUTF8StringOrUTF8StringSequence::SetAsUTF8String()
{
  if (mType == eUTF8String) {
    return mValue.mUTF8String.Value();
  }
  Uninit();
  mType = eUTF8String;
  return mValue.mUTF8String.SetValue();
}



void
OwningUTF8StringOrUTF8StringSequence::DestroyUTF8String()
{
  MOZ_RELEASE_ASSERT(IsUTF8String(), "Wrong type!");
  mValue.mUTF8String.Destroy();
  mType = eUninitialized;
}



bool
OwningUTF8StringOrUTF8StringSequence::TrySetToUTF8StringSequence(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    Sequence<nsCString>& memberSlot = RawSetAsUTF8StringSequence();
    JS::ForOfIterator iter(cx);
    if (!iter.init(value, JS::ForOfIterator::AllowNonIterable)) {
      return false;
    }
    if (!iter.valueIsIterable()) {
      DestroyUTF8StringSequence();
      tryNext = true;
      return true;
    }
    Sequence<nsCString> &arr = memberSlot;
    JS::Rooted<JS::Value> temp(cx);
    while (true) {
      bool done;
      if (!iter.next(&temp, &done)) {
        return false;
      }
      if (done) {
        break;
      }
      nsCString* slotPtr = arr.AppendElement(mozilla::fallible);
      if (!slotPtr) {
        JS_ReportOutOfMemory(cx);
        return false;
      }
      nsCString& slot = *slotPtr;
      if (!ConvertJSValueToString(cx, temp, eStringify, eStringify, slot)) {
        return false;
      }
    }
  }
  return true;
}

bool
OwningUTF8StringOrUTF8StringSequence::TrySetToUTF8StringSequence(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToUTF8StringSequence(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] Sequence<nsCString>&
OwningUTF8StringOrUTF8StringSequence::RawSetAsUTF8StringSequence()
{
  if (mType == eUTF8StringSequence) {
    return mValue.mUTF8StringSequence.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUTF8StringSequence;
  return mValue.mUTF8StringSequence.SetValue();
}

[[nodiscard]] Sequence<nsCString>&
OwningUTF8StringOrUTF8StringSequence::SetAsUTF8StringSequence()
{
  if (mType == eUTF8StringSequence) {
    return mValue.mUTF8StringSequence.Value();
  }
  Uninit();
  mType = eUTF8StringSequence;
  return mValue.mUTF8StringSequence.SetValue();
}


void
OwningUTF8StringOrUTF8StringSequence::DestroyUTF8StringSequence()
{
  MOZ_RELEASE_ASSERT(IsUTF8StringSequence(), "Wrong type!");
  mValue.mUTF8StringSequence.Destroy();
  mType = eUninitialized;
}



bool
OwningUTF8StringOrUTF8StringSequence::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToUTF8StringSequence(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUTF8String(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "sequence<USVString>");
    return false;
  }
  return true;
}

bool
OwningUTF8StringOrUTF8StringSequence::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningUTF8StringOrUTF8StringSequence::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eUTF8String: {
      DestroyUTF8String();
      break;
    }
    case eUTF8StringSequence: {
      DestroyUTF8StringSequence();
      break;
    }
  }
}

bool
OwningUTF8StringOrUTF8StringSequence::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eUTF8String: {
      if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8String.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eUTF8StringSequence: {

      uint32_t length = mValue.mUTF8StringSequence.Value().Length();
      JS::Rooted<JSObject*> returnArray(cx, JS::NewArrayObject(cx, length));
      if (!returnArray) {
        return false;
      }
      // Scope for 'tmp'
      {
        JS::Rooted<JS::Value> tmp(cx);
        for (uint32_t sequenceIdx0 = 0; sequenceIdx0 < length; ++sequenceIdx0) {
          // Control block to let us common up the JS_DefineElement calls when there
          // are different ways to succeed at wrapping the object.
          do {
            if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8StringSequence.Value()[sequenceIdx0], &tmp)) {
              return false;
            }
            break;
          } while (false);
          if (!JS_DefineElement(cx, returnArray, sequenceIdx0, tmp,
                                JSPROP_ENUMERATE)) {
            return false;
          }
        }
      }
      rval.setObject(*returnArray);
      return true;
    }
    default: {
      return false;
    }
  }
}

OwningUTF8StringOrUTF8StringSequence&
OwningUTF8StringOrUTF8StringSequence::operator=(OwningUTF8StringOrUTF8StringSequence&& aOther)
{
  this->~OwningUTF8StringOrUTF8StringSequence();
  new (this) OwningUTF8StringOrUTF8StringSequence (std::move(aOther));
  return *this;
}


OwningUTF8StringOrUTF8StringSequence&
OwningUTF8StringOrUTF8StringSequence::operator=(const OwningUTF8StringOrUTF8StringSequence& aOther)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eUTF8String: {
      SetAsUTF8String() = aOther.GetAsUTF8String();
      break;
    }
    case eUTF8StringSequence: {
      SetAsUTF8StringSequence() = aOther.GetAsUTF8StringSequence();
      break;
    }
  }
  return *this;
}


OwningUTF8StringOrUint8Array::OwningUTF8StringOrUint8Array(OwningUTF8StringOrUint8Array&& aOther)
  : mType(eUninitialized)
{
  switch (aOther.mType) {
    case eUninitialized: {
      MOZ_ASSERT(mType == eUninitialized,
                 "We need to destroy ourselves?");
      break;
    }
    case eUTF8String: {
      mType = eUTF8String;
      mValue.mUTF8String.SetValue(std::move(aOther.mValue.mUTF8String.Value()));
      break;
    }
    case eUint8Array: {
      mType = eUint8Array;
      mValue.mUint8Array.SetValue(std::move(aOther.mValue.mUint8Array.Value()));
      break;
    }
  }
}


bool
OwningUTF8StringOrUint8Array::TrySetToUTF8String(JSContext* cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    nsCString& memberSlot = RawSetAsUTF8String();
    if (!ConvertJSValueToString(cx, value, eStringify, eStringify, memberSlot)) {
      return false;
    }
  }
  return true;
}

[[nodiscard]] nsCString&
OwningUTF8StringOrUint8Array::RawSetAsUTF8String()
{
  if (mType == eUTF8String) {
    return mValue.mUTF8String.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUTF8String;
  return mValue.mUTF8String.SetValue();
}

[[nodiscard]] nsCString&
OwningUTF8StringOrUint8Array::SetAsUTF8String()
{
  if (mType == eUTF8String) {
    return mValue.mUTF8String.Value();
  }
  Uninit();
  mType = eUTF8String;
  return mValue.mUTF8String.SetValue();
}



void
OwningUTF8StringOrUint8Array::DestroyUTF8String()
{
  MOZ_RELEASE_ASSERT(IsUTF8String(), "Wrong type!");
  mValue.mUTF8String.Destroy();
  mType = eUninitialized;
}



bool
OwningUTF8StringOrUint8Array::TrySetToUint8Array(BindingCallContext& cx, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  tryNext = false;
  { // scope for memberSlot
    Uint8Array& memberSlot = RawSetAsUint8Array();
    if (!memberSlot.Init(&value.toObject())) {
      DestroyUint8Array();
      tryNext = true;
      return true;
    }
    if (JS::IsArrayBufferViewShared(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_SHARED>("Uint8Array branch of (USVString or Uint8Array)");
      return false;
    }
    if (JS::IsLargeArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_LARGE>("Uint8Array branch of (USVString or Uint8Array)");
      return false;
    }
    if (JS::IsResizableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_RESIZABLE>("Uint8Array branch of (USVString or Uint8Array)");
      return false;
    }
    if (JS::IsImmutableArrayBufferView(memberSlot.Obj())) {
      cx.ThrowErrorMessage<MSG_TYPEDARRAY_IS_IMMUTABLE>("Uint8Array branch of (USVString or Uint8Array)");
      return false;
    }
  }
  return true;
}

bool
OwningUTF8StringOrUint8Array::TrySetToUint8Array(JSContext* cx_, JS::Handle<JS::Value> value, bool& tryNext, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return TrySetToUint8Array(cx, value, tryNext, passedToJSImpl);
}

[[nodiscard]] Uint8Array&
OwningUTF8StringOrUint8Array::RawSetAsUint8Array()
{
  if (mType == eUint8Array) {
    return mValue.mUint8Array.Value();
  }
  MOZ_ASSERT(mType == eUninitialized);
  mType = eUint8Array;
  return mValue.mUint8Array.SetValue();
}

[[nodiscard]] Uint8Array&
OwningUTF8StringOrUint8Array::SetAsUint8Array()
{
  if (mType == eUint8Array) {
    return mValue.mUint8Array.Value();
  }
  Uninit();
  mType = eUint8Array;
  return mValue.mUint8Array.SetValue();
}


void
OwningUTF8StringOrUint8Array::DestroyUint8Array()
{
  MOZ_RELEASE_ASSERT(IsUint8Array(), "Wrong type!");
  mValue.mUint8Array.Destroy();
  mType = eUninitialized;
}



bool
OwningUTF8StringOrUint8Array::Init(BindingCallContext& cx, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  MOZ_ASSERT(mType == eUninitialized);

  bool done = false, failed = false, tryNext;
  if (value.isObject()) {
    done = (failed = !TrySetToUint8Array(cx, value, tryNext, passedToJSImpl)) || !tryNext;
  }
  if (!done) {
    do {
      done = (failed = !TrySetToUTF8String(cx, value, tryNext)) || !tryNext;
      break;
    } while (false);
  }
  if (failed) {
    return false;
  }
  if (!done) {
    cx.ThrowErrorMessage<MSG_NOT_IN_UNION>(sourceDescription, "Uint8Array");
    return false;
  }
  return true;
}

bool
OwningUTF8StringOrUint8Array::Init(JSContext* cx_, JS::Handle<JS::Value> value, const char* sourceDescription, bool passedToJSImpl)
{
  BindingCallContext cx(cx_, nullptr);
  return Init(cx, value, sourceDescription, passedToJSImpl);
}

void
OwningUTF8StringOrUint8Array::Uninit()
{
  switch (mType) {
    case eUninitialized: {
      break;
    }
    case eUTF8String: {
      DestroyUTF8String();
      break;
    }
    case eUint8Array: {
      DestroyUint8Array();
      break;
    }
  }
}

bool
OwningUTF8StringOrUint8Array::ToJSVal(JSContext* cx, JS::Handle<JSObject*> scopeObj, JS::MutableHandle<JS::Value> rval) const
{
  switch (mType) {
    case eUninitialized: {
      return false;
    }
    case eUTF8String: {
      if (!NonVoidUTF8StringToJsval(cx, mValue.mUTF8String.Value(), rval)) {
        return false;
      }
      return true;
    }
    case eUint8Array: {
      rval.setObject(*mValue.mUint8Array.Value().Obj());
      if (!MaybeWrapNonDOMObjectValue(cx, rval)) {
        return false;
      }
      return true;
    }
    default: {
      return false;
    }
  }
}

void
OwningUTF8StringOrUint8Array::TraceUnion(JSTracer* trc)
{
  switch (mType) {
    case eUint8Array: {
      mValue.mUint8Array.Value().TraceSelf(trc);
      break;
    }
    default: {
    }
  }
}

OwningUTF8StringOrUint8Array&
OwningUTF8StringOrUint8Array::operator=(OwningUTF8StringOrUint8Array&& aOther)
{
  this->~OwningUTF8StringOrUint8Array();
  new (this) OwningUTF8StringOrUint8Array (std::move(aOther));
  return *this;
}

} // namespace mozilla::dom

